package parser;

import java.io.DataInputStream;
import parser.rec.types.*;
import parser.rec.*;
import tom.library.sl.*;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.ANTLRInputStream;
import java.io.*;
import java.util.*;

public class Main {

  %include { rec/Rec.tom }
  %include { sl.tom }

  public static void main(String[] args) throws VisitFailure {
    try {
      if(args.length<=0) {
        System.out.println("usage: java Main <filename>"); 
      } else {
        // Initialize parser
        RecLexer lexer = new RecLexer(new ANTLRInputStream(new FileInputStream(args[0])));
        CommonTokenStream tokens = new CommonTokenStream(lexer);
        RecParser parser = new RecParser(tokens);
        parser.setTreeAdaptor(new RecAdaptor());   
        // Parse the input expression
        RecTree b = (RecTree) parser.program().getTree();
        Program p = (Program) b.getTerm();
        Main main = new Main();
        //System.out.println(p); 
        // Compile the input rec program into gom program
        Program compiledProgram = main.compile(p);
        //System.out.println(compiledProgram); 
        // Generate Tom File 
        StringWriter writer = new StringWriter();
        main.generateGomFile(writer,compiledProgram);
        System.out.println(writer);
      }
    } catch(Exception e) {
      e.printStackTrace();
    }
  }

  public Program compile(Program s) {
    %match(s) {
      Program(SpecRec(n,sorts,operators,v,r),cl) -> {
        Signature signature = `buildSignature(sorts,operators);
        Program newProgram = `Program(SpecGom(n,signature,r),cl);
        try {
          newProgram = (Program) `TopDown(CstToVar(v)).visitLight(newProgram);
          newProgram = (Program) `TopDown(RenameId()).visitLight(newProgram);
        } catch(VisitFailure e) {
          System.out.println("failure");
        }
        return newProgram;
      }
    }
    return s;
  }

  %strategy CstToVar(v:VarList) extends Identity() {
    visit Term {
      Cst(x) -> {
        %match(v) {
          ConcVar(_*,Vars[ListName=ConcName(_*,vn,_*)],_*) && vn<<String x -> {
            return `Var(x);
          }
        }
        return `App(x,ConcTerm());
      }
    }
  }

  %strategy RenameId() extends Identity() {
    visit Term {
      t@(App|Cst|Var)[Name=name] -> {
        return `t.setName(makeGoodName(`name));
      }
    }
    visit Operator {
      t@(SimpleOperator|OperatorWithTheory)[Name=name] -> {
        return `t.setName(makeGoodName(`name));
      }
    }
    visit Sort {
      t@Sort[Name=name] -> {
        return `t.setName(makeGoodName(`name));
      }
    }
  }

  private static String makeGoodName(String s) {
    // special keywords
    if(s.toLowerCase().equals("boolean")) {
      return "Bool";
    } else if(s.equals("true")) {
      return "True";
    } else if(s.equals("false")) {
      return "False";
    } else if(s.toLowerCase().equals("int")) {
      return "Integers";
    } else if(s.equals("try")) {
      return "tryop";
    } else if(s.equals("mergesort")) {
      return "mymergesort";
    } else if(s.equals("sort")) {
      return "sortop";
    }
    // start with digit
    if(s.length() > 0) {
      char c = s.charAt(0);
      if(!Character.isJavaIdentifierStart(c)) {
        return "d" + sanitize(s);
      }
    }
    return sanitize(s);
  }

  /*
   * rename strange character inside an identifier
   */
  private static String sanitize(String s) {
    StringBuffer sb = new StringBuffer();
    for(int i=0 ; i<s.length() ; i++) {
      char c = s.charAt(i);
      switch(c) {
        case '-': 
          sb.append("_");
          break;
        case '$': 
          sb.append("dollar");
          break;
        case '!': 
          sb.append("bang");
          break;
        case '*': 
          sb.append("star");
          break;
        case '/': 
          sb.append("div");
          break;
        case '+':
          sb.append("plus");
          break;
        case '\'':
          sb.append("prime");
          break;
        case '<':
          sb.append("lt");
          break;
        case '>':
          sb.append("gt");
          break;
        case '=':
          sb.append("eq");
          break;
        default:
          if(Character.isJavaIdentifierPart(c)) {
            sb.append(c);
          } else {
            int value = c;
            sb.append("ascii"+value);
          }
      }
    }
    return sb.toString();
  }

  public void generateGomFile(java.io.Writer writer, Program p) throws java.io.IOException {
    %match(p) {
      Program(SpecGom(name,signature,rules),commandlist) -> {
        writer.write(%[
import @makeGoodName(`name).toLowerCase()@.term.*;
import @makeGoodName(`name).toLowerCase()@.term.types.*;

public class @makeGoodName(`name).toUpperCase()@ {

            ]%);
        int i =0;
        writer.write(%[
    %gom{
        module Term
        abstract syntax
            ]%);
        %match(signature) {
          SortList(_*,PairSortOperatorList(Sort(sname),oplist),_*) -> {
            writer.write(%[
          @`sname@ = ]%);
            %match(oplist) {
              ConcOperator(_*,(SimpleOperator|OperatorWithTheory)[Name=opname,Args=args],_*) -> {
                writer.write(%[
              |@`opname@(]%);
                %match(args) {
                  ConcSort(_*,Sort[Name=sortname],X*) -> {
                    writer.write("t_"+`sortname+i+":"+`sortname);
                    i++;
                    if (! `X.isEmptyConcSort()) {
                      writer.write(",");
                    }
                  }
                }
                writer.write(")");
              }

            }
          }
        }
        writer.write(%[
        module Term:rules() {
]%);
        %match(rules) {
          ConcRule(_*,SimpleRule[lhs=lhs,rhs=rhs],_*) -> {
            writer.write(%[
            @generateTerm(`lhs)@ -> @generateTerm(`rhs)@
            ]%);
          }
          ConcRule(_*,CondRule[lhs=lhs,rhs=rhs,cond=conds],_*) -> {
            writer.write(%[
            @generateTerm(`lhs)@ -> @generateTerm(`rhs)@ if 
            ]%);
            %match(conds) {
              ConcCond(_*,c,X*) -> {
                %match(c) {
                  Equal(t1,t2) -> {
                    writer.write(%[@generateTerm(`t1)@ == @generateTerm(`t2)@ ]%);
                  }
                  RewriteTo(t1,t2) -> {
                    writer.write(%[@generateTerm(`t2)@ << @generateTerm(`t1)@ ]%);
                  }
                  NotEqual(t1,t2) -> {
                    writer.write(%[@generateTerm(`t1)@ != @generateTerm(`t2)@ ]%);
                  }

                }
                if(! `X.isEmptyConcCond()) {
                  writer.write(" && ");
                }
              }
            }
            writer.write("\n");
          }
        }
        writer.write(%[
        }
     }

     public static void main(String[] args) {
       long startChrono;
       long stopChrono;
       ]%);
        for(Command command:(parser.rec.types.commandlist.ConcCommand)`commandlist) {
          %match(command) {
            (GetNormalForm|GetAllNormalForms)(t) -> {
        writer.write(%[
       startChrono = System.currentTimeMillis();
       System.out.println(`@generateTerm(`t)@);
       stopChrono  = System.currentTimeMillis();
       System.out.println("reduced in " + (stopChrono-startChrono) + " ms");
       ]%);
            }
            (CheckConfluence|CheckRewriteTo)(t1,t2) -> {
        writer.write(%[
       startChrono = System.currentTimeMillis();
       System.out.println(`@generateTerm(`t1)@==`@generateTerm(`t2)@);
       stopChrono  = System.currentTimeMillis();
       System.out.println("reduced in " + (stopChrono-startChrono) + " ms");
       ]%);
            }
          }
        }
       writer.write(%[
     }

}
            ]%);
      }
    }
  }
  
private String generateTerm(Term t) throws java.io.IOException {
  StringBuffer sb = new StringBuffer();
  %match (t) {
    App(name,args) -> {
      sb.append(`name+"(");
      %match(args) {
        ConcTerm(_*,tt,X*) -> {
          sb.append(generateTerm(`tt));
          if (! `X.isEmptyConcTerm()) {
            sb.append(",");
          }
        }
      }
      sb.append(")");
    }

    Var(name) -> { sb.append(`name); }
  }
  return sb.toString();
}

  public Signature buildSignature(SortList sorts, OperatorList ops) {
    HashMap<Sort,OperatorList> sortMap = new HashMap<Sort,OperatorList>();
    %match(ops) {
      ConcOperator(_*,o@(SimpleOperator|OperatorWithTheory)[Codomain=codomain],_*) -> {
        OperatorList oplist = sortMap.get(`codomain);
        if(oplist == null) {
          oplist = `ConcOperator(o);
        } else {
          oplist = `ConcOperator(o,oplist*);
        }
        sortMap.put(`codomain,oplist);
      }
    }
    Signature signature = `SortList();
    for(Sort key:sortMap.keySet()) {
      signature = `SortList(PairSortOperatorList(key,sortMap.get(key)),signature*);
    }
    return signature;
  }

}
