(****************************************************************************)
(***             CELULAR GENETIC ALGORITHM - Abstract Data Type           ***)
(***                v 1.0, by Enrique Alba Torres, 6/12/96                ***)
(****************************************************************************)

IMPLEMENTATION MODULE cssGA;
FROM IO      IMPORT WrLn, WrLngCard, WrReal, WrCard, WrStr, RdCard;
FROM Lib     IMPORT RAND, RANDOM, RANDOMIZE;
FROM Storage IMPORT ALLOCATE, DEALLOCATE;
FROM Str     IMPORT RealToStr;

CONST
   POP_SIZE   = WIDTH * HEIGHT;                              (*Hidden value!*)
   NEIGH_SIZE = 4;                                     (*Number of neighbors*)

TYPE
   POPULATION = ARRAY[1..WIDTH],[1..HEIGHT] OF INDIVIDUAL;
   cGA        = POINTER TO GA_RECORD;            (*Celular Genetic Algorithm*)
   GA_RECORD  = RECORD
                                (*---Private parameters---*)
                    pop                  : POPULATION;
                    stats                : STATS;
                    actual_step          : LONGCARD;
                                (*---User defined parameters---*)
                    seed                 : CARDINAL;
                    pc, pm               : REAL;
                    fitness_function     : EVAL_FUNC;
                    replacement          : REPLACEMENT_TYPE;
               END; (*GA_RECORD*)

  LOCATION   = RECORD
                    fitness : REAL;
                    where   : POINT;
               END; (*LOCATION*)
  NEIGH_LIST = ARRAY [1..NEIGH_SIZE] OF LOCATION;
  NEIGHBORS  = RECORD
                    who   : POINT;
                    n     : NEIGH_LIST;
                    sum_f : REAL;
               END; (*NEIGHBORS*)

(*Roulette Wheel, Double Point Crossover, Bit Flip Mutation*)
(*Replace if better or always. Best individual in position 'stats.best_pos'*)


(************************* AUXILIAR PROCEDURES ******************************)
PROCEDURE Flip( Pr : REAL ) : BOOLEAN;
BEGIN
    IF((Pr=1.0) OR (Pr>=RAND())) THEN RETURN TRUE;
                                 ELSE RETURN FALSE;
    END; (*IF*)
END Flip;



PROCEDURE Decode( chr : CHROMOSOME ) : PHENOTYPE;
VAR
  ph          : PHENOTYPE;
  i, j, index : CARDINAL;
BEGIN
    index := 1;
    FOR i:=1 TO GENE_NUMBER DO
        ph[i] := REAL(chr[index]);
        INC(index);
        FOR j:=1 TO GENE_LENGTH-1 DO
            ph[i] := 2.0*ph[i] + REAL(chr[index]);
            INC(index);
        END; (*A gene*)
    END; (*All the genes*)

    RETURN ph;

END Decode;

(*Roulette Wheel selection mechanism*)
PROCEDURE RW_Select( neigh : NEIGHBORS ) : CARDINAL;
VAR select_ind   : CARDINAL;
    target, acum : REAL;
BEGIN
    target     := RAND()*neigh.sum_f;
    acum       := neigh.n[1].fitness;
    select_ind := 1;
    WHILE (acum<target) DO
          acum := acum + neigh.n[select_ind].fitness;
          INC(select_ind);
    END; (*WHILE*)
    IF select_ind>NEIGH_SIZE THEN select_ind:=NEIGH_SIZE; END;(*UNNECESSARY!!!*)
    RETURN select_ind;
END RW_Select;

(*Double Point Crossover*)
PROCEDURE Dpx( cga : cGA; i1, i2 : POINT; VAR aux_chrom : CHROMOSOME);
VAR
  i, j, p1, p2, pointer : CARDINAL;
  index                 : POINT;
BEGIN
    IF(NOT Flip(cga^.pc)) THEN
           IF(cga^.pop[i1.x][i1.y].fitness>cga^.pop[i2.x][i2.y].fitness)
              THEN index := i1;
              ELSE index := i2;
           END; (*IF*)
           FOR i:= 1 TO CHROM_LENGTH DO
               aux_chrom[i] := cga^.pop[index.x][index.y].chr[i];
           END; (*FOR*)
           RETURN;
    END; (*IF*)
        (*---Initialisations---*)
    pointer := 1;
        (*---Select and sort the two different crossover points---*)
    p1    := RANDOM(CHROM_LENGTH)+1;
    REPEAT p2 := RANDOM(CHROM_LENGTH)+1; UNTIL (p1<>p2);
    IF(p1>p2) THEN i:=p1; p1:=p2; p2:=i; END;
        (*---Build the new individual---*)
    FOR i:=1 TO p1 DO
        aux_chrom[pointer]:=cga^.pop[i1.x][i1.y].chr[i];   INC(pointer);
    END;
    FOR i:=p1+1 TO p2 DO
        aux_chrom[pointer]:=cga^.pop[i2.x][i2.y].chr[i];   INC(pointer);
    END;
    FOR i:=p2+1 TO CHROM_LENGTH DO
        aux_chrom[pointer]:=cga^.pop[i1.x][i1.y].chr[i];   INC(pointer);
    END;

END Dpx;

(*Mutate individual by inverting bits*)
PROCEDURE Mutate( cga : cGA; VAR ch : CHROMOSOME );
VAR i : CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO
        IF(Flip(cga^.pm)) THEN ch[i]:=1-ch[i]; END;(*IF*)
    END; (*Mutate every allele value*)
END Mutate;

(*Compute the global stats OF the distributed steady state GA*)
PROCEDURE Compute_Stats( VAR cga:cGA );
VAR x, y : CARDINAL;
BEGIN
    cga^.stats.best_pos.x := 1;
    cga^.stats.best_pos.y := 1;
    cga^.stats.worst_pos.x:= 1;
    cga^.stats.worst_pos.y:= 1;
    cga^.stats.min_f      := cga^.pop[1][1].fitness;
    cga^.stats.avg_f      := 0.0;
    cga^.stats.max_f      := cga^.pop[1][1].fitness;
    cga^.stats.sum_f      := 0.0;

    (*Gather statistiscs*)
    FOR x:=1 TO WIDTH DO
      FOR y:=1 TO HEIGHT DO
        IF(cga^.pop[x][y].fitness>cga^.stats.max_f) THEN
           cga^.stats.max_f := cga^.pop[x][y].fitness;
           cga^.stats.best_pos.x := x; cga^.stats.best_pos.y := y;
        END;
        IF(cga^.pop[x][y].fitness<cga^.stats.min_f) THEN
           cga^.stats.min_f := cga^.pop[x][y].fitness;
           cga^.stats.worst_pos.x := x; cga^.stats.worst_pos.y := y;
        END;
        cga^.stats.sum_f := cga^.stats.sum_f + cga^.pop[x][y].fitness;
      END; (*HEIGHT*)
    END; (*WIDTH*)

    cga^.stats.avg_f      := cga^.stats.sum_f / REAL(POP_SIZE);

END Compute_Stats;


(*Inserts in a sorted mode a new individual and remakes the statistics*)
PROCEDURE Insert_New_Ind(     ind:INDIVIDUAL; ind_pos: POINT; cga:cGA;
                          VAR pop:POPULATION  );
VAR i: CARDINAL;
BEGIN
    IF (cga^.replacement=if_better) THEN
       IF (cga^.pop[ind_pos.x][ind_pos.y].fitness>ind.fitness) THEN
          FOR i:=1 TO CHROM_LENGTH DO                          (*COPY OLD*)
              pop[ind_pos.x][ind_pos.y].chr[i]:=cga^.pop[ind_pos.x][ind_pos.y].chr[i];
          END; (*FOR*)
          pop[ind_pos.x][ind_pos.y].fitness := cga^.pop[ind_pos.x][ind_pos.y].fitness;
          RETURN;
       END;
    END; (*IF*)

    FOR i:=1 TO CHROM_LENGTH DO                         (*COPY NEW*)
        pop[ind_pos.x][ind_pos.y].chr[i] := ind.chr[i];
    END; (*FOR*)
    pop[ind_pos.x][ind_pos.y].fitness := ind.fitness;

END Insert_New_Ind;

(*Increments a coordenate in the grid*)
PROCEDURE Inc_Grid( value : CARDINAL; limit : CARDINAL ) : CARDINAL;
BEGIN
    INC(value);
    IF (value>limit) THEN value := 1; END;
    RETURN value;
END Inc_Grid;

(*Decrements a coordenate in the grid*)
PROCEDURE Dec_Grid( value : CARDINAL; limit : CARDINAL ) : CARDINAL;
BEGIN
    IF (value>1) THEN DEC(value);
                 ELSE value := limit;
    END;
    RETURN value;
END Dec_Grid;

(*Computes the list of neighbors for a given individual in the grid.*)
PROCEDURE Compute_Neighbors( cga : cGA; VAR neigh : NEIGHBORS );
VAR i : CARDINAL;
BEGIN
    neigh.n[1].where.x := neigh.who.x;                   (*NORTH*)
    neigh.n[1].where.y := Dec_Grid(neigh.who.y,HEIGHT);  (*NORTH*)
    neigh.n[2].where.x := Inc_Grid(neigh.who.x,WIDTH);   (*EAST*)
    neigh.n[2].where.y := neigh.who.y;                   (*EAST*)
    neigh.n[3].where.x := neigh.who.x;                   (*SOUTH*)
    neigh.n[3].where.y := Inc_Grid(neigh.who.y,HEIGHT);  (*SOUTH*)
    neigh.n[4].where.x := Dec_Grid(neigh.who.x,WIDTH);   (*WEST*)
    neigh.n[4].where.y := neigh.who.y;                   (*WEST*)

    neigh.sum_f := 0.0;
    FOR i:=1 TO NEIGH_SIZE DO
        neigh.n[i].fitness := cga^.pop[neigh.n[i].where.x][neigh.n[i].where.y].fitness;
        neigh.sum_f := neigh.sum_f + neigh.n[i].fitness;
    END; (*FOR*)
END Compute_Neighbors;


(****************************** PUBLIC METHODS ******************************)

(*Computes the actual averaged hamming distance IN pop. Binary strings only!*)
PROCEDURE cCompute_HammingD( VAR cga:cGA );
VAR i, x, y: CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO cga^.stats.h_dist[i] := 0.0; END;(*FOR*)
    FOR i:=1 TO CHROM_LENGTH DO
        FOR x:=1 TO WIDTH DO
            FOR y:=1 TO HEIGHT DO
                cga^.stats.h_dist[i]:=cga^.stats.h_dist[i]+REAL(cga^.pop[x][y].chr[i]);
            END; (*FOR*)
        END; (*FOR*)
        cga^.stats.h_dist[i] := cga^.stats.h_dist[i] / REAL(POP_SIZE);
    END;(*FOR*)
END cCompute_HammingD;

(*Compute the defining length of a schema*)
PROCEDURE Defining_Length( S: SCHEMA ) : LONGCARD;
VAR left, right : LONGCARD;
    i           : CARDINAL;
BEGIN
    left := 0; right := 0;       (*0 means 'no still computed value'*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((left=0) AND (S[i]<>'*')) THEN left :=LONGCARD(i); END;
        IF (S[i]<>'*')                THEN right:=LONGCARD(i); END;
    END;
    RETURN (right-left);
END Defining_Length;

(*Compute the order of a schema*)
PROCEDURE Order( S: SCHEMA ) : LONGCARD;
VAR ndef_pos : LONGCARD;
    i        : CARDINAL;
BEGIN
    ndef_pos := 0; (*Number of defined positions in the schema*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF (S[i]<>'*') THEN INC(ndef_pos); END;
    END;
    RETURN ndef_pos;
END Order;

(*Decides wether a string is an instance of a schema or NOT*)
PROCEDURE Is_Instance( str:CHROMOSOME; S:SCHEMA ) : BOOLEAN;
VAR i: CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((str[i]=1) AND (S[i]='0')) THEN RETURN FALSE; END;
        IF ((str[i]=0) AND (S[i]='1')) THEN RETURN FALSE; END;
    END;
    RETURN TRUE;
END Is_Instance;

(*Count the number of instances of a schema in the population*)
PROCEDURE Schema_Count( cga:cGA; S:SCHEMA; VAR ni:LONGCARD; VAR ssum_f: REAL);
VAR x, y :CARDINAL;
BEGIN
    ni := 0; ssum_f := 0.0;
    FOR x:=1 TO WIDTH DO
        FOR y:=1 TO HEIGHT DO
            IF (Is_Instance(cga^.pop[x][y].chr, S)) THEN
                INC(ni);
                ssum_f := ssum_f + cga^.pop[x][y].fitness;
            END; (*IF*)
        END; (*HEIGHT*)
    END; (*WIDTH*)

END Schema_Count;

(*Compute the stats about the parameter -schema- AND predict the next number*)
PROCEDURE cSchema_Stats( cga: cGA; S: SCHEMA ) : S_STATS;
VAR stat      : S_STATS;
    ssum_f, c : REAL;
BEGIN
    c := 1.0 - (cga^.pc*REAL(Defining_Length(S))/REAL(CHROM_LENGTH-1) -
                cga^.pm*REAL(Order(S)));
    stat.actual_avg_f  := 0.0; stat.actual_instances := 0;
    stat.sch_th_instances := 0;
    Schema_Count( cga, S, stat.actual_instances, ssum_f );
    IF(stat.actual_instances>0) THEN
           stat.actual_avg_f := ssum_f / REAL(stat.actual_instances);
    ELSE  stat.actual_avg_f := 0.0;
    END; (*IF*)
    stat.sch_th_instances := LONGCARD(
           (REAL(stat.actual_instances)*stat.actual_avg_f/cga^.stats.avg_f)*c
           +0.5 (*round*)            );
    RETURN stat;

END cSchema_Stats;


(*Initialise the ga variable*)
PROCEDURE Init_cGA( VAR cga:cGA );
BEGIN
    cga := NIL;
END Init_cGA;

(*Set the parameters for the GA*)
PROCEDURE Set_Up_cGA( VAR cga           : cGA;
                     seed              : CARDINAL;
                     pc, pm            : REAL;
                     rep_type          : REPLACEMENT_TYPE;
                     fitness_function  : EVAL_FUNC
                     );
VAR x, y, j : CARDINAL;


BEGIN
    NEW(cga);
    cga^.seed             := seed;
    cga^.pc               := pc;
    cga^.pm               := pm;
    cga^.replacement      := rep_type;
    cga^.fitness_function := fitness_function;
    cga^.actual_step      := 0;
    cga^.stats.min_f      := 0.0;
    cga^.stats.avg_f      := 0.0;
    cga^.stats.max_f      := 0.0;
    cga^.stats.sum_f      := 0.0;
    cga^.stats.p_on_sum   := 0.0;
    cga^.stats.p_off_sum  := 0.0;
    cga^.stats.f_improve  := 0.0;

    FOR x:=1 TO WIDTH DO
      FOR y:=1 TO HEIGHT DO
        FOR j:=1 TO CHROM_LENGTH DO
            IF(Flip(0.5)) THEN cga^.pop[x][y].chr[j] := 1;
                          ELSE cga^.pop[x][y].chr[j] := 0;
            END; (*IF*)
        END; (*CHROMOSOME*)
        cga^.pop[x][y].fitness := cga^.fitness_function( Decode(cga^.pop[x][y].chr) );
      END; (*HEIGHT*)
    END; (*WIDTH*)

    Compute_Stats(cga);
    cCompute_HammingD(cga);

END Set_Up_cGA;

(*Makes evolution steps. Returns the actual step*)
PROCEDURE cSteps_Up( VAR cga:cGA; steps:LONGCARD ) : LONGCARD;
VAR aux_chrom   : CHROMOSOME;
    aux_pop     : POPULATION;                   c:CHAR;
    aux_ind     : INDIVIDUAL;
    p1, p2, i   : CARDINAL;
    x, y        : CARDINAL;
    fitness     : REAL;
    s           : LONGCARD;
    neigh       : NEIGHBORS;
    ind_pos     : POINT;
BEGIN
 FOR s:=1 TO steps DO
    INC(cga^.actual_step);

    FOR x:=1 TO WIDTH DO
        FOR y:=1 TO HEIGHT DO
            neigh.who.x := x; neigh.who.y := y;
            Compute_Neighbors( cga, neigh );(*Compute fitness list of neighbs.*)

            p1 := RW_Select(neigh);      (*First index in the neighbors list*)
            p2 := RW_Select(neigh);      (*Scnd. index in the neighbors list*)
            Dpx(cga,neigh.n[p1].where,neigh.n[p2].where,aux_chrom);
            Mutate(cga,aux_chrom);
            fitness := cga^.fitness_function( Decode(aux_chrom) );

            aux_ind.fitness      := fitness;
            aux_ind.chr          := aux_chrom;

            ind_pos.x := x; ind_pos.y := y;
            Insert_New_Ind( aux_ind, ind_pos, cga, aux_pop );

            (*No computed stats after every insertion!*)

        END; (*HEIGHT*)
    END; (*WIDTH*)

    cga^.stats.f_improve := cga^.stats.avg_f; (*Record actual average fitness*)

    FOR x:=1 TO WIDTH DO        (*COPY NEW POP ONTO OLD POP!*)
        FOR y:=1 TO HEIGHT DO
            FOR i:=1 TO CHROM_LENGTH DO
                cga^.pop[x][y].chr[i]:=aux_pop[x][y].chr[i];
            END;
            cga^.pop[x][y].fitness := aux_pop[x][y].fitness;
        END; (*HEIGHT*)
    END; (*WIDTH*)

    Compute_Stats(cga);

    (*Performances are only computed after every step of the algorithm*)
    cga^.stats.f_improve  := cga^.stats.avg_f     - cga^.stats.f_improve;
    cga^.stats.p_on_sum   := cga^.stats.p_on_sum  + cga^.stats.avg_f;
    cga^.stats.p_off_sum  := cga^.stats.p_off_sum + cga^.stats.max_f;


  END; (*MAIN FOR*)
    RETURN cga^.actual_step;
END cSteps_Up;

(*Returns a record with statistics of the GA*)
PROCEDURE cGet_Stats(cga:cGA) : STATS;
BEGIN
   RETURN cga^.stats;
END cGet_Stats;

(*Returns the individual number 'ind'*)
PROCEDURE Get_Individual( cga:cGA; ind_pos:POINT ) : INDIVIDUAL;
BEGIN

        IF ((ind_pos.x>=1) AND (ind_pos.x<=WIDTH) AND
            (ind_pos.y>=1) AND (ind_pos.y<=HEIGHT)) THEN
           RETURN cga^.pop[ind_pos.x][ind_pos.y];
        END; (*IF*)
END Get_Individual;



(*Displays the 'item' in the screen. If item='individual' then use 'ind_pos'*)
PROCEDURE Display( cga:cGA; item:ITEM; ind_pos:POINT );
VAR
    i, x, y    : CARDINAL;
    pos        : POINT;
    ok         : BOOLEAN;
    str        : ARRAY[1..32] OF CHAR;
BEGIN
    CASE item OF
         individual :    WrStr("INDIVIDUAL #(");WrCard(ind_pos.x,2);WrStr(",");WrCard(ind_pos.y,2);
                         WrStr(") ===> ");
                         FOR i:=1 TO CHROM_LENGTH DO
                             WrCard(CARDINAL(cga^.pop[ind_pos.x][ind_pos.y].chr[i]),0);
                         END; (*FOR*)
                         WrReal(cga^.pop[ind_pos.x][ind_pos.y].fitness,10,12);
                         WrLn;
                         |
         best       :    Display(cga,individual,cga^.stats.best_pos);
                         |
         worst      :    Display(cga,individual,cga^.stats.worst_pos);
                         |
         population :    FOR x:=1 TO WIDTH DO
                             FOR y:=1 TO HEIGHT DO
                                 pos.x:=x; pos.y:=y;
                                 Display(cga,individual,pos);
                             END; (*HEIGHT*)
                         END; (*WIDTH*)
                         |
         statistics :    (*WrStr("---STATISTICS---"); WrLn;
                         WrStr("Fitness Improvement: "); *)
                         RealToStr(LONGREAL(cga^.stats.f_improve),5,FALSE,str,ok);
                         WrStr(str); WrLn;
                         (*WrStr("Last Explicitly Computed HD: ");
                         FOR i:=1 TO CHROM_LENGTH DO
                             WrReal(cga^.stats.h_dist[i],5,5);
                         END;(*FOR*) WrLn; *)
    END; (*CASE*)
END Display;


BEGIN (*MODULE ssGA*)
      RANDOMIZE;
END cssGA. (*MODULE*)