(****************************************************************************)
(***          DISTRIBUTED GENETIC ALGORITHM - Abstract Data Type          ***)
(***                v 1.0, by Enrique Alba Torres, 4/11/96                ***)
(****************************************************************************)

IMPLEMENTATION MODULE dssGA;
FROM ssGA    IMPORT Init_GA, Set_Up_GA, Get_Stats, Steps_Up, Copy_From,
                    Compute_HammingD, Schema_Stats, Get_Individual,
                    INDIVIDUAL, CHROMOSOME, CHROM_LENGTH, POP_SIZE;
FROM Storage IMPORT ALLOCATE, DEALLOCATE;
FROM Lib     IMPORT RANDOMIZE, RANDOM;

FROM IO IMPORT WrReal, WrStr;


TYPE dGA        = POINTER TO DGA_RECORD;
     POP_ARRAY  = ARRAY[1..POPS_NUMBER] OF GA;
     DGA_RECORD = RECORD
                                  (*---Private parameters---*)
                      subpopulation        : POP_ARRAY;
                      global_stats         : STATS;
                      actual_pstep         : LONGCARD;
                      best_ga_id           : CARDINAL;
                      mig_type             : MIG_TYPE;
                      mig_number           : CARDINAL;
                                  (*---User defined parameters---*)
                      seed                 : CARDINAL;
                      pc, pm               : REAL;
                      fitness_function     : EVAL_FUNC;
                 END; (*DGA_RECORD*)


(************************** PRIVATE PROCEDURES *******************************)

(*Compute the global stats OF the distributed steady state GA*)
PROCEDURE Compute_Global_Stats( VAR dga:dGA );
VAR i           :CARDINAL;
    subpop_stats: STATS;
BEGIN
    dga^.best_ga_id              := 0;
    dga^.global_stats.min_f      := 99999999999999.0;
    dga^.global_stats.avg_f      := 0.0;
    dga^.global_stats.max_f      := 0.0;
    dga^.global_stats.sum_f      := 0.0;
    dga^.global_stats.p_on_sum   := 0.0;
    dga^.global_stats.p_off_sum  := 0.0;
    dga^.global_stats.f_improve  := 0.0;

    (*Gather statistiscs*)
    FOR i:=1 TO POPS_NUMBER DO
        subpop_stats:=Get_Stats(dga^.subpopulation[i]);
        IF (subpop_stats.min_f<dga^.global_stats.min_f) THEN
           dga^.global_stats.min_f := subpop_stats.min_f;
        END; (*IF*)
        IF (subpop_stats.max_f>dga^.global_stats.max_f) THEN
           dga^.global_stats.max_f := subpop_stats.max_f;
           dga^.best_ga_id         := i;
        END; (*IF*)
        dga^.global_stats.sum_f     := dga^.global_stats.sum_f +
                                       subpop_stats.sum_f;
        dga^.global_stats.avg_f     := dga^.global_stats.avg_f +
                                       subpop_stats.avg_f;
        dga^.global_stats.p_on_sum  := dga^.global_stats.p_on_sum +
                                       subpop_stats.p_on_sum;
        dga^.global_stats.p_off_sum := dga^.global_stats.p_off_sum +
                                       subpop_stats.p_off_sum;
        dga^.global_stats.f_improve := dga^.global_stats.f_improve +
                                       subpop_stats.f_improve;

    END; (*FOR*)
    dga^.global_stats.avg_f     := dga^.global_stats.avg_f     / REAL(POPS_NUMBER);
    dga^.global_stats.f_improve := dga^.global_stats.f_improve / REAL(POPS_NUMBER);
END Compute_Global_Stats;


(*Make the migration among the subpopulations in the ring and recompute stats*)
PROCEDURE Ring_Migration( dga: dGA );
VAR i, which_ind:CARDINAL;
BEGIN

    IF (dga^.mig_type=best_ind)
       THEN which_ind := 1;
       ELSE which_ind := RANDOM(POP_SIZE) + 1;

    END; (*IF*)

    FOR i:=1 TO POPS_NUMBER-1 DO
        Copy_From(dga^.subpopulation[i], dga^.subpopulation[i+1],which_ind);
    END; (*FOR*)
    Copy_From(dga^.subpopulation[POPS_NUMBER],dga^.subpopulation[1], which_ind);

    Compute_Global_Stats(dga);
    INC(dga^.mig_number);
END Ring_Migration;

(*RE: Compute the defining length of a schema*)
PROCEDURE Defining_Length( S: SCHEMA ) : LONGCARD;
VAR left, right : LONGCARD;
    i           : CARDINAL;
BEGIN
    left := 0; right := 0;       (*0 means 'no still computed value'*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((left=0) AND (S[i]<>'*')) THEN left :=LONGCARD(i); END;
        IF (S[i]<>'*')                THEN right:=LONGCARD(i); END;
    END;
    RETURN (right-left);
END Defining_Length;

(*RE: Compute the order of a schema*)
PROCEDURE Order( S: SCHEMA ) : LONGCARD;
VAR ndef_pos : LONGCARD;
    i        : CARDINAL;
BEGIN
    ndef_pos := 0; (*Number of defined positions in the schema*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF (S[i]<>'*') THEN INC(ndef_pos); END;
    END;
    RETURN ndef_pos;
END Order;

(*RE: Decides wether a string is an instance of a schema or NOT*)
PROCEDURE Is_Instance( str:CHROMOSOME; S:SCHEMA ) : BOOLEAN;
VAR i: CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((str[i]=1) AND (S[i]='0')) THEN RETURN FALSE; END;
        IF ((str[i]=0) AND (S[i]='1')) THEN RETURN FALSE; END;
    END;
    RETURN TRUE;
END Is_Instance;

(*RE: Count the number of instances of a schema in the population*)
PROCEDURE Schema_Count( ga:GA; S:SCHEMA; VAR ni:LONGCARD; VAR ssum_f: REAL);
VAR i   : CARDINAL;
    ind : INDIVIDUAL;
BEGIN
    ni := 0; ssum_f := 0.0;
    FOR i:=1 TO POP_SIZE DO
        ind := Get_Individual(ga,i);
        IF (Is_Instance(ind.chr, S)) THEN
           INC(ni);
           ssum_f := ssum_f + ind.fitness;
        END; (*IF*)
    END; (*FOR*)

END Schema_Count;



(**************************** PUBLIC METHODS *********************************)

(*You must initialise the dGA before using it!*)
PROCEDURE Init_dGA( VAR dga:dGA );
BEGIN
    dga:=NIL;
END Init_dGA;


(*Set the parameters for the distributed GA*)
PROCEDURE Set_Up_dGA( VAR dga              : dGA;
                          seed             : CARDINAL;              (*unused!*)
                          pc, pm           : REAL;
                          who_mig          : MIG_TYPE;
                          fitness_function : EVAL_FUNC
                    );
VAR i           : CARDINAL;

BEGIN
    NEW(dga);
    FOR i:=1 TO POPS_NUMBER DO
        Init_GA(dga^.subpopulation[i]);
        Set_Up_GA(dga^.subpopulation[i],seed,pc,pm,fitness_function);
    END; (*FOR*)

    dga^.mig_number              := 0;
    dga^.seed                    := seed;
    dga^.pc                      := pc;
    dga^.pm                      := pm;
    dga^.fitness_function        := fitness_function;
    dga^.mig_type                := who_mig;
    dga^.actual_pstep            := 0;

    Compute_Global_Stats(dga);

END Set_Up_dGA;

(*Makes evolution steps in every subpopulation. Returns the actual pstep*)
PROCEDURE dSteps_Up( VAR dga:dGA; psteps:LONGCARD ) : LONGCARD;
VAR trash, i, p: LONGCARD;
BEGIN

    FOR i:=1 TO psteps DO
        FOR p:=1 TO POPS_NUMBER DO
            trash:=Steps_Up(dga^.subpopulation[CARDINAL(p)],1);
        END; (*FOR*)
        INC(dga^.actual_pstep);
        IF((dga^.actual_pstep MOD MIG_GAP)=0) THEN Ring_Migration(dga); END;

    END; (*FOR*)

    Compute_Global_Stats(dga); (*Only at the END we gather statistics*)

    RETURN dga^.actual_pstep;
END dSteps_Up;

(*Computes the actual hamming distance. Call it explicity if you need HD!*)
(*Computations are extended all over the full subpopulations*)
PROCEDURE dCompute_HammingD( VAR dga:dGA );
VAR i:CARDINAL;
BEGIN
    FOR i:=1 TO POPS_NUMBER DO
        Compute_HammingD(dga^.subpopulation[i]);
    END; (*FOR*)
END dCompute_HammingD;

(*Compute the stats about the parameter -schema- in the whole dGA*)
PROCEDURE dSchema_Stats( dga: dGA; S: SCHEMA ) : S_STATS;
          VAR stat      : S_STATS;
              ssum_f, c : REAL;
              ai        : LONGCARD;
              i         : CARDINAL;
BEGIN
    c := 1.0 - (dga^.pc*REAL(Defining_Length(S))/REAL(CHROM_LENGTH-1) -
                dga^.pm*REAL(Order(S)));
    stat.actual_avg_f  := 0.0; stat.actual_instances := 0;
    stat.sch_th_instances := 0;
    FOR i:=1 TO POPS_NUMBER DO
        Schema_Count( dga^.subpopulation[i], S, ai, ssum_f );
        stat.actual_instances := stat.actual_instances+ai;
        stat.actual_avg_f     := stat.actual_avg_f + ssum_f;
    END; (*FOR*)

    IF(stat.actual_instances>0) THEN
           stat.actual_avg_f := stat.actual_avg_f / REAL(stat.actual_instances);
    ELSE  stat.actual_avg_f := 0.0;
    END; (*IF*)

    stat.sch_th_instances := LONGCARD(
           (REAL(stat.actual_instances)*
                (stat.actual_avg_f/dga^.global_stats.avg_f))*c
                 +0.5 (*round*)            );

    RETURN stat;
END dSchema_Stats;

(*Returns a record with statistics of the distributed GA*)
PROCEDURE dGet_Stats( dga:dGA ) : STATS;
BEGIN
    RETURN dga^.global_stats;
END dGet_Stats;

(*Returns a pointer to the desired GA FOR further -external- manipulations*)
PROCEDURE Get_GA( dga:dGA; ga_id:CARDINAL; VAR ga:GA );
BEGIN
    IF (ga_id>=1) AND (ga_id<=POPS_NUMBER)
       THEN ga := dga^.subpopulation[ga_id];
    END; (*IF*)
END Get_GA;

(*Returs the GAid of the subpopulation with the best final solution*)
PROCEDURE Get_Best_GAid( dga: dGA ) : CARDINAL;
BEGIN
    RETURN dga^.best_ga_id;
END Get_Best_GAid;

BEGIN (*dssGA*)
      RANDOMIZE;
END dssGA. (*MODULE*)
