(****************************************************************************)
(***                GENETIC ALGORITHM - Abstract Data Type                ***)
(***                v 2.0, by Enrique Alba Torres, 4/11/96                ***)
(****************************************************************************)
IMPLEMENTATION MODULE ssGA;
FROM IO      IMPORT WrLn, WrLngCard, WrReal, WrCard,     WrStr,RdCard;
FROM Lib     IMPORT RAND, RANDOM, RANDOMIZE;
FROM Storage IMPORT ALLOCATE, DEALLOCATE;
FROM Str     IMPORT RealToStr;

TYPE
   POPULATION= ARRAY[1..POP_SIZE] OF INDIVIDUAL;
   GA        = POINTER TO GA_RECORD;        (*Steady State Genetic Algorithm*)
   GA_RECORD = RECORD
                                (*---Private parameters---*)
                    pop                  : POPULATION;
                    stats                : STATS;
                    actual_step          : LONGCARD;
                                (*---User defined parameters---*)
                    seed                 : CARDINAL;
                    pc, pm               : REAL;
                    fitness_function     : EVAL_FUNC;
               END; (*GA_RECORD*)

(*Roulette Wheel, Double Point Crossover, Complement Mutation, Replace Worst*)
(*Population is ordered. Best individual in position 1*)

(************************* AUXILIAR PROCEDURES ******************************)
PROCEDURE Flip( Pr : REAL ) : BOOLEAN;
BEGIN
    IF((Pr=1.0) OR (Pr>=RAND())) THEN RETURN TRUE;
                                 ELSE RETURN FALSE;
    END; (*IF*)
END Flip;



PROCEDURE Decode( chr : CHROMOSOME ) : PHENOTYPE;
VAR
  ph : PHENOTYPE;
  i, j, index : CARDINAL;
BEGIN
    index := 1;
    FOR i:=1 TO GENE_NUMBER DO
        ph[i] := REAL(chr[index]);
        INC(index);
        FOR j:=1 TO GENE_LENGTH-1 DO
            ph[i] := 2.0*ph[i] + REAL(chr[index]);
            INC(index);
        END; (*A gene*)
    END; (*All the genes*)

    RETURN ph;

END Decode;

(*Roulette Wheel selection mechanism*)
PROCEDURE RW_Select( ga : GA ) : CARDINAL;
VAR select_ind   : CARDINAL;
    target, acum : REAL;
BEGIN
    target     := RAND()*ga^.stats.sum_f;
    acum       := ga^.pop[1].fitness;
    select_ind := 1;
    WHILE (acum<target) DO
          acum := acum + ga^.pop[select_ind].fitness;
          INC(select_ind);
    END; (*WHILE*)
    IF select_ind>POP_SIZE THEN select_ind:=POP_SIZE; END;  (*UNNECESSARY!!!*)
    RETURN select_ind;
END RW_Select;

(*Double Point Crossover*)
PROCEDURE Dpx( ga : GA; i1, i2 : CARDINAL; VAR aux_chrom : CHROMOSOME);
VAR
  i, j, p1, p2, index : CARDINAL;
BEGIN
    IF(NOT Flip(ga^.pc)) THEN
           IF(ga^.pop[i1].fitness>ga^.pop[i2].fitness)
              THEN index := i1;
              ELSE index := i2;
           END; (*IF*)
           FOR i:= 1 TO CHROM_LENGTH DO
               aux_chrom[i] := ga^.pop[index].chr[i];
           END; (*FOR*)
           RETURN;
    END; (*IF*)
        (*---Initialisations---*)
    index := 1;
        (*---Select and sort the two different crossover points---*)
    p1    := RANDOM(CHROM_LENGTH)+1;
    REPEAT p2 := RANDOM(CHROM_LENGTH)+1; UNTIL (p1<>p2);
    IF(p1>p2) THEN i:=p1; p1:=p2; p2:=i; END;
        (*---Build the new individual---*)
    FOR i:=1 TO p1 DO
        aux_chrom[index]:=ga^.pop[i1].chr[i];   INC(index);
    END;
    FOR i:=p1+1 TO p2 DO
        aux_chrom[index]:=ga^.pop[i2].chr[i];   INC(index);
    END;
    FOR i:=p2+1 TO CHROM_LENGTH DO
        aux_chrom[index]:=ga^.pop[i1].chr[i];   INC(index);
    END;

END Dpx;

(*Mutate individual by inverting bits*)
PROCEDURE Mutate( ga : GA; VAR ch : CHROMOSOME );
VAR i : CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO
        IF(Flip(ga^.pm)) THEN ch[i]:=1-ch[i]; END;(*IF*)
    END; (*Mutate every allele value*)
END Mutate;

(*Assign two individuals*)
PROCEDURE Assign_Ind( VAR l : INDIVIDUAL; r : INDIVIDUAL );
VAR i:CARDINAL;
BEGIN
    l := r;
END Assign_Ind;

PROCEDURE Less_Ind( ga:GA; i1:CARDINAL; i2:CARDINAL ):BOOLEAN;      (*i1<i2?*)
BEGIN
    IF(ga^.pop[i1].fitness>ga^.pop[i2].fitness)   (*Criterium is INVERTED!!!*)
       THEN RETURN TRUE;
       ELSE RETURN FALSE;
    END; (*IF*)
END Less_Ind;

PROCEDURE Swap_Ind( VAR i1 :INDIVIDUAL; VAR i2 : INDIVIDUAL );(*Swaps 2 ind.*)
VAR aux : INDIVIDUAL;
BEGIN
    Assign_Ind( aux, i1  );
    Assign_Ind( i1,  i2  );
    Assign_Ind( i2,  aux );
END Swap_Ind;

(*Returns 0 If fitnesses are equal, otherwise returns the index of the left-*
 *most two different fitnesses.                                             *)
PROCEDURE Find_Pivot( VAR pop:POPULATION; i, j:CARDINAL ) : CARDINAL;
VAR firstkey : REAL;        (*First fitness-key value*)
    k        : CARDINAL;    (*Runs left to right looking for a diff. fitness*)
BEGIN
    firstkey := pop[i].fitness;
    FOR k:=i+1 TO j DO (*Scan for different key*)
        IF pop[k].fitness>firstkey THEN (*Select the larger fitness*)
           RETURN k;
        ELSIF pop[k].fitness<firstkey THEN
           RETURN i;
        END; (*IF*)
    END; (*FOR*)
    RETURN 0; (*Different fitness values were never found*)
END Find_Pivot;


(*Makes a partition between i..j so that fitnesses are lower than the pivot *)
PROCEDURE Partition( VAR pop:POPULATION; i,j:CARDINAL; pivot:REAL) : CARDINAL;
VAR l, r : CARDINAL; (*Cursors*)
BEGIN
    l := i;
    r := j;
    REPEAT
         Swap_Ind(pop[l],pop[r]);
         WHILE pop[l].fitness< pivot DO INC(l); END; (*WHILE*)
         WHILE pop[r].fitness>=pivot DO DEC(r); END; (*WHILE*)
    UNTIL l>r;
    RETURN l;
END Partition;

(*Inverts the population order*)
PROCEDURE Invert_Pop( VAR pop : POPULATION );
VAR i : CARDINAL;
BEGIN
    FOR i:=1 TO POP_SIZE DIV 2 DO
        Swap_Ind(pop[i],pop[POP_SIZE-i+1]);
    END; (*FOR*)
END Invert_Pop;

(*Sort the population based on fitnesses values. If all fit. are equal ...!*)
PROCEDURE QSort( VAR pop  : POPULATION; i, j : CARDINAL );
VAR pivot       : REAL;
    pivot_index : CARDINAL;
    k           : CARDINAL;
BEGIN
    pivot_index := Find_Pivot(pop,i,j);
    IF pivot_index<>0 THEN          (*Do nothing if all the values are equal*)
       pivot := pop[pivot_index].fitness;
       k     := Partition(pop,i,j,pivot);
       QSort(pop,i,k-1);
       QSort(pop,k,j  );
    END; (*IF*)
END QSort;

(*Inserts in a sorted mode a new individual and remakes the statistics*)
PROCEDURE Insert_New_Ind( ind:INDIVIDUAL; VAR ga:GA );
VAR insert_point, i : CARDINAL;
BEGIN
            (*---Sorted insertion in the population---*)
    insert_point := 1;
    WHILE (insert_point<=POP_SIZE) AND
          (ind.fitness<ga^.pop[insert_point].fitness)
                   DO INC(insert_point);
    END; (*WHILE*)
    IF(insert_point<POP_SIZE) THEN   (*Otherwise fitness is worst than worst*)
              (*---Worst individual leaves population---*)
          ga^.stats.sum_f := ga^.stats.sum_f - ga^.pop[POP_SIZE].fitness;
          FOR i:=POP_SIZE TO insert_point+1 BY -1 DO
              ga^.pop[i] := ga^.pop[i-1];   (*Record assignment!*)
          END; (*FOR*)
          ga^.pop[insert_point].fitness := ind.fitness;
          ga^.pop[insert_point].chr     := ind.chr;
              (*---Stats changes---*)
          ga^.stats.sum_f     := ga^.stats.sum_f + ind.fitness;
          ga^.stats.avg_f     := ga^.stats.sum_f / REAL(POP_SIZE);
          ga^.stats.max_f     := ga^.pop[1].fitness;
          ga^.stats.min_f     := ga^.pop[POP_SIZE].fitness;
    END; (*IF*)

END Insert_New_Ind;


(****************************** PUBLIC METHODS ******************************)

(*Computes the actual averaged hamming distance IN pop. Binary strings only!*)
PROCEDURE Compute_HammingD( VAR ga:GA );
VAR i, j: CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO ga^.stats.h_dist[i] := 0.0; END;(*FOR*)
    FOR i:=1 TO CHROM_LENGTH DO
        FOR j:=1 TO POP_SIZE DO
            ga^.stats.h_dist[i]:=ga^.stats.h_dist[i]+REAL(ga^.pop[j].chr[i]);
        END; (*FOR*)
        ga^.stats.h_dist[i] := ga^.stats.h_dist[i] / REAL(POP_SIZE);
    END;(*FOR*)
END Compute_HammingD;

(*Compute the defining length of a schema*)
PROCEDURE Defining_Length( S: SCHEMA ) : LONGCARD;
VAR left, right : LONGCARD;
    i           : CARDINAL;
BEGIN
    left := 0; right := 0;       (*0 means 'no still computed value'*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((left=0) AND (S[i]<>'*')) THEN left :=LONGCARD(i); END;
        IF (S[i]<>'*')                THEN right:=LONGCARD(i); END;
    END;
    RETURN (right-left);
END Defining_Length;

(*Compute the order of a schema*)
PROCEDURE Order( S: SCHEMA ) : LONGCARD;
VAR ndef_pos : LONGCARD;
    i        : CARDINAL;
BEGIN
    ndef_pos := 0; (*Number of defined positions in the schema*)
    FOR i:=1 TO CHROM_LENGTH DO
        IF (S[i]<>'*') THEN INC(ndef_pos); END;
    END;
    RETURN ndef_pos;
END Order;

(*Decides wether a string is an instance of a schema or NOT*)
PROCEDURE Is_Instance( str:CHROMOSOME; S:SCHEMA ) : BOOLEAN;
VAR i: CARDINAL;
BEGIN
    FOR i:=1 TO CHROM_LENGTH DO
        IF ((str[i]=1) AND (S[i]='0')) THEN RETURN FALSE; END;
        IF ((str[i]=0) AND (S[i]='1')) THEN RETURN FALSE; END;
    END;
    RETURN TRUE;
END Is_Instance;

(*Count the number of instances of a schema in the population*)
PROCEDURE Schema_Count( ga:GA; S:SCHEMA; VAR ni:LONGCARD; VAR ssum_f: REAL);
VAR i :CARDINAL;
BEGIN
    ni := 0; ssum_f := 0.0;
    FOR i:=1 TO POP_SIZE DO
        IF (Is_Instance(ga^.pop[i].chr, S)) THEN
           INC(ni);
           ssum_f := ssum_f + ga^.pop[i].fitness;
        END; (*IF*)
    END; (*FOR*)

END Schema_Count;

(*Compute the stats about the parameter -schema- AND predict the next number*)
PROCEDURE Schema_Stats( ga: GA; S: SCHEMA ) : S_STATS;
VAR stat      : S_STATS;
    ssum_f, c : REAL;
BEGIN
    c := 1.0 - (ga^.pc*REAL(Defining_Length(S))/REAL(CHROM_LENGTH-1) -
               ga^.pm*REAL(Order(S)));
    stat.actual_avg_f  := 0.0; stat.actual_instances := 0;
    stat.sch_th_instances := 0;
    Schema_Count( ga, S, stat.actual_instances, ssum_f );
    IF(stat.actual_instances>0) THEN
           stat.actual_avg_f := ssum_f / REAL(stat.actual_instances);
    ELSE  stat.actual_avg_f := 0.0;
    END; (*IF*)
    stat.sch_th_instances := LONGCARD(
           (REAL(stat.actual_instances)*stat.actual_avg_f/ga^.stats.avg_f)*c
           +0.5 (*round*)            );
    RETURN stat;

END Schema_Stats;


(*Initialise the ga variable*)
PROCEDURE Init_GA( VAR ga:GA );
BEGIN
    ga := NIL;
END Init_GA;

(*Set the parameters for the GA*)
PROCEDURE Set_Up_GA( VAR ga            : GA;
                     seed              : CARDINAL;
                     pc, pm            : REAL;
                     fitness_function  : EVAL_FUNC
                     );
VAR i, j : CARDINAL;


BEGIN
    NEW(ga);
    ga^.seed             := seed;
    ga^.pc               := pc;
    ga^.pm               := pm;
    ga^.fitness_function := fitness_function;
    ga^.actual_step      := 0;
    ga^.stats.min_f      := 0.0;
    ga^.stats.avg_f      := 0.0;
    ga^.stats.max_f      := 0.0;
    ga^.stats.sum_f      := 0.0;
    ga^.stats.p_on_sum   := 0.0;
    ga^.stats.p_off_sum  := 0.0;
    ga^.stats.f_improve  := 0.0;

    FOR i:=1 TO POP_SIZE DO
        FOR j:=1 TO CHROM_LENGTH DO
            IF(Flip(0.5)) THEN ga^.pop[i].chr[j] := 1;
                          ELSE ga^.pop[i].chr[j] := 0;
            END; (*IF*)
        END; (*CHROMOSOME*)
        ga^.pop[i].fitness := ga^.fitness_function( Decode(ga^.pop[i].chr) );
        ga^.stats.sum_f    := ga^.stats.sum_f + ga^.pop[i].fitness;
    END; (*POPULATION*)

    ga^.stats.avg_f := ga^.stats.sum_f / REAL(POP_SIZE);
    QSort( ga^.pop, 1, POP_SIZE ); (*Sort from worst to best fitness*)
    Invert_Pop(ga^.pop);           (*Sort descending fitness values*)
    ga^.stats.p_on_sum   := ga^.stats.avg_f;
    ga^.stats.p_off_sum  := ga^.pop[1].fitness;
    ga^.stats.min_f      := ga^.pop[POP_SIZE].fitness;
    ga^.stats.max_f      := ga^.pop[1].fitness;
    Compute_HammingD(ga);

    (*Display(ga,population,1);---commented for distributed convenience!!!*)
END Set_Up_GA;

(*Makes evolution steps. Returns the actual step*)
PROCEDURE Steps_Up( VAR ga:GA; steps:LONGCARD ) : LONGCARD;
VAR aux_chrom   : CHROMOSOME;
    aux_ind     : INDIVIDUAL;
    p1, p2, i   : CARDINAL;
    fitness     : REAL;
    s           : LONGCARD;
BEGIN
 FOR s:=1 TO steps DO
    INC(ga^.actual_step);
    p1 := RW_Select(ga);
    p2 := RW_Select(ga);
    Dpx(ga,p1,p2,aux_chrom);
    Mutate(ga,aux_chrom);
    fitness := ga^.fitness_function( Decode(aux_chrom) );

    aux_ind.fitness     := fitness;
    aux_ind.chr         := aux_chrom;
    ga^.stats.f_improve := ga^.stats.avg_f;

    Insert_New_Ind( aux_ind, ga );

    ga^.stats.f_improve := ga^.stats.avg_f     - ga^.stats.f_improve;
    ga^.stats.p_off_sum := ga^.stats.p_off_sum + ga^.pop[1].fitness;
    ga^.stats.p_on_sum  := ga^.stats.p_on_sum  + ga^.stats.avg_f;
  END; (*MAIN FOR*)
    RETURN ga^.actual_step;
END Steps_Up;

(*Copy the ind-th individual from the ga_from to the ga_to*)
PROCEDURE Copy_From ( ga_from:GA; VAR ga_to:GA; ind:CARDINAL );
BEGIN

END Copy_From;


(*Returns a record with statistics of the GA*)
PROCEDURE Get_Stats(ga:GA) : STATS;
BEGIN
   RETURN ga^.stats;
END Get_Stats;

(*Returns the individual number 'ind'*)
PROCEDURE Get_Individual( ga:GA; ind:CARDINAL) : INDIVIDUAL;
BEGIN

        IF ((ind>=1) AND (ind<=POP_SIZE)) THEN
           RETURN ga^.pop[ind];
        END; (*IF*)
END Get_Individual;



(*Displays the 'item' in the screen. If item='individual' then use 'ind_pos'*)
PROCEDURE Display( ga:GA; item:ITEM; ind_pos:CARDINAL );
VAR i : CARDINAL;
    bien : BOOLEAN;
    str  : ARRAY[1..32] OF CHAR;
BEGIN
    CASE item OF
         individual :    WrStr("INDIVIDUAL #");WrCard(ind_pos,4);
                         WrStr("===> ");
                         FOR i:=1 TO CHROM_LENGTH DO
                             WrCard(CARDINAL(ga^.pop[ind_pos].chr[i]),0);
                         END; (*FOR*)
                         WrReal(ga^.pop[ind_pos].fitness,10,12);
                         WrLn;
                         |
         best       :    Display(ga,individual,1);
                         |
         worst      :    Display(ga,individual,POP_SIZE);
                         |
         population :    FOR i:=1 TO POP_SIZE DO
                             Display(ga,individual,i);
                         END; (*FOR*)
                         |
         statistics :    (*WrStr("---STATISTICS---"); WrLn;
                         WrStr("Fitness Improvement: "); *)
                         RealToStr(LONGREAL(ga^.stats.f_improve),5,FALSE,str,bien);
                         WrStr(str); WrLn;
                         (*WrStr("Last Explicitly Computed HD: ");
                         FOR i:=1 TO CHROM_LENGTH DO
                             WrReal(ga^.stats.h_dist[i],5,5);
                         END;(*FOR*) WrLn; *)
    END; (*CASE*)
END Display;


BEGIN (*MODULE ssGA*)
      RANDOMIZE;
END ssGA. (*MODULE*)