% UMA Forum Interpreter
% version 0.1
%
% author: Pablo Lopez
% e-mail: lopez@lcc.uma.es
% www: http://www.lcc.uma.es/personal/lopez
%
% Dept. of Computer Science
% University of Malaga, SPAIN

/*
   This source code is organised into the following major sections:

     - syntax and semantics checkings
     - implementation of modules, operators, clauses, switches,
       and module usage tables
     - access to source files: reading, parsing, analyzing, editing, ...
     - some common predicates and utilities
     - "read-eval-print" loop and the user interface
     - eager sequent calculus
     - lazy sequent calculus
*/

% ------------------ SYNTAX AND SEMANTICS CHECKINGS -----------------

% I use a fairly simple three-step approach to parse Forum programs:
%
%   1. define Forum connectives and keywords as Prolog operators
%   2. rely on the read_term/2 built-in Prolog predicate to read
%      Prolog terms
%   3. check the syntax of Prolog terms and classify them as
%      UMA Forum constructions; i.e. module headers,
%      terms, and operator declarations
%
% This approach has certain shortcomings, some of them commented
% below. I intend to replace it with a low-level parser...

% --------------- UMA Forum connectives ---------------

%  top
%  bot

% :- op(500, fy,  (?)).
:- op(550, yfx, (&)).
:- op(600, yfx, (#)).
:- op(650, xfy, (-*)).
:- op(650, xfy, (=>)).

:- op(700, xfy, (\)).
:- op(700, fy,  (forall)).       % forall X \ p(X).

:- op(750, yfx, (*-)).
:- op(750, yfx, (<=)).

% ------------- UMA Forum keywords -------------

:- op(1000, fx, (linear)).
:- op(1000, fx, (module)).
:- op(1000, fx, (infix)).
:- op(1000, fx, (infixl)).
:- op(1000, fx, (infixr)).
:- op(1000, fx, (prefix)).
:- op(1000, fx, (postfix)).

% --------------- reserved atoms ----------------

isConnective(T) :- isAsynchronous(T) ; isSynchronous(T).

isAsynchronous(top).
isAsynchronous(bot).
% isAsynchronous((?)).
isAsynchronous((&)).
isAsynchronous((#)).
isAsynchronous((-*)).
isAsynchronous((=>)).
isAsynchronous((forall)).
isAsynchronous((*-)).
isAsynchronous((<=)).

isSynchronous(_) :- fail.

isKeyword((module)).
isKeyword((linear)).
isKeyword((infix)).
isKeyword((infixl)).
isKeyword((infixr)).
isKeyword((prefix)).
isKeyword((postfix)).

% predefined operators

                      % syntactic
isPredOp((\)).        % abstraction (forall X \ p(X), lambda terms)
isPredOp((:)).        % declaration separator (operators, type, etc)
isPredOp(',').        % term separator (p(X,y,Z))

isPredOp((=)).        % term unification, identity and decomposition
isPredOp((\=)).
isPredOp((==)).
isPredOp((\==)).
isPredOp((=..)).

isPredOp((<)).        % relational
isPredOp((=<)).
isPredOp((>)).
isPredOp((>=)).

isPredOp((+)).        % arithmetic
isPredOp((-)).
isPredOp((*)).
isPredOp((/)).
isPredOp((mod)).
isPredOp((is)).

isPredOp((.)).        % Prolog lists
isPredOp([]).

% -------------- syntax checking ---------------

% classify and dissect Forum module headers, terms,
% and operator declarations

% isHeader: returns name of the module (N)

isHeader(T,_) :- var(T), !, fail.
isHeader((module N), N) :- isLCID(N).

% isTerm(_,_).   it's a prolog term, so it's a Forum term...

% isOpDecl: returns atom, associativity, and priority (At, As, P)

isOpDecl(T,_,_,_) :- var(T), !, fail.
isOpDecl((infix   A : P), A, (infix),   P) :- atom(A), integer(P).
isOpDecl((infixl  A : P), A, (infixl),  P) :- atom(A), integer(P).
isOpDecl((infixr  A : P), A, (infixr),  P) :- atom(A), integer(P).
isOpDecl((prefix  A : P), A, (prefix),  P) :- atom(A), integer(P).
isOpDecl((postfix A : P), A, (postfix), P) :- atom(A), integer(P).

% check identifier syntax

isLCID(I) :- atom(I), name(I, [Ch|_]), isLower(Ch).

isLower(Ch):- 97 =< Ch, Ch =< 122. %  97 = a, 122 = z

isUCID(I) :- var(I).

isID(I) :- isUCID(I) ; isLCID(I).

% --------------- semantics checking ---------------

% Since we do not have a type system yet, semantics checking is
% trivial. In particular, we check clauses, goals, terms,
% abstractions, and operator declarations.
%
% For terms denoting UMA Forum clauses, we discard variable
% terms.
%
% For any term, we check that the operators have been defined.
% This way, Prolog predefined operators cannot be freely used. Hence,
% the only allowed operators are connectives, predefined, and those
% explicitly defined by UMA Forum operator declarations.
%
% This operator check has the following *LIMITATIONS*:
%
%  1) Prolog operators can be defined several times with different
%     infixities; but currently infixity is *NOT* checked.
%
%     Example: if you define ++ as a postfix Forum operator and
%     your Prolog system has ++ defined as prefix, this check cannot
%     detect illegal prefix use of ++.
%
%  2) Prolog operators remain defined, so they cannot be used as
%     predicate names with the same arity as the predefined operator,
%     since the illegal use of infix notation cannot be detected;
%     so, use of predefined Prolog operators is forbidden although
%     infix notation is not used.
%
%     Example: if your Prolog system defines the prefix operator
%     'dynamic/1', you cannot use dynamic(X) on your Forum code. You
%     can, however, use dynamic(X,Y,...); i.e. dynamic/n as a functor
%     with any arity n \= 1.
%
% Finally, abstraction variables (var \ term) are also checked.
%
% To gain control over operators, a parser is needed instead of using
% the read_term/2 approach...
%

% checkClause(Term, Variables, Singletons)
% success if Term is a Forum clause; fail otherwise

checkClause(V,[N=V],[N]) :-
    var(V),
    !,
    printf("*** ERROR: variable classical clause\n%w.\n", [N]),
    fail.

checkClause((linear V),[N=V],[N]) :-
    var(V),
    !,
    printf("*** ERROR: variable linear clause\nlinear %w.\n",[N]),
    fail.

checkClause(C,V,S) :-
    checkTerms([C]) ->                       % valid clause
         (S = [] ->
               true
           ;
              (warnSingletons(S),            % singleton vars
               printf("%v%w.\n",[V,C])))
       ;
         (printf("%v%w.\n",[V,C]),           % invalid clause
          fail).

% checkGoal(Term, Variables)
% success if Term is a Forum goal; fail otherwise

checkGoal(G,V) :-
   checkTerms([G]) ->                     % valid goal
         true
      ;
         (printf("%v%w.\n",[V,G]),        % invalid goal
          fail).

% checkTerms(Ts) - checks the operator usage

checkTerms([T|Ts]) :- var(T),    !, checkTerms(Ts).         % basic terms
checkTerms([T|Ts]) :- number(T), !, checkTerms(Ts).
checkTerms([T|Ts]) :- isLCID(T), !, checkTerms(Ts).

checkTerms([(V \ T)|Ts]) :-                       % abstraction variables
    isUCID(V) ->
         (!, checkTerms([T|Ts]))
       ;
         (printf("*** ERROR: abstraction variable expected in:\n"), !, fail).

checkTerms([T|Ts]) :-                                    % compound terms
    T =.. [F|Args],
    functor(T,_,N),    % N = length(Args)
    checkFunctor(F,N),
    checkTerms(Args),
    checkTerms(Ts).

checkTerms([]).

checkFunctor(F,N) :-
    current_op(_,I,F),       % it's an operator
    arity(I,N),              % and it's used with its operator arity
    \+(isOperator(F)),       % so I check it's defined...
    printf("*** ERROR: undefined operator '%w' in:\n", [F]),
    !,
    fail.
checkFunctor(_,_).

isOperator(A) :- isConnective(A), !.
isOperator(A) :- isPredOp(A), !.
isOperator(A) :- isKeyword(A), !.
isOperator(A) :- getOperator(A,_).         % user declared operator

arity((fx),  1).
arity((fy),  1).
arity((xf),  1).
arity((yf),  1).
arity((xfx), 2).
arity((yfx), 2).
arity((xfy), 2).
arity((yfy), 2).

warnSingletons([S]) :-
   !,  printf("*** WARNING: singleton variable %w in:\n", [[S]]).
warnSingletons([S|Ss]) :-
   printf("*** WARNING: singleton variables %w in:\n", [[S|Ss]]).

% check operator declarations: atom and priority

checkOpDecl(A,P) :- checkOpAtom(A), checkOpPriority(P).

checkOpAtom(A) :-
    isKeyword(A),
    !,
    printf("*** ERROR: keyword '%w' declared as operator\n", [A]),
    fail.
checkOpAtom(A) :-
    isConnective(A),
    !,
    printf("*** ERROR: connective '%w' declared as operator\n", [A]),
    fail.
checkOpAtom(A) :-
    isPredOp(A),
    !,
    printf("*** ERROR: predefined operator '%w' redeclaration\n", [A]),
    fail.
checkOpAtom(A) :-
    getOperator(A,_),
    !,
    printf("*** ERROR: operator '%w' redeclaration\n", [A]),
    fail.
checkOpAtom(_).

highestPriority(1).
lowestPriority(1200).

checkOpPriority(P) :-
    highestPriority(H),
    lowestPriority(L),
    P >= H,
    P =< L,
    !.
checkOpPriority(P) :-
    highestPriority(H),
    lowestPriority(L),
    printf("*** ERROR: priority '%w' out of range [%w,%w]\n", [P,H,L]),
    fail.

% --------- IMPLEMENTATION OF MODULES, OPERATORS, AND CLAUSES ---------

/*
    The implementation consists of three tables, namely:

                       - Modules table
                       - Operators table
                       - Clauses table

    concerned with the storage of UMA Forum code.

    These tables are dealt with as Abstract Data Types.
    This way, the current implementation could be replaced
    by, say, a blackboard based one.

    The ADT operations for inserting into, reading and deleting
    from a table are:

       - defineX(N,..)    define a X named N with certain args
       - getX(N,T)        get the term T representing a X named N
       - getXs(L)         get a list L of all of defined Xs
       - getX(N,L)        get a list L of the Xs defined in module N
       - deleteX(N)       delete the Xs defined in module N
       - deleteXs         delete the Xs defined in all modules

    where:
               X  ::= Module | Operator | Clause

    The tables are stored as sets of dynamic facts

                             forumX(T)

    where T denotes a term representing module, operator
    or clause details. This term is dealt with as an ADT as well,
    whose basic operations have the form:

        XF(T,V)          access to field F of a X represented by T

*/

% --------------- MODULES TABLE ---------------

% a set of dynamic facts 'forumModule(ModuleName, File).'

:- dynamic forumModule/2.

moduleName(forumModule(Name,_), Name).
moduleFile(forumModule(_,File), File).

defineModule(Name,File) :-
      (forumModule(Name,_) ->                % redefining a module ?
            (retract(forumModule(Name,_)),
             !,
             deleteOperators(Name),
             deleteClauses(Name))
         ;
             true),
      assertz(forumModule(Name,File)).

getModule(N,forumModule(N,F)) :- forumModule(N,F).

getModules(Ms):-
      findall(forumModule(N,F), forumModule(N,F), Ms).

deleteModule(N) :-
      retract(forumModule(N,_)),
      !,
      deleteOperators(N),
      deleteClauses(N),
      unuseModule(N).

deleteModules :-
      retractall(forumModule(_,_)),
      deleteOperators,
      deleteClauses,
      unuseModules.

% -------------- OPERATORS TABLE --------------

% a set of dynamic facts forumOperator(ModuleName, Atom, Assoc, Priority)

:- dynamic forumOperator/4.

operatorModule(forumOperator(N,_,_,_),N).
operatorAtom(forumOperator(_,A,_,_),A).
operatorAssoc(forumOperator(_,_,A,_),A).
operatorPriority(forumOperator(_,_,_,P),P).

defineOperator(N,A,(infix),P) :-
   !, op(P,xfx,A), assertz(forumOperator(N,A,(infix),P)).
defineOperator(N,A,(infixl),P) :-
   !, op(P,yfx,A), assertz(forumOperator(N,A,(infixl),P)).
defineOperator(N,A,(infixr),P) :-
   !, op(P,xfy,A), assertz(forumOperator(N,A,(infixr),P)).
defineOperator(N,A,(prefix),P) :-
   !, op(P,fy,A), assertz(forumOperator(N,A,(prefix),P)).
defineOperator(N,A,(postfix),P) :-
   !, op(P,yf,A), assertz(forumOperator(N,A,(postfix),P)).

getOperator(At,forumOperator(N,At,As,P)) :- forumOperator(N,At,As,P).

getOperators(N,L) :-
   findall(forumOperator(N,At,As,P),forumOperator(N,At,As,P),L).

getOperators(L) :-
   findall(forumOperator(N,At,As,P),forumOperator(N,At,As,P),L).

deleteOperators(N) :-
   retractall(forumOperator(N,_,_,_)).

deleteOperators:-
   retractall(forumOperator(_,_,_,_)).

% --------------- CLAUSES TABLE ---------------

% a set of dynamic facts
%
%      'forumClause(ModuleName, Kind, Formula, Variables).'
%
% where kind is either 'linear' or 'classical'

:- dynamic forumClause/4.

clauseModule(forumClause(N,_,_,_),N).
clauseKind(forumClause(_,K,_,_),K).
clauseFormula(forumClause(_,_,F,_),F).
clauseVariables(forumClause(_,_,_,V),V).

defineClause(N,(linear F),V) :-
    !, assertz(forumClause(N,(linear),F,V)).

defineClause(N,F,V) :-
    assertz(forumClause(N,(classical),F,V)).

getClauses(N,Cs) :-
    findall(forumClause(N,K,C,V), forumClause(N,K,C,V), Cs).

getClauses(Cs) :-
    findall(forumClause(N,K,C,V), forumClause(N,K,C,V), Cs).

deleteClauses :-
    retractall(forumClause(_,_,_,_)).

deleteClauses(N) :-
    retractall(forumClause(N,_,_,_)).

% --------------- SWITCHES TABLE ---------------

/*
   This table contains the names and current values of
   the environment (i.e. interpreter + user interface) switches.
   Its implementation is similar to that of modules, operators
   and clauses.
*/

% a set of dynamic facts 'forumSwitch(SwitchName, Value).'

:- dynamic forumSwitch/2.

switchName(forumSwitch(N,_),N).
switchValue(forumSwitch(_,V),V).

setSwitch(N,V) :-
   retractall(forumSwitch(N,_)),
   !,
   assertz(forumSwitch(N,V)).

getSwitch(N,V) :- forumSwitch(N,V).

getSwitches(S) :-
    findall(forumSwitch(N,V), forumSwitch(N,V), S).

deleteSwitches :-
    retractall(forumSwitch(_,_)).

% access to a particular switch

setEdit(Edit) :- setSwitch(edit,Edit).
getEdit(Edit) :- getSwitch(edit,Edit).

showTime :- setSwitch(showtime,yes).
hideTime :- setSwitch(showtime,no).
getShowTime(YN) :- getSwitch(showtime,YN).

eagerProver  :- setSwitch(strategy,eager).
lazyProver   :- setSwitch(strategy,lazy).
getProver(S) :- getSwitch(strategy,S).

traceOn     :- setSwitch(trace,on).
traceOff    :- setSwitch(trace,off).
getTrace(T) :- getSwitch(trace,T).

% ---------------------------- MODULE USAGE ---------------------------

% a fact 'usedModules(L)' containing a stack (list) of used modules

:- dynamic usedModules/1.

initUsedModules :- assert(usedModules([])).

useModule(M) :-
    usedModules(U),
    delete(U,M,Ms),
    retractall(usedModules(_)),
    assert(usedModules([M|Ms])).

getUsedModules(Ms) :- usedModules(Ms).

usingModule(M) :-
    usedModules(Ms),
    memberchk(M,Ms).

unuseModule(M) :-
    usedModules(U),
    delete(U,M,Ms),
    retractall(usedModules(_)),
    assert(usedModules(Ms)).

unuseModules :- retractall(usedModules(_)).

getUsedClauses(C) :- usedModules(U), getUsedClauses(U,[],C).

getUsedClauses([],Ac,Ac).

getUsedClauses([U|Us],Ac,C) :-
   getClauses(U,UC),
   append(Ac,UC,AcUC),
   getUsedClauses(Us,AcUC,C).

% -------- ACCESS TO SOURCE FILES: READING, PARSING, EDITING... --------

/*
    A source file is a container of UMA Forum modules. We define
    two basic operations:

       - reconsultFile(F): reads, parses, performs semantics checking
         and process definitions of every module contained in the
         source file F. Note that this command *RECONSULTS* modules,
         so prior UMA Forum definitions are overridden by new ones.

       - editFile(F): first edits a source file F using a user defined
         editor, and then *RECONSULTS* the file
*/

% ------- Reading a file: parsing, semantic checking, processing -------

reconsultFile(F) :-
    seeing(OldFile),
    (openFile(F) ->
          (readFile(F),
           seen,
           see(OldFile))
        ;
          printf("*** ERROR: cannot open file '%w'\n", [F])).

readFile(F) :-
    readHeader(N) ->
        (processHeader(F,N),
         readRest(F,N))
      ;
        printf("*** SYNTAX ERROR: module header expected in file '%w'\n",[F]).

readHeader(N) :- readTerm(T,_,_), isHeader(T,N).

readRest(F,N) :-
    readTerm(T,V,S),
    processTerm(F,N,T,V,S,M),
    !,
    readRest(F,M).
readRest(_,_).

processTerm(_,N,T,V,S,N) :- var(T), !, processClause(N,T,V,S).
processTerm(_,_,end_of_file,_,_,_) :- !, fail.
processTerm(F,_,T,_,_,M) :- isHeader(T,M), !, processHeader(F,M).
processTerm(_,N,T,_,_,N) :- isOpDecl(T,At,As,P), !, processOpDecl(N,At,As,P).
processTerm(_,N,T,V,S,N) :- processClause(N,T,V,S).

processHeader(F,N) :-
    printf("reading module '%w' from file '%w'\n", [N,F]),
    defineModule(N,F).

processOpDecl(N,At,As,P) :- checkOpDecl(At,P), !, defineOperator(N,At,As,P).
processOpDecl(_,_,_,_).

processClause(N,T,V,S) :- checkClause(T,V,S), !, defineClause(N,T,V).
processClause(_,_,_,_).

% -------------- editing a source file --------------

editFile(F) :-
   name(F,FileName),
   getEdit(Edit),
   name(Edit,EditName),
   makeCmd(EditName,FileName,EditCmd),
   printf("%w\n", [EditCmd]),
   system(EditCmd),
   reconsultFile(F).

% ---------------------- COMMON PREDICATES ----------------------

% --------------- misc. predicates ------------

isString(S) :- var(S), !, fail.
isString([Ch|S]) :- integer(Ch), isString(S).
isString([]).

openFile(F) :- fileExists(F), see(F).

makeCmd(Cmd,Args,CmdArgs) :-
   append(Cmd," ",Temp),
   append(Temp,Args,CmdArgsStr),
   name(CmdArgs,CmdArgsStr).

replaceFirst(X,[X|Xs],Y,[Y|Xs]) :- !.
replaceFirst(X,[H|Xs],Y,[H|Ys]) :- replaceFirst(X,Xs,Y,Ys).
replaceFirst(_,[],_,[]).

% --------------- printf: formatted ouput ---------------

% All the output generated by the program uses this printf
% procedure. So, it is trivial to replace this implementation to
% send output to, say, a TCL process...

% control chars '\.'

printf([92,110|F],Ts) :- !, nl,      printf(F,Ts).  % 92,110 = \n
printf([92,116|F],Ts) :- !, put(9),  printf(F,Ts).  % 92,116 = \t
printf([92, 92|F],Ts) :- !, put(92), printf(F,Ts).  % 92, 92 = \\
printf([92, 37|F],Ts) :- !, put(37), printf(F,Ts).  % 92, 37 = \%
printf([92, 34|F],Ts) :- !, put(34), printf(F,Ts).  % 92, 34 = \"

% format chars '%.'

printf([37,119|F],[T|Ts]) :- !, write(T),   printf(F,Ts).  % 37,119 = %w
printf([37,113|F],[T|Ts]) :- !, writeq(T),  printf(F,Ts).  % 37,113 = %q
printf([37,100|F],[T|Ts]) :- !, display(T), printf(F,Ts).  % 37,100 = %d
printf([37,115|F],[T|Ts]) :- !, puts(T),    printf(F,Ts).  % 37,115 = %s
printf([37,118|F],[V,T|Ts]) :-
               !, nameTermVars(T,V,Tn), printf(F,[Tn|Ts]). % 37,118 = %v

% plain chars '.'

printf([Ch|F],Ts) :- !, put(Ch), printf(F,Ts).

printf([],_).

printf(F) :- printf(F,[]).

puts([]).
puts([Ch|S]) :- put(Ch), puts(S).

nameTermVars(T,V,Tn) :-
   copy_term((T,V),(Tn,Vn)),
   nameVars(Vn),
   anonymousVar([Tn]).

% nameVars(L) : binds each var to its name
% L is a list of the form [Name=Var|...],

nameVars([N=N|Vs]) :- nameVars(Vs).
nameVars([]).

% anonymousVar: binds free vars to '_'

anonymousVar(['_'|Ts]) :- !, anonymousVar(Ts).
anonymousVar([T|Ts])   :- atomic(T), !, anonymousVar(Ts).
anonymousVar([T|Ts])   :- T =.. [_|Fs],
                          anonymousVar(Fs),
                          anonymousVar(Ts).
anonymousVar([]).

% --------------- main ---------------

forum :- banner,
         init,
         run,
         done,
         bye.

banner :-
   printf("UMA Forum 0.1\n"),
   printf("author: Pablo Lopez\n"),
   printf("e-mail: lopez@lcc.uma.es\n"),
   printf("www: http://www.lcc.uma.es/personal/lopez\n"),
   printf("Dept. of Computer Science\n"),
   printf("University of Malaga, SPAIN\n\n"),
   printf("'help' command available\n\n").

bye :- printf("Bye!...\n\n").

init :- init_defaults,
        (openFile('forum.ini') ->
              (read_ini,
               seen)
           ;
              (printf("file 'forum.ini' not founded... "),
               printf("initialized with defaults\n"))).

init_defaults :- initUsedModules,
                 setEdit(edit),
                 showTime,
                 lazyProver,
                 traceOff.

read_ini :- repeat,
               readTerm(T,V,_),
               eval(T,V),
            isHalt(T),
            !.

done :- deleteModules,
        deleteSwitches.

% --------------- interpreter loop ---------------

run:- repeat,
         prompt,
         readTerm(G,V,_),
         eval(G,V),
      isHalt(G),
      !.

prompt :- printf("Forum> ").

isHalt(T) :- var(T), !, fail.
isHalt(end_of_file).
isHalt(halt).


% --------------- eval switch ---------------

eval(T,V) :- var(T), !, processGoal(T,V), !.

% user commands

eval(help,_)         :- !, helpCmd.
eval(reset,_)        :- !, resetCmd.
eval(dir,_)          :- !, dirCmd.
eval(system(C),_)    :- !, systemCmd(C).
eval(setedit(N),_)   :- !, setEditCmd(N).
eval(modules,_)      :- !, modulesCmd.
eval([F|Fs],_)       :- !, reconsultCmd([F|Fs]).
eval(use(N),_)       :- !, useCmd(N).
eval(unuse(N),_)     :- !, unuseCmd(N).
eval(uses,_)         :- !, usesCmd.
eval(delete(N),_)    :- !, deleteCmd(N).
eval(list(N),_)      :- !, listCmd(N).
eval(listing,_)      :- !, listingCmd.
eval(edit,_)         :- !, editCmd.
eval(edit(N),_)      :- !, editCmd(N).
eval(showtime,_)     :- !, showTime.
eval(hidetime,_)     :- !, hideTime.
eval(eager,_)        :- !, eagerProver.
eval(lazy,_)         :- !, lazyProver.
eval(trace,_)        :- !, traceOn.
eval(notrace,_)      :- !, traceOff.
eval(switches,_)     :- !, switchesCmd.
eval(halt,_)         :- !.
eval(end_of_file,_)  :- !.

% solve a goal

eval(G,V) :- processGoal(G,V), !.

% default

eval(_,_).

% --------------- user commands --------------

helpCmd :-
   printf("This is UMA Forum 0.1\n"),
   interpHelp,
   modulesHelp1,
   modulesHelp2,
   otherHelp.

interpHelp :-
   printf("\nINTERPRETER COMMANDS\n"),
   printf("\t halt               -  halt interpreter\n"),
   printf("\t reset              -  reset interpreter\n"),
   printf("\t eager              -  set eager proof search\n"),
   printf("\t lazy               -  set lazy proof search\n"),
   printf("\t trace              -  trace on\n"),
   printf("\t notrace            -  trace off\n").

modulesHelp1 :-
   printf("\nMODULE COMMANDS\n"),
   printf("\t [file,...]         -  reconsult files [file,...]\n"),
   printf("\t modules            -  list loaded modules\n"),
   printf("\t use(name)          -  use module 'name'\n"),
   printf("\t unuse(name)        -  unuse module 'name'\n"),
   printf("\t uses               -  list used modules\n").

modulesHelp2:-
   printf("\t delete(name)       -  delete module 'name'\n"),
   printf("\t list(name)         -  list module 'name'\n"),
   printf("\t listing            -  list all modules\n"),
   printf("\t edit(name)         -  edit module 'name'\n"),
   printf("\t edit               -  edit most recently reconsulted file\n").

otherHelp :-
   otherHelp1,
   otherHelp2.

otherHelp1 :-
   printf("\nOTHER COMMANDS\n"),
   printf("\t dir                -  list current directory\n"),
   printf("\t system(cmd)        -  invokes system command 'cmd'\n"),
   printf("\t setedit(name)      -  set 'name' as default editor command\n"),
   printf("\t showtime           -  show goal solution time\n"),
   printf("\t hidetime           -  hide goal solution time\n").

otherHelp2 :-
   printf("\t switches           -  list current switches\n"),
   printf("\t help               -  show this help\n").

resetCmd :- done, init.

dirCmd :- showDir, !.

systemCmd(C) :-
   atom(C) ->
        system(C)
     ;
        printf("*** SYNTAX ERROR: command must be an ATOM\n").

setEditCmd(N) :-
   atom(N) ->
        setEdit(N)
     ;
        printf("*** SYNTAX ERROR: editor name must be an ATOM\n").

modulesCmd:-
    getModules(Ms),
    printModuleList(Ms).

printModuleList([M|Ms]) :-
    moduleName(M,N),
    moduleFile(M,F),
    printf("module '%w' from file '%w'\n", [N,F]),
    printModuleList(Ms).

printModuleList([]).

reconsultCmd([F|Fs]) :-
    (atom(F) ->
          reconsultFile(F)
       ;
          printf("*** SYNTAX ERROR: file name must be an ATOM\n")),
    reconsultCmd(Fs).

reconsultCmd([]).

useCmd(N) :-
    (isLCID(N) ->
          (getModule(N,_) ->
                useModule(N)
              ;
                printf("sorry, module '%w' not loaded\n", [N]))
        ;
          printf("*** SYNTAX ERROR: module name must be a LCID\n")).

unuseCmd(N) :-
     (isLCID(N) ->
         (usingModule(N) ->
                unuseModule(N)
             ;
                printf("sorry, module '%w' not used\n", [N])
         )
       ;
         printf("*** SYNTAX ERROR: module name must be a LCID\n")).

usesCmd :-
   getUsedModules(Ms),
   printUsedModules(Ms).

printUsedModules([U|Us]) :-
   printf("%w\n", [U]),
   printUsedModules(Us).

printUsedModules([]).

deleteCmd(N) :-
   isLCID(N) ->
       (getModule(N,_) ->
              deleteModule(N)
           ;
              printf("sorry, module '%w' not loaded\n", [N]))
     ;
        printf("*** SYNTAX ERROR: module name must be a LCID\n").

listCmd(N) :-
   isLCID(N) ->
      (getModule(N,_) ->
             ( printf("module %w.\n", [N]),
               getOperators(N,Is),
               printOperators(Is),
               getClauses(N,Cs),
               printClauses(Cs))
          ;
               printf("sorry, module '%w' not loaded\n", [N]))
     ;
       printf("*** SYNTAX ERROR: module name must be a LCID\n").

printOperators([I|Is]) :-
   operatorAtom(I,F),
   operatorAssoc(I,A),
   operatorPriority(I,P),
   printf("%w %w : %w\n", [A,F,P]),
   printOperators(Is).

printOperators([]).

printClauses([C|Cs]) :-
   clauseKind(C,(linear)),
   !,
   clauseFormula(C,F),
   clauseVariables(C,V),
   printf("linear %v%w.\n",[V,F]),
   printClauses(Cs).

printClauses([C|Cs]) :-
   clauseFormula(C,F),
   clauseVariables(C,V),
   printf("%v%w.\n",[V,F]),
   printClauses(Cs).

printClauses([]).

listingCmd :-
   getModules(Ms),
   printModules(Ms).

printModules([M|Ms]) :-
   moduleName(M,N),
   listCmd(N),
   printf("\n"),
   printModules(Ms).

printModules([]).

editCmd :-
   getModules([M|Ms]) ->
        (last([M|Ms],L),
         moduleFile(L,F),
         editFile(F))
     ;
         printf("sorry, no modules loaded\n").

editCmd(N) :-
   isLCID(N) ->
       (getModule(N,M) ->
              (moduleFile(M,F),
               editFile(F))
          ;
               printf("sorry, module '%w' not loaded\n", [N]))
     ;
        printf("*** SYNTAX ERROR: module name must be a LCID\n").

switchesCmd :-
   getSwitches(S),
   printSwitches(S).

printSwitches([S|Ss]) :-
   switchName(S,N),
   switchValue(S,V),
   printf("%w \t\t %w\n", [N,V]),
   printSwitches(Ss).

printSwitches([]).

% --------------------------- SOLVE A GOAL ----------------------------

% we count the number of solutions so far in numberSols(Number)

:- dynamic numberSols/1.

resetNumberSols :-
    retractall(numberSols(_)),
    assert(numberSols(0)).

getNumberSols(N) :- numberSols(N).

incNumberSols :-
    numberSols(N),
    retractall(numberSols(_)),
    N1 is N+1,
    assert(numberSols(N1)).

processGoal(G,V) :-
     (checkGoal(G,V) ->
           (collectProgram(P),
            emptyAtoms(A),
            resetNumberSols,
            !,
            solveGoal(P,A,G,V))
        ;
            true).

solveGoal(P,A,G,V) :-
    statistics(runtime,_),
    (getProver(lazy) ->
          lazyRight(P,[],A,[],[G],[],[])
       ;
          eagerRight(P,A,[G])),
    statistics(runtime,[_,T]),
    printVars(V),
    printTime(T),
    incNumberSols,
    enoughSolutions,
    !,
    getNumberSols(N),
    printf("%w solutions.\n", [N]).

solveGoal(_,_,_,_) :-
    getNumberSols(N),
    (N = 0 ->
           printf("no solution.\n")
       ;
          (printf("%w solutions.\n", [N]),
           printf("no more solutions.\n"))).

printVars([N=V|Vs]) :-
    printf("%w = %q\n", [N,V]),
    printVars(Vs).
printVars([]).

printTime(T) :-
   getShowTime(yes) ->
       printf("%w ms\n", [T])
     ;
       true.

enoughSolutions :-
    printf("->"),
    get0(C),
    C \= 10.

% ---------------- EAGER SEQUENT CALCULUS ---------------

% -------------- logic program implementation ---------------

/*
    The logic program (i.e. left context Psi; Delta) is dealt
    with as an Abstract Data Type. It is represented as a term

                       'program(NC,NL,Fs)'

    where Fs is a list of terms

                  'classical(F)' and 'linear(F)',

    denoting a Forum formula F either classical or linear; and
    NC and NL are the numbers of classical and linear formulas
    contained in Fs respectively.

    The predicates below define the logic program ADT operations.
*/

% emptyProgram(P)
% true if program P is empty

emptyProgram(program(0,0,[])).

% collectProgram(P)
% collects the clauses of the modules being used, yielding the
% logic program P

collectProgram(program(NC,NL,F)) :-
   getUsedClauses(C),
   labelClauses(C,F,0,NC,0,NL).

% labelClauses(C,LC,AC,NC,AL,NL)
% labels clauses as either 'classical' or 'linear' and counts them

labelClauses([],[],NC,NC,NL,NL).

labelClauses([C|Cs],[linear(F)|Fs],AC,NC,AL,NL) :-
   clauseKind(C,(linear)),
   !,
   clauseFormula(C,F),
   AL1 is AL+1,
   labelClauses(Cs,Fs,AC,NC,AL1,NL).

labelClauses([C|Cs],[classical(F)|Fs],AC,NC,AL,NL) :-
   clauseFormula(C,F),
   AC1 is AC+1,
   labelClauses(Cs,Fs,AC1,NC,AL,NL).

% isClassical(P)
% true if program P contains classical clauses only (i.e. Delta is empty)

isClassical(program(_,0,_)).

% addLinear(C,P,CP)
% adds the linear clause C to program P yielding program CP

addLinear(C,program(NC,NL,P),program(NC,NL1,[linear(C)|P])) :-
   NL1 is NL+1.

% addClassical(C,P,CP)
% adds the classical clause C to program P yielding program CP

addClassical(C,program(NC,NL,P),program(NC1,NL,[classical(C)|P])) :-
   NC1 is NC+1.

% focus(PC,C,P)
% selects a clause C from program PC, yielding program P

focus(program(NC,NL,Fs),F,program(NC,NL1,Fs1)) :-
   selectFocus(Fs,Fc,Fs1),
   Fc =.. [K,F],
   (K = (linear) ->
         NL1 is NL-1
      ;
         NL1 is NL).

selectFocus([linear(F)|Fs],linear(F1),Fs) :-
   copy_term(F,F1).

selectFocus([classical(F)|Fs],classical(F1),[classical(F)|Fs]) :-
   copy_term(F,F1).

selectFocus([F|Fs],Fc,[F|Fs1]) :- selectFocus(Fs,Fc,Fs1).

% splitProgram(P,P1,P2)
% splits program P into P1 and P2

splitProgram(program(NC,NL,P),program(NC,NL1,P1),program(NC,NL2,P2)) :-
   splitProgram(P,NL,P1,P2,0,0,NL1,NL2).

splitProgram(P,0,P,P,NL1,NL2,NL1,NL2) :- !.

splitProgram([classical(F)|Fs],NL,
             [classical(F)|Bs],[classical(F)|Cs],AL1,AL2,NL1,NL2) :-
   !,
   splitProgram(Fs,NL,Bs,Cs,AL1,AL2,NL1,NL2).

splitProgram([linear(F)|Fs],NL,[linear(F)|Bs],Cs,AL1,AL2,NL1,NL2) :-
   AL11 is AL1+1,
   NL_1 is NL-1,
   splitProgram(Fs,NL_1,Bs,Cs,AL11,AL2,NL1,NL2).

splitProgram([linear(F)|Fs],NL,Bs,[linear(F)|Cs],AL1,AL2,NL1,NL2) :-
   AL22 is AL2+1,
   NL_1 is NL-1,
   splitProgram(Fs,NL_1,Bs,Cs,AL1,AL22,NL1,NL2).

% takeClassical(P,CP)
% select all the classical clauses from program P yielding program CP

takeClassical(program(NC,NL,Fs),program(NC,0,Cs)) :-
   takeClassical(NC,NL,Fs,Cs).

takeClassical(_,0,Cs,Cs) :- !.

takeClassical(0,_,[],[]) :- !.

takeClassical(NC,NL,[classical(F)|Fs],[classical(F)|Cs]) :-
   NC1 is NC-1,
   takeClassical(NC1,NL,Fs,Cs).

takeClassical(NC,NL,[linear(_)|Fs],Cs) :-
   NL1 is NL-1,
   takeClassical(NC,NL1,Fs,Cs).

% -------------- atomic goal implementation ---------------

/*
    Atomic goals (right subcontext A) are dealt with as an
    Abstract Data Type. They are represented as a term

                           'atoms(As)'

    where As is a list of atoms.

    The predicates below define the atomic goals ADT operations.
*/

% emptyAtoms(A)
% true if A is empty

emptyAtoms(atoms([])).

% addAtom(A,As,AAs)
% adds atom A to atoms As yielding atoms AAs

addAtom(A,atoms(As),atoms([A|As])).

% singletonAtom(As,A)
% As contains the atom A only

singletonAtom(atoms([A]),A).

% splitAtoms(A,A1,A2)
% splits atoms A into A1 and A2

splitAtoms(atoms(A),atoms(B),atoms(C)) :-
   splitAtomsL(A,B,C).

splitAtomsL([A|As],[A|Bs],Cs) :- splitAtomsL(As,Bs,Cs).
splitAtomsL([A|As],Bs,[A|Cs]) :- splitAtomsL(As,Bs,Cs).
splitAtomsL([],[],[]).

% --------------- built-in predicates ---------------

isBuiltIn(T) :- var(T), !, fail.

isBuiltIn(printf(_,_)).
isBuiltIn(printf(_)).
isBuiltIn(read(_)).

isBuiltIn((_  = _)).
isBuiltIn((_ \= _)).
isBuiltIn((_ == _)).
isBuiltIn((_ \== _)).

isBuiltIn((_ is _)).
isBuiltIn((_  < _)).
isBuiltIn((_ =< _)).
isBuiltIn((_  > _)).
isBuiltIn((_ >= _)).

isBuiltIn(var(_)).
isBuiltIn(nonvar(_)).
isBuiltIn(atom(_)).
isBuiltIn(integer(_)).
isBuiltIn(float(_)).
isBuiltIn(number(_)).
isBuiltIn(atomic(_)).
isBuiltIn(ground(_)).
isBuiltIn(functor(_,_,_)).
isBuiltIn(arg(_,_,_)).
isBuiltIn((_ =.. _)).
isBuiltIn(name(_,_)).
isBuiltIn(copy_term(_,_)).

doBuiltIn(G) :- G.

isAtomic(V) :- var(V), !.
isAtomic(G) :- G =.. [F|_], \+ isConnective(F).

% --------------- THE THEOREM PROVER (EAGER VERSION) ---------------

% --------------- right rules: goal decomposition ---------------

% eagerRight(P,A,G)
%
%     P - program (classical+linear)          (Psi+Delta)
%     A - atomic goals                        (A)
%     G - goals                               (Gamma)

eagerRight(P,A,G) :-
   getTrace(on),
   printf("RIGHT\n P: %w\n A: %w\n G: %w\n\n",[P,A,G]),
   get0(_),
   fail.

eagerRight(P,A,[G|Gs]) :-                     % atomic
   isAtomic(G),
   !,
   addAtom(G,A,GA),
   eagerRight(P,GA,Gs).

eagerRight(_,_,[top|_]) :- !.                 % top R

eagerRight(P,A,[bot|Gs]) :-                   % bot R
   !,
   eagerRight(P,A,Gs).

eagerRight(P,A,[(B & C)|Gs]) :-               % & R
   !,
   eagerRight(P,A,[B|Gs]),
   eagerRight(P,A,[C|Gs]).

eagerRight(P,A,[(B # C)|Gs]) :-               % # R
   !,
   eagerRight(P,A,[B,C|Gs]).

eagerRight(P,A,[(C -* B)|Gs]) :-              % -* R
   !,
   addLinear(C,P,CP),
   eagerRight(CP,A,[B|Gs]).

eagerRight(P,A,[(B *- C)|Gs]) :-              % *- R
   !,
   addLinear(C,P,CP),
   eagerRight(CP,A,[B|Gs]).

eagerRight(P,A,[(C => B)|Gs]) :-              % => R
   !,
   addClassical(C,P,CP),
   eagerRight(CP,A,[B|Gs]).

eagerRight(P,A,[(B <= C)|Gs]) :-              % <= R
   !,
   addClassical(C,P,CP),
   eagerRight(CP,A,[B|Gs]).

eagerRight(_,_,[(forall _ \ _)|_]) :-         % forall R
   printf("*** ERROR: forall right\n"),
   !,
   fail.

eagerRight(P,A,[]) :-
   !,
   eagerDecide(P,A).

% --------------- decide rules ---------------

%  eagerDecide(P,A)
%
%     P - program (classical+linear)          (Psi+Delta)
%     A - atomic goals                        (A)


eagerDecide(P,A) :-                           % built-in
   isClassical(P),
   singletonAtom(A,G),
   isBuiltIn(G),
   !,
   doBuiltIn(G).

eagerDecide(PF,A) :-                          % decide, decide !
   focus(PF,F,P),
   eagerLeft(P,F,A).

% --------------- left rules ---------------

% eagerLeft(P,F,A)
%
%     P - program (classical+linear)          (Psi+Delta)
%     F - proof focus                         (F)
%     A - atomic goals                        (A)

eagerLeft(P,F,A) :-
   getTrace(on),
   printf("LEFT\n P: %w\n F: %w\n A: %w\n\n",[P,F,A]),
   get0(_),
   fail.

eagerLeft(P,A,As) :-                          % initial
   isClassical(P),
   singletonAtom(As,A),
   !.

eagerLeft(P,bot,A) :-                         % bottom L
   isClassical(P),
   emptyAtoms(A),
   !.

eagerLeft(P,(B & C),A) :-                     % & L
   !,
   (   eagerLeft(P,B,A)
     ;
      eagerLeft(P,C,A)
   ).

eagerLeft(P,(B # C),A) :-                     % # L
   !,
   splitProgram(P,P1,P2),
   splitAtoms(A,A1,A2),
   eagerLeft(P1,B,A1),
   eagerLeft(P2,C,A2).

eagerLeft(P,(B -* C),A) :-                    % -* L
   !,
   splitProgram(P,P1,P2),
   splitAtoms(A,A1,A2),
   eagerLeft(P1,C,A1),
   eagerRight(P2,A2,[B]).

eagerLeft(P,(C *- B),A) :-                    % *- L
   !,
   splitProgram(P,P1,P2),
   splitAtoms(A,A1,A2),
   eagerLeft(P1,C,A1),
   eagerRight(P2,A2,[B]).

eagerLeft(P,(B => C),A) :-                    % => L
   !,
   takeClassical(P,CP),
   eagerLeft(P,C,A),
   emptyAtoms(E),
   eagerRight(CP,E,[B]).

eagerLeft(P,(C <= B),A) :-                    % <= L
   !,
   takeClassical(P,CP),
   eagerLeft(P,C,A),
   emptyAtoms(E),
   eagerRight(CP,E,[B]).

eagerLeft(_,(forall _ \ _),_,_) :-            % forall L
   printf("*** ERROR: forall left\n"),
   !,
   fail.

% ---------------- LAZY SEQUENT CALCULUS ---------------

% -------------- logic program implementation ---------------

/*
    The logic program (i.e. left context Psi; Delta; Pi) is dealt
    with as an Abstract Data Type. It is represented by two terms:
    a non-returnable logic program (Psi+Delta) represented as in the
    eager version and a returnable logic program (Psi+Pi) represented
    as a stack (list) of logic programs.

    The returned program Pi' is represented as a stack (list) of
    logic programs.

    The predicates below define the logic program ADT operations.
*/

% consumeProgramS(P,P1)
% consumes a portion of the linear clauses of the stack of programs P,
% yielding the stack of programs P1

consumeProgramS([P|Ps],[P1|P1s]) :-
   consumeProgram(P,P1),
   consumeProgramS(Ps,P1s).

consumeProgramS([],[]).

consumeProgram(program(NC,NL,P),program(NC,NL1,P1)) :-
   consumeLinear(P,NL,P1,0,NL1).

consumeLinear(P,0,P,NL1,NL1) :- !.

consumeLinear([classical(C)|P],NL,[classical(C)|P1],AL1,NL1) :-
   !,
   consumeLinear(P,NL,P1,AL1,NL1).

consumeLinear([linear(C)|P],NL,[linear(C)|P1],AL1,NL1) :-
   AL11 is AL1+1,
   NL_1 is NL-1,
   consumeLinear(P,NL_1,P1,AL11,NL1).

consumeLinear([linear(_)|P],NL,P1,AL1,NL1) :-
   NL_1 is NL-1,
   consumeLinear(P,NL_1,P1,AL1,NL1).

% focusS(P,F,P1)
% selects a focus F from the stack of programs P, yielding the
% stack of programs P1

focusS([P|Ps],F,[P1|Ps]) :-
   focus(P,F,P1).

focusS([P|Ps],F,[P|Ps1]) :-
   focusS(Ps,F,Ps1).

% takeClassicalS(P,P1)
% select all the classical clauses from the stack of programs P,
% yielding the stack of programs P1

takeClassicalS([P|Ps],[P1|P1s]) :-
   takeClassical(P,P1),
   takeClassicalS(Ps,P1s).

takeClassicalS([],[]).

pushProgram(D,P,[D|P]).
popProgram([D|P],D,P).

% -------------- atomic goal implementation ---------------

/*
    Atomic goals (right subcontext A) are dealt with as an
    Abstract Data Type. They are represented by two terms:
    non-returnable atomic goals, represented as in the eager
    version and returnable atomic goals, represented as a
    stack of atomic goals.

    The returned atomic goals B' are represented as a stack
    of atomic goals.

    The predicates below define the atomic goals ADT operations.
*/

% consumeAtomS(A,A1)
% consumes a portion of the atoms of the stack of atoms A,
% yielding the stack of atoms A1

consumeAtomS([A|As],[A1|A1s]) :-
   consumeAtoms(A,A1),
   consumeAtomS(As,A1s).

consumeAtomS([],[]).

consumeAtoms(atoms(A),atoms(B)) :-
   consumeAtomsL(A,B).

consumeAtomsL([A|As],[A|Bs]) :- consumeAtomsL(As,Bs).
consumeAtomsL([_|As],Bs)     :- consumeAtomsL(As,Bs).
consumeAtomsL([],[]).

% selectAtomS(A,F,A1)
% selects an atom F from the stack of atoms A, yielding the
% stack of atoms A1

selectAtomS([A|As],F,[A1|As]) :-
   selectAtom(A,F,A1).

selectAtomS([A|As],F,[A|A1s]) :-
   selectAtomS(As,F,A1s).

selectAtom(atoms(A),F,atoms(A1)) :-
   select(F,A,A1).

pushAtoms(A,B,[A|B]).
popAtoms([A|B],A,B).

% --------------- THE THEOREM PROVER (LAZY VERSION) ---------------

% --------------- right rules ---------------

% lazyRight(D,P,A,B,G,RP,RB)
%
%    D  -  non-returnable program                   (Psi+Delta)
%    P  -  returnable program                       (Psi+Pi)
%    A  -  non-returnable atomic goals              (Ascript)
%    B  -  returnable atomic goals                  (Bscript)
%    G  -  goals                                    (Gamma)
%    RP -  returned program                         (Pi')
%    RB -  returned atomic goals                    (Bscript')

lazyRight(D,P,A,B,G,RP,RB) :-
getTrace(on),
printf("RIGHT\n D: %w\n P: %w\n A: %w\n B: %w\n G: %w\n RP: %w\n RB: %w\n",
       [D,P,A,B,G,RP,RB]),
get0(_),
fail.

lazyRight(D,P,A,B,[G|Gs],RP,RB) :-                  % atomic
   isAtomic(G),
   !,
   addAtom(G,A,GA),
   lazyRight(D,P,GA,B,Gs,RP,RB).

lazyRight(_,P,_,B,[top|_],RP,RB) :-                 % top R
   !,
   consumeProgramS(P,RP),
   consumeAtomS(B,RB).

lazyRight(D,P,A,B,[bot|Gs],RP,RB) :-                % bot R
   !,
   lazyRight(D,P,A,B,Gs,RP,RB).

lazyRight(D,P,A,B,[G1 # G2|Gs],RP,RB) :-            % # R
   !,
   lazyRight(D,P,A,B,[G1,G2|Gs],RP,RB).

lazyRight(D,P,A,B,[G1 & G2|Gs],RP,RB) :-            % & R
   !,
   lazyRight(D,P,A,B,[G1|Gs],RP,RB),
   lazyRight(D,P,A,B,[G2|Gs],RP,RB).

lazyRight(D,P,A,B,[G1 -* G2|Gs],RP,RB) :-           % -* R
   !,
   addLinear(G1,D,G1D),
   lazyRight(G1D,P,A,B,[G2|Gs],RP,RB).

lazyRight(D,P,A,B,[G2 *- G1|Gs],RP,RB) :-           % *- R
   !,
   addLinear(G1,D,G1D),
   lazyRight(G1D,P,A,B,[G2|Gs],RP,RB).

lazyRight(D,P,A,B,[G1 => G2|Gs],RP,RB) :-           % => R
   !,
   addClassical(G1,D,G1D),
   lazyRight(G1D,P,A,B,[G2|Gs],RP,RB).

lazyRight(D,P,A,B,[G2 <= G1|Gs],RP,RB) :-           % <= R
   !,
   addClassical(G1,D,G1D),
   lazyRight(G1D,P,A,B,[G2|Gs],RP,RB).

lazyRight(_,_,_,_,[(forall _ \ _)|_],_,_) :-        % forall R
   printf("*** ERROR: forall right\n"),
   !,
   fail.

lazyRight(D,P,A,B,[],RP,RB) :-
   !,
   lazyDecide(D,P,A,B,RP,RB).

% --------------- decide rules ---------------

%  lazyDecide(D,P,A,B,RP,RB)
%
%    D  -  non-returnable program                   (Psi+Delta)
%    P  -  returnable program                       (Psi+Pi)
%    A  -  non-returnable atomic goals              (Ascript)
%    B  -  returnable atomic goals                  (Bscript)
%    RP -  returned program                         (Pi')
%    RB -  returned atomic goals                    (Bscript')


lazyDecide(D,P,A,B,P,B) :-                          % built-in
   isClassical(D),
   singletonAtom(A,G),
   isBuiltIn(G),
   !,
   doBuiltIn(G).

lazyDecide(DF,P,A,B,RP,RB) :-                       % decide, decide !
   focus(DF,F,D),
   lazyLeft(D,P,F,A,B,RP,RB).

lazyDecide(D,PF,A,B,RP,RB) :-                       % decide, decide !
   focusS(PF,F,P),
   lazyLeft(D,P,F,A,B,RP,RB).

% --------------- left rules ---------------

% lazyLeft(D,P,F,A,B,RP,RB)
%
%    D  -  non-returnable program                   (Psi+Delta)
%    P  -  returnable program                       (Psi+Pi)
%    F  -  proof focus
%    A  -  non-returnable atomic goals              (Ascript)
%    B  -  returnable atomic goals                  (Bscript)
%    RP -  returned program                         (Pi')
%    RB -  returned atomic goals                    (Bscript')

lazyLeft(D,P,F,A,B,RP,RB) :-
getTrace(on),
printf("LEFT\n D: %w\n P: %w\n F: %w\n A: %w\n B: %w\n RP: %w\n RB: %w\n",
       [D,P,F,A,B,RP,RB]),
get0(_),
fail.

lazyLeft(D,P,F,A,B,P,B) :-                          % initial A
   isAtomic(F),
   isClassical(D),
   singletonAtom(A,F),
   !.

lazyLeft(D,P,F,A,B,P,RB) :-                         % initial B
   isAtomic(F),
   isClassical(D),
   emptyAtoms(A),
   !,
   selectAtomS(B,F,RB).

lazyLeft(D,P,bot,A,B,P,B) :-                        % bot L
   isClassical(D),
   emptyAtoms(A),
   !.

lazyLeft(D,P,(F1 & F2),A,B,RP,RB) :-                % & L
   !,
   (lazyLeft(D,P,F1,A,B,RP,RB)
     ;
    lazyLeft(D,P,F2,A,B,RP,RB)).

lazyLeft(D,P,(F1 # F2),A,B,RP,RB) :-                % # L
   !,
   pushProgram(D,P,DP),
   pushAtoms(A,B,AB),
   emptyProgram(EP),
   emptyAtoms(EA),
   lazyLeft(EP,DP,F1,EA,AB,RDP,RAB),
   popProgram(RDP,D2,P2),
   popAtoms(RAB,A2,B2),
   lazyLeft(D2,P2,F2,A2,B2,RP,RB).

lazyLeft(D,P,(F1 -* F2),A,B,RP,RB) :-               % -* L
   !,
   pushProgram(D,P,DP),
   pushAtoms(A,B,AB),
   emptyProgram(EP),
   emptyAtoms(EA),
   lazyLeft(EP,DP,F2,EA,AB,RDP,RAB),
   popProgram(RDP,D1,P1),
   popAtoms(RAB,A1,B1),
   lazyRight(D1,P1,A1,B1,[F1],RP,RB).

lazyLeft(D,P,(F2 *- F1),A,B,RP,RB) :-               % *- L
   !,
   pushProgram(D,P,DP),
   pushAtoms(A,B,AB),
   emptyProgram(EP),
   emptyAtoms(EA),
   lazyLeft(EP,DP,F2,EA,AB,RDP,RAB),
   popProgram(RDP,D1,P1),
   popAtoms(RAB,A1,B1),
   lazyRight(D1,P1,A1,B1,[F1],RP,RB).

lazyLeft(D,P,(F1 => F2),A,B,RP,RB) :-               % => L
   !,
   lazyLeft(D,P,F2,A,B,RP,RB),
   takeClassical(D,Psi1),
   takeClassicalS(P,Psi2),
   emptyAtoms(EA),
   lazyRight(Psi1,Psi2,EA,[],[F1],_,[]).

lazyLeft(D,P,(F2 <= F1),A,B,RP,RB) :-               % <= L
   !,
   lazyLeft(D,P,F2,A,B,RP,RB),
   takeClassical(D,Psi1),
   takeClassicalS(P,Psi2),
   emptyAtoms(EA),
   lazyRight(Psi1,Psi2,EA,[],[F1],_,[]).

lazyLeft(_,_,(forall _ \ _),_,_,_,_) :-             % forall L
   printf("*** ERROR: forall left\n"),
   !,
   fail.
