//--------------------------------------------------------------------------
// Copyright (c) 2012, 2013 Vicente Benjumea, University of Malaga, Spain
// 
// This file is part of the <PTR> Library
// 
// This library is free software: you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation, either
// version 2.1 of the License, or (at your option) any later
// version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General
// Public License along with this library. If not, see
// <http://www.gnu.org/licenses/>.
//--------------------------------------------------------------------------
/*
 *  Version 1.2 (11-jun-2015)
 */
//--------------------------------------------------------------------------
/*
 * NOTA: La versión actual no es adecuada para trabajar con
 *       Herencia/Polimorfismo
 *
 * Adaptado para gcc-3.2 o superior sobre GNU/Linux o MinGW/Windows
 *
 * Cuando se desea definir un tipo Puntero con control en tiempo de
 * ejecución, se debe definir de la siguiente forma, y se utiliza como
 * cualquier otro tipo puntero:
 *
 *    #include <ptr.hpp>
 *    struct Nodo;
 *    typedef Ptr<Nodo> PNodo;
 *
 * ejemplo de compilación:
 *
 *    g++ -ansi -Wall -Werror -g -D_GLIBCXX_DEBUG -o prueba prueba.cpp
 *
 * Si se define la opción de compilación -D_GLIBCXX_DEBUG, entonces el
 * tipo Ptr<> realiza chequeos en tiempo de ejecución. Sin embargo, si
 * no se define dicho símbolo, entonces el tipo Ptr<> se comporta como
 * un tipo puntero normal.
 *
 * Se debe compilar el programa con la opción de compilación -g para
 * que se pueda mostrar el numero de línea del error
 *
 * Si se desea que si se compruebe la asignación de punteros no
 * inicializados o ya liberados, entonces se debe compilar el programa
 * con la opcion de compilación -DCOPYCHECK
 */
//--------------------------------------------------------------------------
#ifndef COPYCHECK
#define NOCOPYCHECK		1
#endif
//--------------------------------------------------------------------------
#ifndef __cplusplus
#error "Debe compilar para C++"
#endif
//------------------------------------
#ifndef umalcc_ptr_hpp__
#define umalcc_ptr_hpp__
//--------------------------------------------------------------------------
#ifdef CXX0X
#define NOEXCEPT	noexcept
#else
#define NOEXCEPT	throw()
#endif
#ifndef _GLIBCXX_DEBUG
//--------------------------------------------------------------------------
//-- NO-DEBUG-MODE ---------------------------------------------------------
//--------------------------------------------------------------------------
namespace umalcc {
	//----------------------------------------------------------------------
	template <typename TIPO_BASE>
	class Ptr {
		typedef TIPO_BASE			_Tipo_base;
		typedef _Tipo_base&			_Tipo_base_ref;
		typedef _Tipo_base*			_Tipo_ptr;
		//--------------------------
		_Tipo_ptr _ptr;
		//--------------------------
	public:
		//------------------------
		~Ptr() NOEXCEPT {}
		Ptr() {} // Sin inicializar, para que tenga el mismo comportamiento
		Ptr(_Tipo_ptr ptr) : _ptr(ptr) {}
		Ptr(const Ptr& orig) : _ptr(orig._ptr) {}
		//------------------------
		Ptr& operator= (_Tipo_ptr ptr) { _ptr = ptr; return *this; }
		Ptr& operator= (const Ptr& orig) { _ptr = orig._ptr; return *this; }
		//------------------------
		operator _Tipo_ptr () const { return _ptr; }
		_Tipo_ptr __get() const { return _ptr; }
		//------------------------
		_Tipo_base_ref operator*() const { return *_ptr; }
		_Tipo_ptr operator->() const { return _ptr; }
		//------------------------
	};
	//----------------------------------------------------------------------
}// namespace umalcc
//--------------------------------------------------------------------------
#else
//--------------------------------------------------------------------------
//-- DEBUG-MODE ------------------------------------------------------------
//--------------------------------------------------------------------------
#include <cstdlib>
#include <iostream>
#include <stdexcept>
#include <typeinfo>
#include <string>
#include <sstream>
#include <cstdio>
//--------------------------------------------------------------------------
#if defined __GNUC__ //&& ( defined __linux__ || defined __MINGW32__ )
//#include <execinfo.h>
#include <cxxabi.h>
#define RETADDR (__builtin_extract_return_addr(__builtin_return_address(0)))
#if defined __linux__
// man program_invocation_name // glibc
extern "C" char* program_invocation_name;
#define PROGRAM_NAME__ program_invocation_name
#elif defined __MINGW32__
extern "C" char** _argv;
#define PROGRAM_NAME__ _argv[0]
#else
extern "C" char* __progname_full;
#define PROGRAM_NAME__ __progname_full
#endif
#else
#define RETADDR (0)
#endif
//--------------------------------------------------------------------------
#ifndef UNUSED__
#ifdef __GNUC__
#define UNUSED__ __attribute__((unused))
#else
#define UNUSED__
#endif
#endif
//--------------------------------------------------------------------------
#ifndef NOINLINE__
#ifdef __GNUC__
#define NOINLINE__ __attribute__((noinline))
#else
#define NOINLINE__
#endif
#endif
//----------------------------------------------------------------------
#ifdef __GNUC__
extern "C" {
	FILE* popen(const char *cmd, const char *mode);
	int pclose(FILE *stream);
}
#endif
//--------------------------------------------------------------------------
namespace umalcc {
	//----------------------------------------------------------------------
	template <typename TIPO_BASE> class Ptr;
	//----------------------------------------------------------------------
	//-- Auxiliar ----------------------------------------------------------
	//----------------------------------------------------------------------
	namespace aux {
		//------------------------------------------------------------------
		enum CheckMode {
			CHECK_NO, CHECK_THROW, CHECK_ABORT,	// keep this order
			CHECK_MODE=
#if defined NDEBUG
			CHECK_NO
#elif defined PTR_THROW__
			// el modo de lanzamiento de excepciones tiene problemas
			// ya que un destructor no debe lanzar excepciones, por
			// lo que los errores producidos en la destruccion de variables
			// no serian detectados.
			CHECK_THROW
#else
			CHECK_ABORT
#endif
			, CHECK_MODE_ON = (CHECK_MODE > CHECK_NO)
			, CHECK_MODE_ABORT = (CHECK_MODE > CHECK_THROW)
		};
		//------------------------------------------------------------------
		struct PtrFailure : public std::logic_error { 
			explicit PtrFailure(const char* w) : std::logic_error(w) {}
		};
		//------------------------------------------------------------------
		inline std::string __demangle(const char* mangled_name)
		{
			std::string name(mangled_name);
#ifdef __GNUC__
			char* nm = abi::__cxa_demangle(mangled_name, NULL,
										   NULL, NULL);
			if (nm != NULL) {
				name = nm;
				std::free(nm);
			}
#endif
			return name;
		}
		//--------------------------------------------------------------
		inline void __print_filename_line(const void* caller, bool decr=true)
		{
#ifdef __GNUC__
			if (caller == 0) {
				std::cerr << "Error: no es posible calcular la direccion de retorno"
						  << " [" << PROGRAM_NAME__ << "]"
						  << std::endl;
			} else {
				const void* return_address = (const char*)caller-1;
				char cmd[1024];
				std::FILE *fp;
				std::snprintf(cmd, 1024, "addr2line -e \"%s\" %p",
							  PROGRAM_NAME__, return_address);
				/*----------------------------*/
				fp = popen(cmd, "r");
				if (fp) {
					char filename[256] = "??";
					int line = 0;
					if (std::fscanf(fp, " %255[^: ]:%d",
									filename, &line) > 0) {
						if ((filename[0] != '?') && (line != 0)) {
							std::cerr << filename << ":"
									  << (line + (decr ? 0 : 1))
									  << std::endl;
						} else {
							std::cerr << "Error: debe compilar con la opcion -g"
									  << std::endl;
						}
						pclose(fp);
					} else {
						pclose(fp);
						system(cmd);
					}
				} else {
					system(cmd);
				}
			}
#endif
		}
		//--------------------------------------------------------------
#undef PROGRAM_NAME__
		//--------------------------------------------------------------
		inline void __terminar(const std::string& msg,
							   const void* caller=RETADDR,
							   bool decr=true)
		{
			if (CHECK_MODE_ABORT) {
				std::cerr << "\nERROR: " << msg << "." << std::endl;
				//------------------------------------------------------
				__print_filename_line(caller, decr);
				//------------------------------------------------------
#if defined _WIN32 || defined _MSC_VER || defined __WIN32__
				std::system("pause");
#endif
				std::abort();
			} else {
				throw PtrFailure(msg.c_str());
			}
		}
		//-----------------------------------------------------------------
		// No es adecuado para Threads, ya que utiliza memoria estática
		inline const void* reg_caller(const void* c) {
			static const void* caller = 0;
			if (c != 0) {
				if (caller == c) {
					caller = 0;
				} else if (caller == 0) {
					caller = c;
				}
			}
			return caller;
		}
		//-----------------------------------------------------------------
		static void* const PTR_INVALIDO = reinterpret_cast<void*>(-1);
		//------------------------------------------------------------------
		//-- Cuenta_Ref ----------------------------------------------------
		//------------------------------------------------------------------
		// Este sistema utilizado no es adecuado para soportar la
		// herencia y polimorfismo.
		//------------------------------------------------------------------
		// struct Nodo;
		// typedef Ptr<Nodo> PNodo;
		// PNodo p = new Nodo;
		// PNodo q = p;
		//
		//		 Ptr
		//		+---+
		//	   p| o-------+
		//		+---+	  |
		//				  V
		//		+---+	+---+CntRef
		//	   q| o---->| F	|
		//		+---+	+---+
		//				| 2	|	 Nodo
		//				+---+	+----+
		//				| o---->|DATA|
		//				+---+	+----+
		// 
		//------------------------------------------------------------------
		template <typename TIPO_BASE>
		class Cuenta_Ref {
			friend class Ptr<TIPO_BASE>;
			typedef TIPO_BASE			_Tipo_base;
			typedef _Tipo_base*			_Tipo_ptr;
			//------------------------
			_Tipo_ptr		_puntero;
			unsigned short	_cntref;
			bool 			_liberado;
			//------------------------
			Cuenta_Ref(_Tipo_ptr ptr)
				: _puntero(ptr), _cntref(0), _liberado(false) {}
			void operator delete[](void*ptr); // Si se utiliza -> Error Compil.
		public:
			//------------------------
			NOINLINE__
			void operator delete(void*ptr) {
				// if (ptr == NULL) return;
				if (umalcc::aux::CHECK_MODE_ON) {
					if (ptr == NULL) {
						std::ostringstream msg;
						msg << "[delete] Liberar Ptr<"
							<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
							<<"> NULO";
						umalcc::aux::__terminar(msg.str());
					} else if (ptr == umalcc::aux::PTR_INVALIDO) {
						std::ostringstream msg;
						msg << "[delete] Liberar Ptr<"
							<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
							<<"> no inicializado";
						umalcc::aux::__terminar(msg.str());
					} else if (static_cast<Cuenta_Ref<TIPO_BASE>*>(ptr)->_cntref
							   == 0){
						::operator delete(ptr); // liberar el nodo cntref
					} else if (static_cast<Cuenta_Ref<TIPO_BASE>*>(ptr)
							   ->_liberado) {
						std::ostringstream msg;
						msg << "[delete] Liberar Ptr<"
							<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
							<<"> una zona de memoria ya liberada " << ptr;
						umalcc::aux::__terminar(msg.str());
					} else {
						static_cast<Cuenta_Ref<TIPO_BASE>*>(ptr)->_liberado
							= true;
						// OJO, en este caso puede que al destruir el
						// objeto apuntado se pierda la referencia a
						// alguna zona de memoria por lo que el
						// unwind_level se debe aumentar (+=2) para el
						// destruir de Ptr [ver Ptr<>::destruir]
						umalcc::aux::reg_caller(RETADDR);
						delete static_cast<Cuenta_Ref<TIPO_BASE>*>(ptr)
							->_puntero;
						umalcc::aux::reg_caller(RETADDR);
					}
				} else {
					::operator delete(ptr);
				}
			}
			//------------------------
		};
	}// namespace aux
	//----------------------------------------------------------------------
	//-- Ptr ---------------------------------------------------------------
	//----------------------------------------------------------------------
	template <typename TIPO_BASE>
	std::ostream& operator<<(std::ostream&, const Ptr<TIPO_BASE>&);
	//----------------------------------------------------------------------
	template <typename TIPO_BASE>
	class Ptr {
		typedef TIPO_BASE			_Tipo_base;
		typedef _Tipo_base&			_Tipo_base_ref;
		typedef _Tipo_base*			_Tipo_ptr;
		typedef umalcc::aux::Cuenta_Ref<TIPO_BASE>	_Tipo_cntref;
		typedef _Tipo_cntref* 					_Tipo_cntref_ptr;
		//--------------------------
		_Tipo_cntref_ptr _ptr_cntref;
		//--------------------------
	public:
		//------------------------
		NOINLINE__
#if defined PTR_THROW__ && ! defined NDEBUG
		~Ptr() { destruir("{}"); }
#else
		~Ptr() NOEXCEPT { try { destruir("{}"); } catch (...) {} }
#endif
		//------------------------
		Ptr()
			: _ptr_cntref(reinterpret_cast<_Tipo_cntref_ptr>(umalcc::aux::PTR_INVALIDO))
		{}
		NOINLINE__
		Ptr(_Tipo_ptr ptr)
		//: _ptr_cntref(reinterpret_cast<_Tipo_cntref_ptr>(ptr))
		{ copiar_nuevo(ptr,"="); }
		NOINLINE__
		Ptr(const Ptr& orig)
		//: _ptr_cntref(orig._ptr_cntref)
		{ copiar(orig._ptr_cntref,"="); }
		//------------------------
		NOINLINE__
		Ptr& operator= (_Tipo_ptr ptr) {
			destruir("=");
			copiar_nuevo(ptr,"=");
			return *this;
		}
		NOINLINE__
		Ptr& operator= (const Ptr& orig) {
			if (this != &orig) {
				destruir("=");
				copiar(orig._ptr_cntref,"=");
			}
			return *this;
		}
		//------------------------
		NOINLINE__
		bool operator== (const _Tipo_ptr orig) const {
			check_util("==");
			return static_cast<const void*>(_ptr_cntref) == static_cast<const void*>(orig);
		}
		NOINLINE__
		bool operator== (const Ptr& orig) const {
			check_util("==");
			orig.check_util("==");
			return _ptr_cntref == orig._ptr_cntref;
		}
		NOINLINE__
		bool operator!= (const _Tipo_ptr orig) const {
			check_util("!=");
			return static_cast<const void*>(_ptr_cntref) != static_cast<const void*>(orig);
		}
		NOINLINE__
		bool operator!= (const Ptr& orig) const {
			check_util("!=");
			orig.check_util("!=");
			return _ptr_cntref != orig._ptr_cntref;
		}
		//------------------------
		NOINLINE__
		_Tipo_base_ref operator*() const {
			check_desref("*");
			return *_ptr_cntref->_puntero;
		}
		//------------------------
		NOINLINE__
		_Tipo_ptr operator->() const {
			check_desref("->");
			return _ptr_cntref->_puntero;
		}
		//------------------------
		operator _Tipo_cntref_ptr () const {
			//check_util("delete");// Control realizado en CntRef::delete
			return _ptr_cntref;
		}
		//------------------------
		_Tipo_cntref_ptr __get() const {
			return _ptr_cntref;
		}
		//------------------------
		NOINLINE__
		void __set(_Tipo_cntref_ptr p) {
			copiar(p, "dynamic_cast");
		}
		//------------------------
		NOINLINE__
        friend std::ostream& operator<< (std::ostream&sal,const Ptr<TIPO_BASE>&ptr){
			ptr.check_util("<<");
			return sal << ptr._ptr_cntref;
		}
		//------------------------
	private:
		//void swap(Ptr& orig) {
		//	_Tipo_cntref_ptr aux = orig._ptr_cntref;
		//	orig._ptr_cntref = this->_ptr_cntref;
		//	this->_ptr_cntref = aux;
		//}
		void check_util(const char* op, const void* caller=RETADDR) const {
			if (umalcc::aux::CHECK_MODE_ON) {
				if (_ptr_cntref == umalcc::aux::PTR_INVALIDO) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Utilizacion de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> no inicializado";
					umalcc::aux::__terminar(msg.str(), caller);
				} else if ((_ptr_cntref != NULL)&&(_ptr_cntref->_liberado)) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Utilizacion de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> ya liberado " << _ptr_cntref;
					umalcc::aux::__terminar(msg.str(), caller);
				}
			}
		}
		void check_desref(const char* op, const void* caller=RETADDR) const {
			if (umalcc::aux::CHECK_MODE_ON) {
				if (_ptr_cntref == NULL) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Desreferenciacion de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> NULO";
					umalcc::aux::__terminar(msg.str(), caller);
				} else if (_ptr_cntref == umalcc::aux::PTR_INVALIDO) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Desreferenciacion de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> no incializado";
					umalcc::aux::__terminar(msg.str(), caller);
				} else if (_ptr_cntref->_liberado) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Desreferenciacion de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> ya liberado " << _ptr_cntref;
					umalcc::aux::__terminar(msg.str(), caller);
				}
			}
		}
		void destruir(const char* op, const void* caller=RETADDR) {
			if (umalcc::aux::CHECK_MODE_ON) {
				if ((_ptr_cntref != NULL)&&(_ptr_cntref != umalcc::aux::PTR_INVALIDO)) {
					if ((! _ptr_cntref->_liberado)&&(_ptr_cntref->_cntref == 1)) {
						std::ostringstream msg;
						// OJO, en este caso puede que al destruir el objeto apuntado
						// se pierda la referencia a alguna zona de memoria
						// por lo que el unwind_level se debe aumentar (+=2)
						// para el destruir de Ptr [ver Cuenta_Ref<>::delete]
						if (umalcc::aux::reg_caller(0) == 0) {
							msg << "["<<op<<"] "<< "Se pierde la referencia al Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> " << _ptr_cntref;
							umalcc::aux::__terminar(msg.str(),caller,(op[0]!='{'));
						} else {
							msg << "[delete] "<< "Se pierde la referencia al Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> " << _ptr_cntref;
							umalcc::aux::__terminar(msg.str(),
												 umalcc::aux::reg_caller(0));
						}
					}
					--_ptr_cntref->_cntref;
					if (_ptr_cntref->_cntref == 0) {
						delete _ptr_cntref;
					}
				}
			}
		}
		void copiar_nuevo(_Tipo_ptr ptr,const char*,
						  const void* caller=RETADDR) {
			_ptr_cntref = reinterpret_cast<_Tipo_cntref_ptr>(ptr);
			if (umalcc::aux::CHECK_MODE_ON) {
				if ((ptr != NULL)&&(ptr != umalcc::aux::PTR_INVALIDO)) {
					_ptr_cntref = new _Tipo_cntref(ptr);
					++_ptr_cntref->_cntref;
				}
			}
			(void)caller;
		}
		void copiar(_Tipo_cntref_ptr pcntrf,const char* op,
					const void* caller=RETADDR) {
			_ptr_cntref = pcntrf;
			if (umalcc::aux::CHECK_MODE_ON) {
#ifndef NOCOPYCHECK
				if (_ptr_cntref == umalcc::aux::PTR_INVALIDO) {
					std::ostringstream msg;
					msg << "["<<op<<"] "<< "Copia de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> no inicializado";
					umalcc::aux::__terminar(msg.str(), caller);
				}
#endif
				if ((_ptr_cntref != NULL)&&(_ptr_cntref != umalcc::aux::PTR_INVALIDO)) {
#ifndef NOCOPYCHECK
					if (_ptr_cntref->_liberado) {
						std::ostringstream msg;
						msg << "["<<op<<"] "<< "Copia de Ptr<"
						<< umalcc::aux::__demangle(typeid(_Tipo_base).name())
						<<"> ya liberado " << _ptr_cntref;
						umalcc::aux::__terminar(msg.str(), caller);
					}
#endif
					++_ptr_cntref->_cntref;
				}
			}
			(void)op; (void)caller;
		}
	};// class Ptr
	//-----------------------------------------------------------------
}// namespace umalcc
//--------------------------------------------------------------------------
#endif
//--------------------------------------------------------------------------
//--------------------------------------------------------------------------
//--------------------------------------------------------------------------
using umalcc::Ptr;
//---------------------------------------------------------------------
#endif
