function [Samples]=GenerateSamples3D(NdxDataset,NumSamples)
% Generate samples from some well known 3D datasets
% Inputs:
%   NdxDataset      Index of the desired 3D dataset
%   NumSamples      Number of samples to generate
% Based on Todd Wittman's MANIfold Learning Matlab Demo, http://www.math.ucla.edu/~wittman/mani/index.html

% ExParam=Extra parameter
N=NumSamples;
ExParam=1;

switch NdxDataset
    
    case 1  % Swiss Roll
        tt = (3*pi/2)*(1+2*rand(1,N));  
        height = 21*rand(1,N);
        Samples = [tt.*cos(tt); height; ExParam*tt.*sin(tt)]';

    case 2  % Swiss Hole
        % Swiss Roll w/ hole example taken from Donoho & Grimes
        tt = (3*pi/2)*(1+2*rand(1,2*N));  
        height = 21*rand(1,2*N);
        kl = repmat(0,1,2*N);
        for ii = 1:2*N
            if ( (tt(ii) > 9)&(tt(ii) < 12))
                if ((height(ii) > 9) & (height(ii) <14))
                    kl(ii) = 1;
                end;
            end;
        end;
        kkz = find(kl==0);
        tt = tt(kkz(1:N));
        height = height(kkz(1:N));
        Samples = [tt.*cos(tt); height; ExParam*tt.*sin(tt)]';     

    case 3  % Punctured Sphere by Saul & Roweis
        xx=-5+10*rand(N,1);
        yy=-5+10*rand(N,1);
        rr2 = xx(:).^2 + yy(:).^2;
        [tmp ii] = sort(rr2);
        Y = [xx(ii(1:N))'; yy(ii(1:N))'];
        a = 4./(4+sum(Y.^2));
        Samples = [a.*Y(1,:); a.*Y(2,:); ExParam*2*(1-a)]';

    case 4  % Twin Peaks by Saul & Roweis
        inc = 1.5 / sqrt(N);
        [xx2,yy2] = meshgrid(-1:inc:1);
        zz2 = sin(pi*xx2).*tanh(3*yy2);
        xy = 1-2*rand(2,N);
        Samples = [xy; sin(pi*xy(1,:)).*tanh(3*xy(2,:))]';
        Samples(:,3) = ExParam * Samples(:,3);

    case 5  % Toroidal Helix by Coifman & Lafon
        noiseSigma=0.05;   % noise parameter
        t = (1:N)'/N;
        t = t.^(ExParam)*2*pi;
        Samples = [(2+cos(8*t)).*cos(t) (2+cos(8*t)).*sin(t) sin(8*t)]+noiseSigma*randn(N,3);

    case 6  % Gaussian randomly sampled
        X = ExParam * randn(N,3);
        X(:,3) = 1 / (ExParam^2 * 2 * pi) * exp ( (-X(:,1).^2 - X(:,2).^2) / (2*ExParam^2) );
        Samples = X;

    case 7  % Ueda's spiral
        epsilon = 0.5;
        t = (([0:N-1]/(N-1))*4*pi)';
        Samples = [(13-(0.5*t)).*cos(t) -(13-(0.5*t)).*sin(t) t] + epsilon*randn(N,3);        
end

Samples=Samples';



