function [lp,lpte,LowerMargin,UpperMargin,Result] = OptimizedDHT(X, Xte, NumHistograms)
%  Optimize the parameters of the Discrete Histogram Transform
%  Coded by Ezequiel Lopez-Rubio, November 2013
%  Inputs:
%     X             DxN matrix of N training points in D dimensions
%     Xte           DxM matrix of M test points in D dimensions
%     NumHistograms Number of histograms for DHT
%  Outputs:
%     LowerMargin   Optimized value of the lower scale parameter theta_min
%     UpperMargin   Optimized value of the upper scale parameter theta_max
%     Result        Auxiliary results matrix with log densities for pilot
%                   values of the parameters
%     lp            Nx1 vector of log densities of X
%     lpte          Mx1 vector of log densities of Xte

% Number of histograms
if nargin<3
    NumHistograms = 50;
end

% Compute log densities for a grid of pilot values to initialize
% Nelder-Mead
CntLower=1;
IntervalLower=-2:0.5:2;
IntervalRange=0.5:0.5:3;
Result=zeros(numel(IntervalLower),numel(IntervalRange));
for LowerMargin=IntervalLower
    CntUpper=1;
    for Range=IntervalRange
        UpperMargin=LowerMargin+Range;
        Result(CntLower,CntUpper)=mean(lphtloo(X(:,1:100:end),NumHistograms,LowerMargin,UpperMargin));
        CntUpper=CntUpper+1;
    end
    CntLower=CntLower+1;
end

% Find the best values over the grid of pilot values
[Maximum Index]=max(Result(:));
[BestRow BestCol]=ind2sub(size(Result),Index);
w=zeros(2,1);
w(1)=IntervalLower(BestRow);
w(2)=LowerMargin+IntervalRange(BestCol);

% Nelder-Mead optimization, initialized with the best value over the grid
% of pilot values
OptimOptions = optimset('fminsearch');
OptimOptions = optimset(OptimOptions,'Display','none','MaxFunEvals',100,'TolX',0.001,'TolFun',0.001);
w=fminsearch(@(y) nmlphtloo(y, X(:,1:100:end), NumHistograms),w,OptimOptions);

% Get the optimal results
LowerMargin=w(1);
UpperMargin=w(2);

% Get the log densities over the training and test sets
lp   = lpht(X,X,NumHistograms,LowerMargin,UpperMargin);
lpte = lpht(X,Xte,NumHistograms,LowerMargin,UpperMargin);
end


% Mean negative log density and derivatives, computed by leave-one-out cross validation
function [nmlp,dnmlp] = nmlphtloo(w, X, NumHistograms)
nmlp = -mean(lphtloo(X,NumHistograms, w(1), w(2)));
if nargout>1
    dnmlp=zeros(2,1);
    h = 1e-4;
    dnmlp(1) = (-mean(lphtloo(X,NumHistograms, w(1)+h, w(2))) - nmlp)/h;    
    h = 1e-4;
    dnmlp(2) = (-mean(lphtloo(X,NumHistograms, w(1), w(2)+h )) - nmlp)/h;
end
end



% Log density computed by leave-one-out cross validation
function lpte = lphtloo(X,NumHistograms,LowerMargin,UpperMargin)
rand('twister',5489);
Model=HistogramTransform(X,NumHistograms,LowerMargin,UpperMargin);
TestDensity=TestHTLOOMEX(Model,X);
lpte=log(TestDensity);
end

% Log density computed over a test set
function lpte = lpht(X,Xte,NumHistograms,LowerMargin,UpperMargin)
rand('twister',5489);
Model=HistogramTransform(X,NumHistograms,LowerMargin,UpperMargin);
TestDensity=TestHTMEX(Model,Xte);
lpte=log(TestDensity);
end

