function [EstimatedSamples,EstimatedSamplesModel]=EstimateMissingDataMMED(Model,Samples)
% Estimate missing values (marked with NaN) by means of a MMED
% Inputs:
%	Model=The MMED model used to estimate missing values
%	Samples=Test data with missing values (marked with NaNs)
% Outputs:
%	EstimatedSamples=Test data with the missing values replaced by their estimations
%		according to the MMED model.
%		For each input column vector Samples(:,j) we get an estimation EstimatedSamples(:,j)
%	EstimatedSamplesModel=Test data with the missing values replaced by their estimations
%		according to each GMC of the MMED model.
%		For each input column vector Samples(:,j) and each GMC i we get an estimation
%		EstimatedSamplesModel{i}(:,j)

Dimension=size(Model.Samples,1);
EstimatedSamples=zeros(size(Samples));

for NdxSample=1:size(Samples,2)
    MySample=Samples(:,NdxSample);
    NdxAbsent=find(isnan(MySample));
    
    
    % See whether there is any absent value to be estimated
    if ~isempty(NdxAbsent)
        
        NdxPresent=find(isfinite(MySample));
        % Remove the NaNs
        MySample(NdxAbsent)=0;
        SumResp=0;
        
        % See which GMC yields the highest likelihood
        NdxWinner=1;
        LogDensWinner=-inf;
        
        
        for NdxGroup=1:Model.NumGroups
            
            % Get the covariance matrix
            Covariances=Model.C{NdxGroup};
            
            % Compute the covariance submatrices and the B matrix
            PresentPresentCovariances=Covariances(NdxPresent,NdxPresent);
            PresentAbsentCovariances=Covariances(NdxPresent,NdxAbsent);
            InvPresentPresentCovariances=inv(PresentPresentCovariances);
            MatrixB=InvPresentPresentCovariances*PresentAbsentCovariances;
            
            % Mean of the absent values
            MeanAbsent=Model.Means{NdxGroup};
            MeanAbsent(NdxPresent)=0;
            
            % Deviation of the present values with respect to their mean
            DevMeanPresent=MySample-Model.Means{NdxGroup};
            DevMeanPresent=DevMeanPresent(NdxPresent);
            
            % Estimation of the deviation of the absent values
            CompressedDevMeanAbsent=MatrixB'*DevMeanPresent;
            DevMeanAbsent=zeros(Dimension,1);
            DevMeanAbsent(NdxAbsent)=CompressedDevMeanAbsent;
            
            % Carry out the estimation
            EstimatedSamplesModel{NdxGroup}(:,NdxSample)=MySample+MeanAbsent+DevMeanAbsent;
                       
            % Find p(present values | i)
            DetCov=det(PresentPresentCovariances);
            
            % A priori probability of this GMC
            ProbAPrioriGroup=sum(Model.Pi((1+Model.NumUnitsPerGroup*(NdxGroup-1)):...
                Model.NumUnitsPerGroup*NdxGroup));
            
            if DetCov==0
                MyResp=0;
            else      
                MyLogDens=-0.5*length(NdxPresent)-0.5*log(DetCov)...
                    -0.5*DevMeanPresent'*InvPresentPresentCovariances*DevMeanPresent;
                % See whether this is the winning (most likely) GMC
                if MyLogDens>LogDensWinner
                    LogDensWinner=MyLogDens;
                    NdxWinner=NdxGroup;
                end
                MyResp=ProbAPrioriGroup*exp(MyLogDens);
            end
            SumResp=SumResp+MyResp;
            
            % The global estimation is the weighted mean of the estimations of the GMCs:
            % (estimated t sub n)= summation on i { (R sub ni) *(t sub n estimated by the
            %      i-th GMC) } / summation on j { R sub nj } 
            EstimatedSamples(:,NdxSample)=EstimatedSamples(:,NdxSample)+MyResp*...
                EstimatedSamplesModel{NdxGroup}(:,NdxSample);
            
        end
        
        % Normalize the weighted mean
        if SumResp==0
            % In case of numerical errors, use the estimation from the most likely GMC
            EstimatedSamples(:,NdxSample)=EstimatedSamplesModel{NdxWinner}(:,NdxSample);
        else
            EstimatedSamples(:,NdxSample)=EstimatedSamples(:,NdxSample)/SumResp;
        end
        
    else
        % No absent values to estimate
        EstimatedSamples(:,NdxSample)=MySample;
        for NdxGroup=1:Model.NumGroups
            EstimatedSamplesModel{NdxGroup}(:,NdxSample)=MySample;
        end
        
    end
end