#include "mex.h"
#include "Mates.h"
#include "Debugging.h"
#include "gmp.h"
#include "mpfr.h"
#include <stdio.h>
#include <math.h>
#include <time.h>
#include <float.h>
#include <string.h>

#ifndef M_PI
#define M_PI 3.141592653589793238462643
#endif

/* 

Reference:
E. Lopez-Rubio, Stochastic Approximation Learning for Mixtures of Multivariate Elliptical Distributions.
Neurocomputing.

In order to compile this MEX function, type the following at the MATLAB prompt:
32-bit Windows:
mex MMEDANLLMEX.c MatesLap.c lapack.a blas.a libf2c.a libmpfr.a libgmp.a Debugging.c
64-bit Windows:
mex LINKFLAGS="$LINKFLAGS /NODEFAULTLIB:LIBCMT" MMEDANLLMEX.c MatesLap.c Debugging.c clapack_nowrap.lib BLAS_nowrap.lib libf2c.lib mpir.lib mpfr.lib

[ANLL,LogProbDensities,ResponsibilitiesUnits,ResponsibilitiesGroups,DistMahal] = MMEDANLLMEX(Samples,Model)

ResponsibilitiesUnits(NdxUnit,NdxSample)=Responsibility of the unit in the generation of the sample
ResponsibilitiesGroups(NdxGroups,NdxSample)=Responsibility of the group in the generation of the sample
DistMahal(NdxGroup,NdxSample)=Squared Mahalanobis distance of the sample with respect to the group

*/


/*----------------------------------------------------------------------------------*/


/* Find log(p(t sub n)), the responsibilities of the units and the groups and the
           squared Mahalanobis distances with respect to the covariance matrix */       
void WinnerMMEDProbMEX(mxArray* Model,double *ptrSample,
    double *LogDensityProb,double *ptrResponsibilitiesUnits,
    double *ptrResponsibilitiesGroups,
    double *ptrDistMahal);

/* Finding log(p(t sub n | i), where i is a unit
Important note: any change to this function must be propagated to all the files of the model,
since the interpretation of the units of the groups would change */
void FindLogDensityUnit(int NdxUnit,int NumUnitsPerGroup,
     double DistMahal,double D,double LogDetC,double *MyLogDensity);
    
double gammln(double xx);

double besselkn(int nn, double x);


void mexFunction(int nlhs, mxArray* plhs[],
                 int nrhs, const mxArray* prhs[])
{  
    int SpaceDimension,NumSamples,IndexPat,NumGroups,NumUnitsPerGroup;
    const int *DimSamples;
    const int *DimModel;
    double *ptrSamples,*Pattern,*ptrANLL,*ptrLogProbDensities;
    double *ptrResponsibilitiesUnits,*ptrResponsibilitiesGroups;
    double *ptrDistMahal;
    double SumLogProbDensities;
    mxArray *Pi;

    
    
    /* Get input data */
    DimSamples=mxGetDimensions(prhs[0]);
    SpaceDimension=DimSamples[0];
    NumSamples=DimSamples[1];
    ptrSamples=mxGetPr(prhs[0]);
    Pi=mxGetField(prhs[1],0,"Pi");
    DimModel=mxGetDimensions(Pi);
    NumUnitsPerGroup=DimModel[0];
    NumGroups=DimModel[1];
    
    /* Create output matrices */
    plhs[0]=mxCreateNumericMatrix(1, 1, mxDOUBLE_CLASS, mxREAL);
    ptrANLL=mxGetPr(plhs[0]);
    plhs[1]=mxCreateNumericMatrix(1, NumSamples, mxDOUBLE_CLASS, mxREAL);
    ptrLogProbDensities=mxGetPr(plhs[1]);
    plhs[2]=mxCreateNumericMatrix(NumGroups*NumUnitsPerGroup, NumSamples, mxDOUBLE_CLASS, mxREAL);
    ptrResponsibilitiesUnits=mxGetPr(plhs[2]);
    plhs[3]=mxCreateNumericMatrix(NumGroups, NumSamples, mxDOUBLE_CLASS, mxREAL);
    ptrResponsibilitiesGroups=mxGetPr(plhs[3]);
	plhs[4]=mxCreateNumericMatrix(NumGroups, NumSamples, mxDOUBLE_CLASS, mxREAL);
	ptrDistMahal=mxGetPr(plhs[4]);
    

    /* Main loop to find the logarithms of the densities */
    SumLogProbDensities=0.0;
    for(IndexPat=0;IndexPat<NumSamples;IndexPat++)
    {
        
        /* Pattern = Samples(:,IndexPat); */
        Pattern=ptrSamples+IndexPat*SpaceDimension;
        
        
        /* Find log(p(t sub n)), the responsibilities of the units and the groups 
           and the squared Mahalanobis distances with respect to the covariance matrix */
        WinnerMMEDProbMEX((mxArray*) prhs[1],Pattern,
                ptrLogProbDensities+IndexPat,
                ptrResponsibilitiesUnits+IndexPat*NumGroups*NumUnitsPerGroup,
                ptrResponsibilitiesGroups+IndexPat*NumGroups,
                ptrDistMahal+IndexPat*NumGroups);
        /* Accumulate to find ANLL*/
        SumLogProbDensities+=ptrLogProbDensities[IndexPat];
        
    }    
    (*ptrANLL)=-SumLogProbDensities/(double)NumSamples;  
    
    
     
}    



/*--------------------------------------------------------------------*/


/* Find log(p(t sub n)), the responsibilities of the units and the groups and the
           squared Mahalanobis distances with respect to the covariance matrix */        
void WinnerMMEDProbMEX(mxArray* Model,double *ptrSample,
    double *LogDensityProb,double *ptrResponsibilitiesUnits,
    double *ptrResponsibilitiesGroups,
    double *ptrDistMahal)
{   
    
    const int *DimModel;
    const int *DimSamples;
    
    int SpaceDimension,NumGroups,NumUnitsPerGroup,Index;
    int NdxGroup,NdxUnit,NdxDim;
    mxArray *Samples,*Means,*C,*CInv,*Pi;
    double *ptrVectorDif,*ptrVectorProd,*ptrMyMean,*ptrMyC,*ptrMyCInv,*ptrPi;
    double *ptrMyEigenValues,*ptrMyEigenVectors;
    mpfr_t *ptrDensitiesUnits,*ptrDensitiesGroups,*ptrAux;
    double LogDetC,MyLogDensity,D,SumPis,ProbUnitGivenGroup;
    mpfr_t aux,sum;
    
    
    mpfr_set_default_prec(100);
    
    /* Get the input data*/
    Means=mxGetField(Model,0,"Means");
    C=mxGetField(Model,0,"C");
    CInv=mxGetField(Model,0,"CInv");
    Pi=mxGetField(Model,0,"Pi");
    ptrPi=mxGetPr(Pi);
    DimModel=mxGetDimensions(Pi);
    NumUnitsPerGroup=DimModel[0];
    NumGroups=DimModel[1];
    Samples=mxGetField(Model,0,"Samples");
    DimSamples=mxGetDimensions(Samples);
    SpaceDimension=DimSamples[0];
    
    
    
    /* Create auxiliary matrices */
    ptrVectorDif=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrVectorProd=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrMyEigenValues=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrMyEigenVectors=mxMalloc(SpaceDimension*SpaceDimension*sizeof(double));
    
    
    ptrDensitiesUnits=mxMalloc(NumGroups*NumUnitsPerGroup*sizeof(mpfr_t));
    ptrDensitiesGroups=mxMalloc(NumGroups*sizeof(mpfr_t));    
    ptrAux=mxMalloc(NumGroups*NumUnitsPerGroup*sizeof(mpfr_t));
    for(Index=0;Index<NumGroups*NumUnitsPerGroup;Index++)
    {
        mpfr_init(ptrDensitiesUnits[Index]);        
        mpfr_init(ptrAux[Index]);        
    }     
    for(Index=0;Index<NumGroups;Index++)
    {
        mpfr_init(ptrDensitiesGroups[Index]);        
    }   
    mpfr_init(aux);
    mpfr_init(sum);


    /* Finding the a posteriori probability densities of the units, p(t sub n | i) */
    Index=0;    
    for(NdxGroup=0;NdxGroup<NumGroups;NdxGroup++)
    {
                                                 
        /* Prepare auxiliary matrices */
        ptrMyMean=mxGetPr(mxGetCell(Means,NdxGroup));
        ptrMyC=mxGetPr(mxGetCell(C,NdxGroup));
        ptrMyCInv=mxGetPr(mxGetCell(CInv,NdxGroup));
        
        
        
        /* VectorDif=Samples(:,NdxSample) - Model.Mu{NdxGroup}; */
        MatrixDifference(ptrSample,ptrMyMean,
                ptrVectorDif,SpaceDimension,1);
        
        
        /* Find squared Mahalanobis distance with respect to the covariance matrix */
        MatrixProduct(ptrMyCInv,ptrVectorDif,ptrVectorProd,SpaceDimension,
                SpaceDimension,1);
        MatrixProduct(ptrVectorDif,ptrVectorProd,ptrDistMahal+NdxGroup,
                1,SpaceDimension,1);
                
        
        /* Find the eigenvalues for computing the determinant of the covariance matrix */
        EigValVec(ptrMyC,ptrMyEigenValues,ptrMyEigenVectors,SpaceDimension);    
        
        /* Finding log(det C) */
        LogDetC=0.0;
        for(NdxDim=0;NdxDim<SpaceDimension;NdxDim++)
        {
            LogDetC+=log(ptrMyEigenValues[NdxDim]);
        }
        
        D=(double)SpaceDimension;
                                                 
        for(NdxUnit=0;NdxUnit<NumUnitsPerGroup;NdxUnit++)
        {
                                                         
            /* Finding log(p(t sub n | i)) */
            
            FindLogDensityUnit(NdxUnit,NumUnitsPerGroup,
                 ptrDistMahal[NdxGroup],D,LogDetC,&MyLogDensity);
            
            
            /* This is p(t sub n | i), where i is a unit */
            /*ptrDensities[Index]=exp(MyLogDensity);*/
            mpfr_set_d(ptrDensitiesUnits[Index],MyLogDensity,GMP_RNDN);
            mpfr_exp(ptrDensitiesUnits[Index], ptrDensitiesUnits[Index], GMP_RNDN); /* Exponential e ^ datum */
            
            /* Index to cycle over the units */
            Index++;
     
        }    
 
    }    
    
    
    
    /* Finding the responsibilities of the units, p(i | t sub n)*/
    mpfr_set_d(sum,0,GMP_RNDN);
    for(Index=0;Index<NumGroups*NumUnitsPerGroup;Index++)
    {
        mpfr_set_d(aux,ptrPi[Index],GMP_RNDN);
        /* We store in ptrAux[Index] the unnormalized value of p(i | t sub n), where i is a unit */
        mpfr_mul(ptrAux[Index],aux,ptrDensitiesUnits[Index],GMP_RNDN);
        /* Accumulate the sum of the unnormalized values */
        mpfr_add(sum,sum,ptrAux[Index],GMP_RNDN);
    }
    /* Normlize and convert to double */
    for(Index=0;Index<NumGroups*NumUnitsPerGroup;Index++)
    {
        /* Normlize the value p(i | t sub n) */
        mpfr_div(ptrAux[Index],ptrAux[Index],sum,GMP_RNDN);
        /* Convert to double */
        ptrResponsibilitiesUnits[Index]=mpfr_get_d(ptrAux[Index],GMP_RNDN);
    }
    
    
    
    /* Find the a priori probability densities of the groups, p(t sub n | j),
    and the logarithm of the overall probability density, log(p(t sub n)) */
    Index=0;
    mpfr_set_d(sum,0,GMP_RNDN);
    for(NdxGroup=0;NdxGroup<NumGroups;NdxGroup++)
    {    
        /* Find the sum of the a priori probabilities of the units of the group,
        which is the a priori probability of the group, P(j) */
        SumPis=0.0;
        for(NdxUnit=0;NdxUnit<NumUnitsPerGroup;NdxUnit++)
        {
             SumPis+=ptrPi[NdxUnit+NdxGroup*NumUnitsPerGroup];
        }
		
        /* Finding p(t sub n |j) */        
		mpfr_set_d(ptrDensitiesGroups[NdxGroup],0,GMP_RNDN);
		if (SumPis>0.0)
		{			
			for(NdxUnit=0;NdxUnit<NumUnitsPerGroup;NdxUnit++)
			{
				 /* Probability P(i | j), where i is a unit and j is a group */
				 ProbUnitGivenGroup=ptrPi[Index]/SumPis;
				 /* Find P(i | j)*p(t sub n | i) */
				 mpfr_set_d(aux,ProbUnitGivenGroup,GMP_RNDN);
				 mpfr_mul(aux,aux,ptrDensitiesUnits[Index],GMP_RNDN);
				 /* Accumulate to find p(t sub n |j) */
				 mpfr_add(ptrDensitiesGroups[NdxGroup],ptrDensitiesGroups[NdxGroup],aux,GMP_RNDN);
				 /* Index to cycle over the units */
				 Index++;
			}
			/* Accumulate to find p(t sub n)*/
			mpfr_set_d(aux,SumPis,GMP_RNDN);
			mpfr_mul(aux,aux,ptrDensitiesGroups[NdxGroup],GMP_RNDN);
			mpfr_add(sum,sum,aux,GMP_RNDN);
		}
		else
		{
			Index+=NumUnitsPerGroup;
		}
    }
    /* Find log(p(t sub n)) and convert it to double*/	
    mpfr_log(sum,sum,GMP_RNDN);
    (*LogDensityProb)=mpfr_get_d(sum,GMP_RNDN);         
    
    /* Find the responsibilities of the groups, p(j | t sub n)*/
    for(NdxGroup=0;NdxGroup<NumGroups;NdxGroup++)
    {
        /* Find the responsibility of the group from the responsibilities of its units */
        SumPis=0.0;
        for(NdxUnit=0;NdxUnit<NumUnitsPerGroup;NdxUnit++)
        {
             SumPis+=ptrResponsibilitiesUnits[NdxUnit+NdxGroup*NumUnitsPerGroup];
        }
		ptrResponsibilitiesGroups[NdxGroup]=SumPis;
    }	
    
    /* Release memory */
    mpfr_clear(aux);
    mpfr_clear(sum);    
    for(Index=0;Index<NumGroups*NumUnitsPerGroup;Index++)
    {
        mpfr_clear(ptrDensitiesUnits[Index]);        
        mpfr_clear(ptrAux[Index]);        
    } 
    for(Index=0;Index<NumGroups;Index++)
    {
        mpfr_clear(ptrDensitiesGroups[Index]);            
    }
    mxFree(ptrVectorDif);
    mxFree(ptrVectorProd);
    mxFree(ptrDensitiesUnits);
    mxFree(ptrDensitiesGroups);    
    mxFree(ptrAux);
    mxFree(ptrMyEigenValues);
    mxFree(ptrMyEigenVectors);
    
}    


/* Finding log(p(t sub n | i), where i is a unit
Important note: any change to this function must be propagated to all the files of the model,
since the interpretation of the units of the groups would change */
void FindLogDensityUnit(int NdxUnit,int NumUnitsPerGroup,
     double DistMahal,double D,double LogDetC,double *MyLogDensity)
{
     int TypeDistribution,SubNdxUnit;
     static double NuParameters[25]={2.01, 1000.0, 5.0, 100.0, 10.0, 50.0, 20.0, 500.0, 40.0, 30.0,
            3.0, 4.0, 6.0, 7.0, 8.0, 9.0, 15.0, 25.0, 35.0, 45.0, 200.0, 300.0, 400.0, 600.0, 700.0};
     static double Delta2Parameters[25]={ 1.0, 3.0, 0.5, 6.0, 0.25, 75.0, 0.125, 10.0, 0.1, 1.5,
            0.75, 40.0, 0.2, 15.0, 0.3, 30.0, 0.8, 0.01, 100.0, 0.05, 20.0, 0.02, 50.0, 0.0001, 1000.0};
     double MyNu,MyDist,MyDist2,LogNormlizingConstant;
     double LambdaParameter,DistGamma;
     
     
     /* Check whether the inputs are valid */
     if (NumUnitsPerGroup<=0)
     {
         mexErrMsgTxt("The value of NumUnitsPerGroup must be at least 1 in MMEDANLLMEX.c");    
     }
     if (NumUnitsPerGroup>78)
     {
         mexErrMsgTxt("The value of NumUnitsPerGroup can not exceed from 78 in MMEDANLLMEX.c");    
     }
     
     /* Choose a distribution from the value of NdxUnit */
     if (NdxUnit<=2)
     {
          /* Distributions with no additional parameters */
          switch(NdxUnit)
          {
              case 0:
                  /* Multivariate Gaussian density */
                  (*MyLogDensity)=-0.5*D*1.837877066409345-0.5*LogDetC-0.5*DistMahal;                
                   break;
                   
              case 1:
                   /* Exponential square root density */
                   LogNormlizingConstant=0.5*D*log(D-1.0)+gammln(0.5*D)-gammln(D)
                        -0.5*LogDetC-0.5*D*1.144729885849400;
                   (*MyLogDensity)=LogNormlizingConstant+log(sqrt(D-1.0))
                       -sqrt((D-1.0)*DistMahal)-log(2.0*sqrt(DistMahal));   
                                  
                   break;
                   
              case 2:
                   /* Multivariate Laplacian density */
                   if ((D>=6.0) && (DistMahal>0.0))
                   {                          
                       LambdaParameter=pow(exp(LogDetC),1.0/D);
                       /* Distance with respect to the Gamma matrix */
                       DistGamma=LambdaParameter*DistMahal;
					   /* Probability density */
                       (*MyLogDensity)=-0.5*D*1.837877066409345+0.693147180559945
                          -log(LambdaParameter)
                          +log(besselkn((int)(0.5*D-1.0), sqrt(2.0*DistMahal)))
                          -(0.5*D-1.0)*log(sqrt(0.5*LambdaParameter*DistGamma));  
                   }                    
                   else
                   {
                       /* The dimension D is too small for besselkn() to work, so we set a nearly zero density */
					   /* We also get here if DistMahal==0.0, which yields numerical problems */
                       (*MyLogDensity)=-500.0;
                   }  
                   break;
                   
          }
     }
     else
     {     
         /* It is a distribution with additional parameters */
         TypeDistribution=(NdxUnit-3)%3;
         SubNdxUnit=(NdxUnit-3)/3;
         
         /* Switch depending on the kind of distribution */
         switch(TypeDistribution)
         {
             case 0:
                  /* Multivariate Student-t density */
                  MyNu=NuParameters[SubNdxUnit];
                  (*MyLogDensity)=gammln(0.5*D+0.5*MyNu)-gammln(0.5*MyNu)-0.5*LogDetC
                      -0.5*D*(1.144729885849400+log(MyNu-2.0))
                      -0.5*(D+MyNu)*log(1.0+DistMahal/(MyNu-2.0));
                  break;

             case 1:
                  /* Uniform distribution on the hyperellipsoid
                  { x | (x-Media)' * inv(C) * (x-Media) <= Delta } */
                  MyDist2=Delta2Parameters[SubNdxUnit];
                  /* Check whether the point falls inside the hyperellipsoid */
                  if (DistMahal<MyDist2)
                  {
                      /* It falls inside: the probability density is the inverse of the volume of the hyperellipsoid */
                     (*MyLogDensity)=-0.5*LogDetC-0.5*D*(log(MyDist2)+1.144729885849400)
                          +gammln(0.5*D+1.0);
                          
                  }
                  else
                  {
                      /* It falls outside: we set a nearly zero probability density */
                      (*MyLogDensity)=-500.0;
                  }
                      
                  break;
                  
             case 2:
				  /* Multivariate triangular density */
				  MyDist2=Delta2Parameters[SubNdxUnit];	/* delta ^2 */			  
                  /* Check whether the point falls inside the hyperellipsoid */
                  if (DistMahal<MyDist2)
                  {
                      /* It falls inside: the probability density decreases with the Mahalanobis distance to the 
					  mean (the center of the hyperellipsoid) */
					  MyDist=sqrt(MyDist2); /* delta */
                     (*MyLogDensity)=gammln(0.5*D)-0.693147180559945-0.5*LogDetC
						 -0.5*D*1.144729885849400
						 -(D+1.0)*log(MyDist)
						 -log( (1.0/D)-(1.0/(D+1.0)) )
                          +log(fabs(MyDist-sqrt(DistMahal))); /* fabs to correct numerical errors */
                          
                  }
                  else
                  {
                      /* It falls outside: we set a nearly zero probability density */
                      (*MyLogDensity)=-500.0;
                  }

                  break;    
             
         }
         
     }
     
     
            
}




/*

  Purpose:

    ALNGAM computes the logarithm of the gamma function.

  Modified:

    13 January 2008

  Author:

    Allan Macleod
    C++ version by John Burkardt

  Reference:

    Allan Macleod,
    Algorithm AS 245,
    A Robust and Reliable Algorithm for the Logarithm of the Gamma Function,
    Applied Statistics,
    Volume 38, Number 2, 1989, pages 397-402.

  Parameters:

    Input, double XVALUE, the argument of the Gamma function.

    Output, int IFAULT, error flag.
    0, no error occurred.
    1, XVALUE is less than or equal to 0.
    2, XVALUE is too big.

    Output, double ALNGAM, the logarithm of the gamma function of X.
*/
double alngam ( double xvalue, int *ifault )
{
  double alr2pi = 0.918938533204673;
  double r1[9] = {
    -2.66685511495, 
    -24.4387534237, 
    -21.9698958928, 
     11.1667541262, 
     3.13060547623, 
     0.607771387771, 
     11.9400905721, 
     31.4690115749, 
     15.2346874070 };
  double r2[9] = {
    -78.3359299449, 
    -142.046296688, 
     137.519416416, 
     78.6994924154, 
     4.16438922228, 
     47.0668766060, 
     313.399215894, 
     263.505074721, 
     43.3400022514 };
  double r3[9] = {
    -2.12159572323E+05, 
     2.30661510616E+05, 
     2.74647644705E+04, 
    -4.02621119975E+04, 
    -2.29660729780E+03, 
    -1.16328495004E+05, 
    -1.46025937511E+05, 
    -2.42357409629E+04, 
    -5.70691009324E+02 };
  double r4[5] = {
     0.279195317918525, 
     0.4917317610505968, 
     0.0692910599291889, 
     3.350343815022304, 
     6.012459259764103 };
  double value;
  double x;
  double x1;
  double x2;
  double xlge = 510000.0;
  double xlgst = 1.0E+30;
  double y;

  x = xvalue;
  value = 0.0;
/*
  Check the input.
*/
  if ( xlgst <= x )
  {
    *ifault = 2;
    return value;
  }

  if ( x <= 0.0 )
  {
    *ifault = 1;
    return value;
  }

  *ifault = 0;
/*
  Calculation for 0 < X < 0.5 and 0.5 <= X < 1.5 combined.
*/
  if ( x < 1.5 )
  {
    if ( x < 0.5 )
    {
      value = - log ( x );
      y = x + 1.0;
/*
  Test whether X < machine epsilon.
*/
      if ( y == 1.0 )
      {
        return value;
      }
    }
    else
    {
      value = 0.0;
      y = x;
      x = ( x - 0.5 ) - 0.5;
    }

    value = value + x * (((( 
        r1[4]   * y 
      + r1[3] ) * y 
      + r1[2] ) * y 
      + r1[1] ) * y 
      + r1[0] ) / (((( 
                  y 
      + r1[8] ) * y 
      + r1[7] ) * y 
      + r1[6] ) * y 
      + r1[5] );

    return value;
  }
/*
  Calculation for 1.5 <= X < 4.0.
*/
  if ( x < 4.0 )
  {
    y = ( x - 1.0 ) - 1.0;

    value = y * (((( 
        r2[4]   * x 
      + r2[3] ) * x 
      + r2[2] ) * x 
      + r2[1] ) * x 
      + r2[0] ) / (((( 
                  x 
      + r2[8] ) * x 
      + r2[7] ) * x 
      + r2[6] ) * x 
      + r2[5] );
  }
/*
  Calculation for 4.0 <= X < 12.0.
*/
  else if ( x < 12.0 ) 
  {
    value = (((( 
        r3[4]   * x 
      + r3[3] ) * x 
      + r3[2] ) * x 
      + r3[1] ) * x 
      + r3[0] ) / (((( 
                  x 
      + r3[8] ) * x 
      + r3[7] ) * x 
      + r3[6] ) * x 
      + r3[5] );
  }
/*
  Calculation for 12.0 <= X.
*/
  else
  {
    y = log ( x );
    value = x * ( y - 1.0 ) - 0.5 * y + alr2pi;

    if ( x <= xlge )
    {
      x1 = 1.0 / x;
      x2 = x1 * x1;

      value = value + x1 * ( ( 
             r4[2]   * 
        x2 + r4[1] ) * 
        x2 + r4[0] ) / ( ( 
        x2 + r4[4] ) * 
        x2 + r4[3] );
    }
  }

  return value;
}

/* Natural logarithm of the gamma function */
double gammln(double argument)
{
	int ifault;
    
    return alngam(argument,&ifault);

}



/*************************************************************************
Modified Bessel function, second kind, integer order

Returns modified Bessel function of the second kind
of order n of the argument.

The range is partitioned into the two intervals [0,9.55] and
(9.55, infinity).  An ascending power series is used in the
low range, and an asymptotic expansion in the high range.

ACCURACY:

                     Relative error:
arithmetic   domain     # trials      peak         rms
   IEEE      0,30        90000       1.8e-8      3.0e-10

Error is high only near the crossover point x = 9.55
between the two expansions used.

Cephes Math Library Release 2.8:  June, 2000
Copyright 1984, 1987, 1988, 2000 by Stephen L. Moshier
*************************************************************************/
double besselkn(int nn, double x)
{
    double k;
    double kf;
    double nk1f;
    double nkf;
    double zn;
    double t;
    double s;
    double z0;
    double z;
    double ans;
    double fn;
    double pn;
    double pk;
    double zmn;
    double tlg;
    double tox;
    int i;
    int n;
    double eul;
    double result;


    eul = 5.772156649015328606065e-1;
    if( nn<0 )
    {
        n = -nn;
    }
    else
    {
        n = nn;
    }
    
	if (n>31)
	{
		mexErrMsgTxt("Overflow in BesselKN");
	}
	if(x<=0.0)
	{
		mexErrMsgTxt("Domain error in BesselKN");
	}

    if( x<=9.55 )
    {
        ans = 0.0;
        z0 = 0.25*x*x;
        fn = 1.0;
        pn = 0.0;
        zmn = 1.0;
        tox = 2.0/x;
        if( n>0 )
        {
            pn = -eul;
            k = 1.0;
            for(i=1; i<=n-1; i++)
            {
                pn = pn+1.0/k;
                k = k+1.0;
                fn = fn*k;
            }
            zmn = tox;
            if( n==1 )
            {
                ans = 1.0/x;
            }
            else
            {
                nk1f = fn/n;
                kf = 1.0;
                s = nk1f;
                z = -z0;
                zn = 1.0;
                for(i=1; i<=n-1; i++)
                {
                    nk1f = nk1f/(n-i);
                    kf = kf*i;
                    zn = zn*z;
                    t = nk1f*zn/kf;
                    s = s+t;
					if( (DBL_MAX - fabs(t)) < fabs(s) )
					{
						mexErrMsgTxt("Overflow in BesselKN");
					}
					if( (tox > 1.0) && ((DBL_MAX/tox) < zmn) )
					{
						mexErrMsgTxt("Overflow in BesselKN");
					}				
                    zmn = zmn*tox;
                }
                s = s*0.5;
                t = fabs(s);
                if( (zmn > 1.0) && ((DBL_MAX/zmn) < t) )
				{
					mexErrMsgTxt("Overflow in BesselKN");
				}
				if( (t > 1.0) && ((DBL_MAX/t) < zmn) )
				{
					mexErrMsgTxt("Overflow in BesselKN");
				}
                ans = s*zmn;
            }
        }
        tlg = 2.0*log(0.5*x);
        pk = -eul;
        if( n==0 )
        {
            pn = pk;
            t = 1.0;
        }
        else
        {
            pn = pn+1.0/n;
            t = 1.0/fn;
        }
        s = (pk+pn-tlg)*t;
        k = 1.0;
        do
        {
            t = t*(z0/(k*(k+n)));
            pk = pk+1.0/k;
            pn = pn+1.0/(k+n);
            s = s+(pk+pn-tlg)*t;
            k = k+1.0;
        }
        while(fabs(t/s) > DBL_EPSILON);
        s = 0.5*s/zmn;
        if( n%2!=0 )
        {
            s = -s;
        }
        ans = ans+s;
        result = ans;
        return result;
    }
    if( x>log(DBL_MAX) )
    {
        result = 0;
        return result;
    }
    k = n;
    pn = 4.0*k*k;
    pk = 1.0;
    z0 = 8.0*x;
    fn = 1.0;
    t = 1.0;
    s = t;
    nkf = DBL_MAX;
    i = 0;
    do
    {
        z = pn-pk*pk;
        t = t*z/(fn*z0);
        nk1f = fabs(t);
        if( (i >= n) && (nk1f > nkf) )
        {
            break;
        }
        nkf = nk1f;
        s = s+t;
        fn = fn+1.0;
        pk = pk+2.0;
        i = i+1;
    }
    while(fabs(t/s) > DBL_EPSILON);
    result = exp(-x) * sqrt( M_PI/(2.0*x) ) * s;
    return result;
}