function [Constants]=PrecomputeConstants(Dimension,Limits)
% Precompute the constants needed for histogram analysis
% Inputs:
%	Dimension=Input space dimension
%	Limits=Boundaries among the bins of the histogram, expressed as Mahalanobis distances,
%		rho=sqrt( (Sample-Mu)' *inv(C)* (Sample-Mu) )
% Output:
%	Constants=Constants for histogram analysis,
%		Constants(NdxUnit,NdxBin)=P( Limits(NdxBin) < rho < Limits(NdxBin+1) | NdxUnit )
%		where rho stands for the Mahalanobis distance of a sample
% Note: the integral corresponding to each bin is approximated by 100 points


NumBins=numel(Limits)-1;
Constants=zeros(78,NumBins);
% Create a dummy MMED model which will be used to evaluate the probability densities
Model.Samples=zeros(Dimension,100*NumBins);
Model.NumGroups=1;
Model.NumUnitsPerGroup=78;
Model.NumParameters=79;
Model.Means{1}=zeros(Dimension,1);
Model.C{1}=eye(Dimension);
Model.CInv{1}=eye(Dimension);

% Compute the points where we are going to evaluate the probability density so as to 
% estimate the integral
for NdxLim=1:NumBins
    Start=Limits(NdxLim);
    Finish=Limits(NdxLim+1);
    StepSize(NdxLim)=(Finish-Start)/100;
    Values=Start:StepSize(NdxLim):Finish;    
    Model.Samples(1,(NdxLim-1)*100+1:NdxLim*100)=Values(1:100);
end
    

% Evaluate the integrals for each of the units of the dummy MMED model
for NdxUnit=1:78
    Model.Pi=zeros(78,1);
    Model.Pi(NdxUnit)=1;
    [ANLL,LogProbDensities,ResponsibilitiesUnits,ResponsibilitiesGroups,DistMahal] = MMEDANLLMEX(Model.Samples,Model);
    Densities=exp(LogProbDensities);
    Densities(~isfinite(Densities))=0;
    % The integrand is the probability density multiplied by the input space dimension minus one
    Integrands=Densities.*(Model.Samples(1,:).^(Dimension-1));
    for NdxLim=1:NumBins
        Constants(NdxUnit,NdxLim)=(2*(pi^(0.5*Dimension))/gamma(0.5*Dimension))*StepSize(NdxLim)*sum(Integrands((NdxLim-1)*100+1:NdxLim*100));
    end
    % Normalize (they are probabilities which should sum up to one). In practice
    % they do not sum up to one because of numerical errors when estimating the integrals
    Constants(NdxUnit,:)=Constants(NdxUnit,:)/sum(Constants(NdxUnit,:));
    
end