function [Model,ElapsedTime]= TrainMMED(Samples,NumGroups,NumUnitsPerGroup,NumSteps,PruningThreshold,varargin)
% Training of Mixtures of Multivariate Elliptical Distributions (MMED)
% Inputs:
%   Samples: Matrix [Dimension,NumSamples] with the training data, or a string with the name
%       of the function which will supply the training data 
%   NumGroups: Number of Groups of Mixture Components (GMCs)
%   NumUnitsPerGroup: Number of units (mixture components) per group.
%		If NumUnitsPerGroup==1, then each GMC has only one unit, which is a Gaussian.
%		So we obtain a standard mixture of multivariate Gaussians, but we train it by
%		stochastic approximation.
%   NumSteps: Number of training steps.
%   PruningThreshold: A priori probability threshold to prune units
%   Optionally we can pass in varargin the precomputed constants and the limits (in this order)
% Outputs:
%   Model: The trained MMED model
%   ElapsedTime: Training time (the precomputation of the constants is excluded from this)

% Note:
% Probabilities of the families:
%   Gaussian: Model.Pi(1,:)
%   Exponential square root: Model.Pi(2,:)
%   Multivariate Laplacian: Model.Pi(3,:)
%   Student-t: Model.Pi(4:3:end,:)
%   Uniform: Model.Pi(5:3:end,:)
%   Gamma: Model.Pi(6:3:end,:)

t0 = clock;

if ischar(Samples)
    Result= feval(Samples,0);
    Dimension=Result(1);
    NumSamples=Result(2);
else
    [Dimension, NumSamples] = size(Samples);
end

% Structure initialization

Model.Samples=Samples;
Model.NumGroups=NumGroups;
% Set how many units (mixture components) we have in each GMC. Each
% unit of a GMC has a different shape (Gaussian, Student-t,...)
Model.NumUnitsPerGroup=NumUnitsPerGroup;

% Number of free parameters of the MMED model
Model.NumParameters=NumGroups*Dimension*(0.5*Dimension+1.5)+...
    NumUnitsPerGroup*NumGroups-1;

% We choose some samples for each GMC at random, and we initialize its parameters 
% from them
rand('state',sum(100*clock));
NumPatIni = floor(NumSamples / NumGroups);
fprintf('Initializing MMED')
for NdxGroup=1:NumGroups

    IndexPatIni = round(rand(1,NumPatIni)*(NumSamples-1))+1;
    if ischar(Samples)
        PatIni=feval(Samples,IndexPatIni);
    else
        PatIni = Samples(:,IndexPatIni);
    end
    
    Model.Means{NdxGroup}=mean(PatIni,2);
    Model.C{NdxGroup}=cov(PatIni');
    while rcond(Model.C{NdxGroup})<1e-10
        % The covariance matrix is singular, so we must regularize it
        Model.C{NdxGroup}=Model.C{NdxGroup}+0.1*eye(size(Model.C{NdxGroup}));
    end
    Model.CInv{NdxGroup}=inv(Model.C{NdxGroup});

    fprintf('.')

end



% Equal a priori probabilities
Model.Pi=(1/(Model.NumGroups*Model.NumUnitsPerGroup))*ones(Model.NumUnitsPerGroup,Model.NumGroups);
fprintf('\nThe MMED model is initialized\n')

ElapsedTime=etime(clock,t0);

% Manage precomputed constants and limits
if size(varargin,2)>1
    Constants=varargin{1};
    Limits=varargin{2};
else
    % Precalcular constantes
    fprintf('Precomputing constants...\n')
    % Limits which span a volume proportional to the specified values
    % (we get the limits by raising to the (1/Dimension) power
    % Limit=1 is the hyperellipse of unit Mahalanobis distance
    Limits=[0 0.4 0.8 1.1 1.4 1.8 2.25 2.5 3 4.5 6 10 100];
    Limits=Limits.^(1/Dimension);
    Constants=PrecomputeConstants(Dimension,Limits);
    Constants(find(isnan(Constants)))=0;
    fprintf('Constants are precomputed\n')
end

t0 = clock;

fprintf('Training MMED model...\n')
clear PatIni;
%----------------------------------------------------
% Once the model is initialized, we train it
%----------------------------------------------------
Model=TrainMMEDMEX(Model,NumSteps,PruningThreshold);  
Model=PruneHistMMED(Model,Constants,Limits); 

% Find the number of free parameters after the pruning
NumUnits=sum(Model.Pi(:)>0);
NumGroups=sum(sum(Model.Pi,1)>0);
Model.NumParameters=NumGroups*Dimension*(0.5*Dimension+1.5)+...
    NumUnits-1;

fprintf('Training is finished\n')

ElapsedTime=ElapsedTime+etime(clock,t0);

                
