function [Model]=pcasom(Samples,GridRows,GridCols,...
   NumBasisVectors,LearningRateFunc,TopologyFunc,SampleFunc,NumEpochs,BatchSize,...
   AdaptFunc)
% PCASOM algorithm

% By E. Lopez-Rubio, J. Muoz-Perez and J.A. Gomez-Ruiz
% School of Computer Engineering
% Campus de Teatinos, s/n
% 29071 Malaga. Spain.
% From the paper 'A Principal Components Analysis Self-Organizing Map',
% Neural Networks 17 (2004), 261-270

% Input:
%	Samples = Matrix with samples (one per column)
% 	GridRows = Number of rows of the self-organizing map 
% 	GridCols = Number of columns of the self-organizing map 
%	NumBasisVectors = Number of basis vectors per neuron
%	LearningRateFunc = Given the index of the epoch and the total number of epochs,
%			this function returns the learning rate for that epoch
%	TopologyFunc = Given the index of the epoch, the total number of epochs and the
%			lateral distance from a neuron to the winning neuron, this function
%			returns the topologic neighbourhood coefficient for that neuron. It must
%			hold that feval(TopologyFunc,a,b,0)==1 for all a and b.
%	NumEpochs = Total number of epochs to run
%	AdaptFunc = Network adaptation function
%	BatchSize = Number of epochs to run between two consecutive updates of the basis vectors
% Output:
%	Model.Means=Mean vectors (final state)
%	Model.Covariances=Covariance matrices (final state)





% Compute the dimension of the vector space
[Dimension,NumSamples]=size(Samples);

% Initialize variables
OPTS_EIGS.issym=true;
OPTS_EIGS.isreal=true;
OPTS_EIGS.disp=0;


% Random initialization of the network
Model.Covariances=cell(GridRows,GridCols);
Model.Means=cell(GridRows,GridCols);
Model.GridRows=GridRows;
Model.GridCols=GridCols;
Model.NumBasisVectors=NumBasisVectors;

for ndx1=1:GridRows
   for ndx2=1:GridCols
      Model.Means{ndx1,ndx2}=rand(Dimension,1);
      % We randomly perturb the unit matrix
      RandomMatrix=0.1*rand(Dimension);
      RandomSymmetric=diag(diag(RandomMatrix))+triu(RandomMatrix)+...
         triu(RandomMatrix)'; % Must be symmetric
      MyCorrelation=eye(Dimension)+RandomSymmetric; 
      Model.Covariances{ndx1,ndx2}=MyCorrelation;
      [Vectors Values]=eigs(Model.Covariances{ndx1,ndx2},Model.NumBasisVectors,'LM',OPTS_EIGS);
      Model.BasisVectors{ndx1,ndx2}=Vectors;
   end
end


% Learning
for NdxEpoch=1:NumEpochs
   % Print the epoch if we are in Windows
   if (rem(NdxEpoch,1000)==0) & strcmp(computer,'PCWIN')
      NdxEpoch
   end;
   % Sample generation
   TheSample=feval(SampleFunc,NdxEpoch,Samples);
   % Search of the winning neuron
   [WinningRow,WinningColumn]=PCASOMCompetition(TheSample,Model);
   % Adapt the network to the sample
   Model=feval(AdaptFunc,TheSample,Model,...
      WinningRow,WinningColumn,...
      LearningRateFunc,TopologyFunc,NdxEpoch,NumEpochs);
   % Update the basis vectors
   if rem(NdxEpoch,BatchSize)==0
       for ndx1=1:GridRows
            for ndx2=1:GridCols
                [Vectors Values]=eigs(Model.Covariances{ndx1,ndx2},Model.NumBasisVectors,'LM',OPTS_EIGS);
                Model.BasisVectors{ndx1,ndx2}=Vectors;
                Model.EigenValues{ndx1,ndx2}=diag(Values);
            end
       end
   end
   
      
end

