function [Model]= TrainPPCASOM(Samples,NumRowsMap,NumColsMap,NumVecBasis,NumSteps)
% Inputs
%   Samples: Matrix [Dimension,NumSamples] with the training data, or the
%   name of the function that provides the samples (string)
%   NumRowsMap,NumColsMap: size of the self organizing map
%   NumSteps: Number of steps of the training phase
%   NumVecBasis: number of latent variables


if ischar(Samples)
    Result= feval(Samples,0);
    Dimension=Result(1);
    NumSamples=Result(2);
else
    [Dimension, NumSamples] = size(Samples);
end


Model.Samples=Samples;
Model.NumRowsMap=NumRowsMap;
Model.NumColsMap=NumColsMap;

%   We choose some samples at random to initialize each neuron
rand('state',sum(100*clock));
NumNeurons=NumRowsMap*NumColsMap;
NumPatIni = floor(NumSamples / NumNeurons);
fprintf('Initializing neurons')
for NdxRow=1:NumRowsMap
    for NdxCol=1:NumColsMap
        
        IndicePatIni = round(rand(1,NumPatIni)*(NumSamples-1))+1;
        if ischar(Samples)
            PatIni=feval(Samples,IndicePatIni);
        else
            PatIni = Samples(:,IndicePatIni);
        end
        Model.NumSamples{NdxRow,NdxCol}=size(PatIni,2);
        
        % Mean
        Model.Means{NdxRow,NdxCol} = mean(PatIni')';
        
        % For the matrix W, we do the exact computation only for input
        % dimensionality <5
        if Dimension<5
            % Exact computation
            MyCovariances = zeros(Dimension,Dimension);
            for NdxPat = 1 : NumPatIni
                DevMean = PatIni(:,NdxPat) - Model.Means{NdxRow,NdxCol};
                MyCovariances = MyCovariances + DevMean * DevMean';
            end
            MyCovariances = MyCovariances / NumPatIni;

            [EigenVectors,MatrixEigenValues]=eigs(MyCovariances,NumVecBasis,'LM');
            TraceCovariances=trace(MyCovariances);
            clear MyCovariances;
            EigenValues=diag(MatrixEigenValues);
            [Order,Indices]=sort(-EigenValues);   % Sort in descending order

            % The residual variance Sigma2 is the mean of the Dimension - q
            % smallest eigenvalues
            Model.Sigma2{NdxRow,NdxCol} = ...
               (TraceCovariances-sum(EigenValues)) / (Dimension - NumVecBasis);

            % Matrix W
            U = EigenVectors(:,Indices(1:NumVecBasis));
            KsubQ = diag(-Order(1:NumVecBasis));
            Model.W{NdxRow,NdxCol} = ...
                U * (KsubQ - Model.Sigma2{NdxRow,NdxCol}* eye(NumVecBasis)).^(1/2); 
            Model.Uq{NdxRow,NdxCol}=U;
            Model.Lambdaq{NdxRow,NdxCol}=diag(KsubQ);
            Model.R{NdxRow,NdxCol}=eye(Dimension);
        else
            % Approximate computation (iterative)
            W=PatIni(:,1:NumVecBasis);
            DevTip=std(reshape(W,1,prod(size(W))));
            W=W+0.1*DevTip*(rand(size(W))-0.5); % Add a little noise
            W=W-Model.Means{NdxRow,NdxCol}*ones(1,NumVecBasis);
            
            % Iterations
            for NdxIter=1:20
                Sv=zeros(Dimension,NumVecBasis);
                for NdxSample=1:NumPatIni
                    TnMenosMu=PatIni(:,NdxSample)-Model.Means{NdxRow,NdxCol};
                    for NdxVecBasis=1:NumVecBasis
                        Sv(:,NdxVecBasis)=Sv(:,NdxVecBasis)+...
                            TnMenosMu*(TnMenosMu'*W(:,NdxVecBasis));
                    end        
                end
                Sv=Sv/NumPatIni;
                if NdxIter<20
                    W=orth(Sv);
                else
                    for NdxVecBasis=1:NumVecBasis
                        EigenValues(NdxVecBasis)=norm(Sv(:,NdxVecBasis));
                    end
                end
            end
            Model.W{NdxRow,NdxCol}=W;
            TraceS=0;
            for NdxSample=1:NumPatIni
                TnMenosMu=PatIni(:,NdxSample)-Model.Means{NdxRow,NdxCol};
                TraceS=TraceS+norm(TnMenosMu)^2;
            end
            TraceS=TraceS/NumPatIni;     
            Model.Sigma2{NdxRow,NdxCol}=(TraceS-sum(EigenValues))/(Dimension-NumVecBasis);  
            % Matrices R, Uq, Lambdaq
            [RTras,EigenValuesBasura]=eig(W'*W);
            Model.R{NdxRow,NdxCol}=RTras';
            WInvR=W*inv(RTras');    % W*inv(R)
            % Find the principal directions of the subspace
            for NdxVector=1:NumVecBasis
                Norma=norm(WInvR(:,NdxVector));
                Model.Lambdaq{NdxRow,NdxCol}(NdxVector)=Norma^2+Model.Sigma2{NdxRow,NdxCol};
                Model.Uq{NdxRow,NdxCol}(:,NdxVector)=WInvR(:,NdxVector)/Norma;
            end
        end
            
        
        % Matrices M and inv(M)
        Model.M{NdxRow,NdxCol} = ...
            Model.Sigma2{NdxRow,NdxCol}* eye(NumVecBasis)+...
                Model.W{NdxRow,NdxCol}'*Model.W{NdxRow,NdxCol};
        Model.MInv{NdxRow,NdxCol} = inv(Model.M{NdxRow,NdxCol});
        % Find accumulated sums
        Model.SumDevMeanXN{NdxRow,NdxCol}=zeros(Dimension,NumVecBasis);
        Model.SumMeanXNXNT{NdxRow,NdxCol}=zeros(NumVecBasis);
        for NdxPat = 1 : NumPatIni
            DevMean = PatIni(:,NdxPat) - Model.Means{NdxRow,NdxCol};
            % < x sub n >, equation A.22 of Tipping & Bishop
            MeanXN=Model.MInv{NdxRow,NdxCol}*Model.W{NdxRow,NdxCol}'*DevMean;
            % < x sub n * x sub n ' >, equation A.23 of Tipping & Bishop
            MeanXNXNT=Model.Sigma2{NdxRow,NdxCol}*Model.MInv{NdxRow,NdxCol}+...
                MeanXN*MeanXN';
            % Summation on n: (t sub n - mu) * < x sub n ' >
            Model.SumDevMeanXN{NdxRow,NdxCol}=Model.SumDevMeanXN{NdxRow,NdxCol}+...
                DevMean*MeanXN';
            % Summation on n: < x sub n * x sub n ' >
            Model.SumMeanXNXNT{NdxRow,NdxCol}=...
                Model.SumMeanXNXNT{NdxRow,NdxCol}+...
                MeanXNXNT;
        end
        fprintf('.')
    end
end
Model.NumVictories=zeros(NumRowsMap,NumColsMap);
%   Equal a priori probabilities
Model.Pi=(1/(NumRowsMap*NumColsMap))*ones(NumRowsMap,NumColsMap);
fprintf('\nThe neurons are initialized\n')
fprintf('Training the map\n')
clear PatIni;
%----------------------------------------------------
% Training phase
%----------------------------------------------------

Model = TrainPPCASOMMEX(Model,NumSteps);


fprintf('Training finished\n')
        
                
