#include "mex.h"
#include "Mates.h"
#include "Debugging.h"


#include "gmp.h"
#include "mpfr.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <float.h>
#include <string.h>

/* 
In order to compile this MEX function, type the following at the MATLAB prompt:
32-bit Windows:
mex TrainPPCASOMMEX.c MatesLap.c lapack.a blas.a libf2c.a libmpfr.a libgmp.a Debugging.c
64-bit Windows:
mex LINKFLAGS="$LINKFLAGS /NODEFAULTLIB:LIBCMT" TrainPPCASOMMEX.c MatesLap.c Debugging.c clapack_nowrap.lib BLAS_nowrap.lib libf2c.lib mpir.lib mpfr.lib

Usage:
[Model] = TrainPPCASOMMEX(Model,NumSteps);


*/

void WinnerPPCASOMProbMEX(mxArray* Model,double *ptrSample,double NeighbourhoodRadius,
    int *NdxWinningRow,int *NdxWinningCol,double *LogDensityProb,double *ptrResponsibilities);



void mexFunction(int nlhs, mxArray* plhs[],
                 int nrhs, const mxArray* prhs[])
{  
    int SpaceDimension,NumSamples,NumRowsMap,NumColsMap,NumVecBasis,NdxVecBasis,Index;
    mxArray *Means,*W,*Uq,*Sigma2,*Lambdaq,*MyMean,*MyW,*NumVictories,*M,*MInv;
    mxArray *SumDevMeanXN,*SumMeanXNXNT,*R,*Samples,*Pi;
    unsigned int Seed;
    int RandomNumber;
    double MaxNeighbourhoodRadius,NeighbourhoodRadius,LogDensityProb;
    double LearningRate,OneMinusLearningRate,Trace,NeighbourhoodFunction;
    double *ptrNumVictories,*ptrW,*ptrUqT,*ptrSigma2,*ptrLambdaq,*Pattern,*ptrNumSteps,*ptrR;
    int NdxCol,NdxRow,NdxWinningRow,NdxWinningCol,NumSteps,NdxStep,IndexPat,NdxDatum,ContinueIt;
    double *ptrDevMean,*ptrSamples;
    const int *DimMap;
    const int *DimSamples;
    const int *DimW;
    double *ptrWTDevMean,*ptrMeanXN,*ptrSigma2MInv,*ptrMeanXNMeanXNT,*ptrCoefMeanXNMeanXNT;
    double *ptrDevMeanMeanXNT,*ptrInvSumMeanXNXNT,*ptrWNew;
    double *ptrSumDevMeanXN,*ptrSumMeanXNXNT,*ptrRTras,*ptrInvR;
    double *ptrWNewTDevMean,*ptrWNewTWNew,*ptrMatrixTrace,*ptrWNewT;
    double *ptrUq,*ptrMInv,*ptrM,*ptrMeans,*ptrWT,*ptrResponsibilities,*ptrPi;
    double *ptrMySumDevMeanXN,*ptrMySumMeanXNXNT;
    double ScalarValue,Norm2DevMean,NormWNewMinusW,NormW,MyNorm,MySigma2;
    double CoefNew,CoefOld,AntPi,SumPis;
        

    
    
    /* Get input data */
    ptrNumSteps=mxGetPr(prhs[1]);
    NumSteps=(*ptrNumSteps);
    
     
    
    /* Setup pseudorandom number generator */
    Seed=time(NULL);
    srand(Seed);
    
    /* Create output array */
    plhs[0]=mxDuplicateArray(prhs[0]);
    
    /* Get working variables */
    Means=mxGetField(plhs[0],0,"Means");
    W=mxGetField(plhs[0],0,"W");
    M=mxGetField(plhs[0],0,"M");
    R=mxGetField(plhs[0],0,"R");
    Pi=mxGetField(plhs[0],0,"Pi");
    ptrPi=mxGetPr(Pi);
    SumDevMeanXN=mxGetField(plhs[0],0,"SumDevMeanXN");
    SumMeanXNXNT=mxGetField(plhs[0],0,"SumMeanXNXNT");
    MInv=mxGetField(plhs[0],0,"MInv");
    Uq=mxGetField(plhs[0],0,"Uq");
    Sigma2=mxGetField(plhs[0],0,"Sigma2");
    Lambdaq=mxGetField(plhs[0],0,"Lambdaq");
    DimMap=mxGetDimensions(Sigma2);
    NumRowsMap=DimMap[0];
    NumColsMap=DimMap[1];
    Samples=mxGetField(plhs[0],0,"Samples");
    DimSamples=mxGetDimensions(Samples);
    SpaceDimension=DimSamples[0];
    NumSamples=DimSamples[1];
    ptrSamples=mxGetPr(Samples);
    NumVictories=mxGetField(plhs[0],0,"NumVictories");
    ptrNumVictories=mxGetPr(NumVictories);
    
    
    /* Prepare auxiliary matrices */
    ptrDevMean=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrResponsibilities=mxMalloc(NumRowsMap*NumColsMap*sizeof(double));
    
    
    
    /* MaxNeighbourhoodRadius=mean([NumRowsMap NumColsMap])/4; */
    MaxNeighbourhoodRadius=((double)NumRowsMap+(double)NumColsMap)/8.0;
    

    
    
    /* Main loop */
    for(NdxStep=0;NdxStep<NumSteps;NdxStep++)
    {
        
             
        /* Choose a pattern */
        /*IndexPat = round(rand(1)*(NumSamples-1))+1; */
        RandomNumber=rand();
        
        IndexPat=RandomNumber%NumSamples;
        
        /* Pattern = Samples(:,IndexPat); */
        Pattern=ptrSamples+IndexPat*SpaceDimension;
        
        NeighbourhoodRadius=MaxNeighbourhoodRadius*(1.0-(double)NdxStep/NumSteps);
        
        /* Find the winning distribution */
        WinnerPPCASOMProbMEX(plhs[0],Pattern,NeighbourhoodRadius,
                &NdxWinningRow,&NdxWinningCol,&LogDensityProb,ptrResponsibilities);
        
        
             
        /* Increment Model.NumVictories(NdxWinningRow,NdxWinningCol) */
        ptrNumVictories[NdxWinningRow+NumRowsMap*NdxWinningCol]++;
        
    
        Index=0;
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                /* Get pointers to this neuron */
                MyMean=mxGetCell(Means,Index);
                ptrMeans=mxGetPr(MyMean);
                MyW=mxGetCell(W,Index);
                DimW=mxGetDimensions(MyW);
                NumVecBasis=DimW[1];
                ptrW=mxGetPr(MyW);
                ptrUq=mxGetPr(mxGetCell(Uq,Index));
                ptrLambdaq=mxGetPr(mxGetCell(Lambdaq,Index));
                ptrSigma2=mxGetPr(mxGetCell(Sigma2,Index));
                ptrMInv=mxGetPr(mxGetCell(MInv,Index));
                ptrM=mxGetPr(mxGetCell(M,Index));
                ptrR=mxGetPr(mxGetCell(R,Index));                
                ptrSumDevMeanXN=mxGetPr(mxGetCell(SumDevMeanXN,Index));
                ptrSumMeanXNXNT=mxGetPr(mxGetCell(SumMeanXNXNT,Index));
                
                /* Find the update rate of this neuron from its responsibility */
                
                /* Values of LearningRate larger than 0.8 may hang this function 
                because either CoefOld or CoefNew become 1 for all neurons,
                and this makes the algorithm converge very slowly.
                On the other hand, we always have CoefOld+CoefNew==1.0 */
                /* Linear decay also works, but it is less justifiable from a mathematical
                point of view:
                LearningRate=0.01*(1.0-(double)(NdxStep+1.0)/(NumSteps+1.0));*/
                                    
                LearningRate=1.0/(0.01*(double)NdxStep+100.0);
                OneMinusLearningRate=1.0-LearningRate;
                AntPi=ptrPi[Index];
                ptrPi[Index]=OneMinusLearningRate*ptrPi[Index]+
                                LearningRate*ptrResponsibilities[Index];
                CoefOld=(OneMinusLearningRate*AntPi)/ptrPi[Index];
                CoefNew=(LearningRate*ptrResponsibilities[Index])/ptrPi[Index];
                
           
                
                
                /* Model.Means{NdxRow,NdxCol}=...
                    CoefOld*Model.Means{NdxRow,NdxCol}+...
                     CoefNew*Pattern; */
                for(NdxDatum=0;NdxDatum<SpaceDimension;NdxDatum++)
                {
                    ptrMeans[NdxDatum]=CoefOld*ptrMeans[NdxDatum]+CoefNew*Pattern[NdxDatum];
                }
         
                
                /* DevMean = Pattern - Model.Means{NdxRow,NdxCol}; */
                              
                MatrixDifference(Pattern,ptrMeans,ptrDevMean,SpaceDimension,1);
                
                
                               
                /* Prepare auxiliary matrices for Tipping & Bishop's EM algorithm */
                ptrWTDevMean=mxMalloc(NumVecBasis*1*sizeof(double));
                ptrMeanXN=mxMalloc(NumVecBasis*1*sizeof(double));
                ptrSigma2MInv=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrMeanXNMeanXNT=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrCoefMeanXNMeanXNT=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrDevMeanMeanXNT=mxMalloc(SpaceDimension*NumVecBasis*sizeof(double));
                ptrInvSumMeanXNXNT=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrWNew=mxMalloc(SpaceDimension*NumVecBasis*sizeof(double));
                ptrWNewT=mxMalloc(NumVecBasis*SpaceDimension*sizeof(double));
                ptrWT=mxMalloc(NumVecBasis*SpaceDimension*sizeof(double));
                ptrWNewTDevMean=mxMalloc(NumVecBasis*1*sizeof(double));
                ptrWNewTWNew=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                
                ptrMatrixTrace=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrMySumDevMeanXN=mxMalloc(SpaceDimension*NumVecBasis*sizeof(double));
                ptrMySumMeanXNXNT=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));

                
                /* Tipping & Bishop's EM algorithm */
                ContinueIt=1;
                
               
                MySigma2=(*ptrSigma2);
                while(ContinueIt)
                {
                    /* < x sub n >, equation A.22 from Tipping & Bishop
                    MeanXN=MInv*W'*DevMean; */
                    Traspose(ptrW,ptrWT,SpaceDimension,NumVecBasis);
                    MatrixProduct(ptrWT,ptrDevMean,ptrWTDevMean,NumVecBasis,SpaceDimension,1);
                    MatrixProduct(ptrMInv,ptrWTDevMean,ptrMeanXN,NumVecBasis,NumVecBasis,1);
                    
                    
                    /* < x sub n * x sub n ' >, equation A.23 from Tipping & Bishop
                    MeanXNXNT=Sigma2*MInv+...
                       MeanXN*MeanXN'; */
                    ScalarMatrixProduct(MySigma2,ptrMInv,ptrSigma2MInv,NumVecBasis,NumVecBasis);
                    MatrixProduct(ptrMeanXN,ptrMeanXN,ptrMeanXNMeanXNT,NumVecBasis,1,NumVecBasis);
                    MatrixSum(ptrSigma2MInv,ptrMeanXNMeanXNT,ptrMeanXNMeanXNT,NumVecBasis,NumVecBasis);
                    
                     
                     
                    /* Summation on n: (t sub n - mu) * < x sub n ' >
                    SumDevMeanXN=...
                    CoefOld*Model.SumDevMeanXN{NdxRow,NdxCol}+...
                    CoefNew*DevMean*MeanXN'; */
                    MatrixProduct(ptrDevMean,ptrMeanXN,ptrDevMeanMeanXNT,SpaceDimension,1,NumVecBasis);
                    ScalarMatrixProduct(CoefNew,ptrDevMeanMeanXNT,ptrDevMeanMeanXNT,SpaceDimension,NumVecBasis);
                    ScalarMatrixProduct(CoefOld,ptrSumDevMeanXN,ptrMySumDevMeanXN,SpaceDimension,NumVecBasis);                    
                    MatrixSum(ptrMySumDevMeanXN,ptrDevMeanMeanXNT,ptrMySumDevMeanXN,SpaceDimension,NumVecBasis);

                    
                    /* Summation on n: < x sub n * x sub n ' >
                    SumMeanXNXNT=...
                    CoefOld*Model.SumMeanXNXNT{NdxRow,NdxCol}+...
                    CoefNew*MeanXNXNT; */
                    
                    ScalarMatrixProduct(CoefOld,ptrSumMeanXNXNT,ptrMySumMeanXNXNT,NumVecBasis,NumVecBasis);
                    ScalarMatrixProduct(CoefNew,ptrMeanXNMeanXNT,ptrCoefMeanXNMeanXNT,NumVecBasis,NumVecBasis);
                    MatrixSum(ptrMySumMeanXNXNT,ptrCoefMeanXNMeanXNT,ptrMySumMeanXNXNT,NumVecBasis,NumVecBasis);

                    
                    /* Matrix W, equation A.24 from Tipping & Bishop
                    WNew=SumDevMeanXN*...
                    inv(SumMeanXNXNT); */
                    Inverse(ptrMySumMeanXNXNT,ptrInvSumMeanXNXNT,NumVecBasis);
                    MatrixProduct(ptrMySumDevMeanXN,ptrInvSumMeanXNXNT,ptrWNew,SpaceDimension,NumVecBasis,NumVecBasis);
                    
                    
                    /* Sigma^2 (residual variances), equation A.25 from
                    Tipping & Bishop
                    Sigma2=...
                    CoefOld*Model.Sigma2{NdxRow,NdxCol}+...
                    (CoefNew/Dimension)* ( norm(DevMean)^2-...
                    2*MeanXN'*WNew'*DevMean+...
                    trace(MeanXNXNT*WNew'*WNew)); */
                    Traspose(ptrWNew,ptrWNewT,SpaceDimension,NumVecBasis);
                    MatrixProduct(ptrWNewT,ptrDevMean,ptrWNewTDevMean,NumVecBasis,SpaceDimension,1);
                    MatrixProduct(ptrWNewT,ptrWNew,ptrWNewTWNew,NumVecBasis,SpaceDimension,NumVecBasis);
                    
                    MatrixProduct(ptrMeanXNMeanXNT,ptrWNewTWNew,ptrMatrixTrace,NumVecBasis,NumVecBasis,NumVecBasis);
                    MatrixProduct(ptrMeanXN,ptrWNewTDevMean,&ScalarValue,1,NumVecBasis,1);
                    SquaredNorm(ptrDevMean,&Norm2DevMean,SpaceDimension);
                    Trace=0.0;
                    for(NdxDatum=0;NdxDatum<NumVecBasis;NdxDatum++)
                    {
                        Trace+=ptrMatrixTrace[NdxDatum+NdxDatum*NumVecBasis];
                    }
                    MySigma2=CoefOld*(*ptrSigma2)+(CoefNew/SpaceDimension)*
                         (Norm2DevMean-2.0*ScalarValue+Trace);
                    
                    
                    
                    /*Matrices M and inv(M) 
                     M = ...
                    Sigma2* eye(NumVecBasis)+...
                        WNew'*WNew;  */
                    memcpy((void *)ptrM,(void *)ptrWNewTWNew,NumVecBasis*NumVecBasis*sizeof(double));
                    SumDiagonalConstant(ptrM,MySigma2,NULL,NumVecBasis);
                    
                                  
                    /* MInv = inv(M); */
                    Inverse(ptrM,ptrMInv,NumVecBasis);
                    
                                        
                    /* ContinueIt= (norm(WNew-W)/norm(W))>0.001; */
                    
                    InverseNorm(ptrW,NULL,&NormW,NULL,SpaceDimension,NumVecBasis);
                    MatrixDifference(ptrWNew,ptrW,ptrW,SpaceDimension,NumVecBasis);
                    InverseNorm(ptrW,NULL,&NormWNewMinusW,NULL,SpaceDimension,NumVecBasis);
                    
                    ContinueIt= ((NormWNewMinusW/NormW) > 0.001);
                    
                    
                    /* W=WNew; */
                    memcpy((void *)ptrW,(void *)ptrWNew,SpaceDimension*NumVecBasis*sizeof(double));
                    
                    
                    
                }    
                 
                /* Model.SumDevMeanXN{NdxRow,NdxCol}=SumDevMeanXN;*/
                memcpy(ptrSumDevMeanXN,ptrMySumDevMeanXN,SpaceDimension*NumVecBasis*sizeof(double));
                
                /* Model.SumMeanXNXNT{NdxRow,NdxCol}=SumMeanXNXNT; */
                memcpy(ptrSumMeanXNXNT,ptrMySumMeanXNXNT,NumVecBasis*NumVecBasis*sizeof(double));
                
                /* Model.Sigma2{NdxRow,NdxCol}=Sigma2; */
                (*ptrSigma2)=MySigma2;
                
                
                /* Compute the remaining parameters of the neuron */
                ptrRTras=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                ptrInvR=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
                
                /* [RTras,EigenValues]=eig(W'*W); */
                EigValVec(ptrWNewTWNew,ptrLambdaq,ptrRTras,NumVecBasis);
                
                               
                /* Model.R{NdxRow,NdxCol}=RTras'; */
		        Traspose(ptrRTras,ptrR,NumVecBasis,NumVecBasis);
                
                /* WInvR=W*inv(RTras');    % W*inv(R) */
                InverseNorm(ptrR,ptrInvR,NULL,NULL,NumVecBasis,NumVecBasis);
                MatrixProduct(ptrW,ptrInvR,ptrUq,SpaceDimension,NumVecBasis,NumVecBasis);
                
                
                /* Find the principal directions of the subspace */
                for(NdxDatum=0;NdxDatum<NumVecBasis;NdxDatum++)
                {
                    /* Norm=norm(WInvR(:,NdxVector));*/
                    SquaredNorm(ptrUq+NdxDatum*SpaceDimension,ptrLambdaq+NdxDatum,SpaceDimension);
                    MyNorm=sqrt(ptrLambdaq[NdxDatum]);
                    /* Model.Lambdaq{NdxRow,NdxCol}(NdxVector)=Norm^2+Sigma2;*/
                    ptrLambdaq[NdxDatum]+=(*ptrSigma2);
                    /* Model.Uq{NdxRow,NdxCol}(:,NdxVector)=WInvR(:,NdxVector)/Norm; */
                    ScalarMatrixProduct(1.0/MyNorm,ptrUq+NdxDatum*SpaceDimension,
                         ptrUq+NdxDatum*SpaceDimension,SpaceDimension,1);
                }     
                
                    
                        
                mxFree(ptrRTras);
                mxFree(ptrInvR);
                
                /* Release auxiliary matrices */
                mxFree(ptrWTDevMean);
                mxFree(ptrMeanXN);
                mxFree(ptrSigma2MInv);
                mxFree(ptrMeanXNMeanXNT);
                mxFree(ptrCoefMeanXNMeanXNT);
                mxFree(ptrDevMeanMeanXNT);
                mxFree(ptrInvSumMeanXNXNT);
                mxFree(ptrWNew);
                mxFree(ptrWNewTDevMean);
                mxFree(ptrMatrixTrace);
                mxFree(ptrWNewT);
                mxFree(ptrWNewTWNew);
                mxFree(ptrWT);
                mxFree(ptrMySumDevMeanXN);
                mxFree(ptrMySumMeanXNXNT);

                
                /* Index to iterate through the cells */
                Index++;
            }
        }
           
        
    }        
    
    
    /* ptrPi[Index] is proportional to the a priori probability of i, P(i). 
    Now we must normalize so that the sum is one */
    SumPis=0.0;
    for(Index=0;Index<NumRowsMap*NumColsMap;Index++)
    {
        SumPis+=ptrPi[Index];
    }    
    for(Index=0;Index<NumRowsMap*NumColsMap;Index++)
    {
        ptrPi[Index]/=SumPis;
    }    


    
    /* Release auxiliary matrices */
    mxFree(ptrDevMean);
    mxFree(ptrResponsibilities);
    
}    







/*--------------------------------------------------------------------*/

void WinnerPPCASOMProbMEX(mxArray* Model,double *ptrSample,double NeighbourhoodRadius,
    int *NdxWinningRow,int *NdxWinningCol,double *LogDensityProb,double *ptrResponsibilities)
{   
    double *ptrW,*ptrWT,*ptrWTW,*ptrInvWTW,*ptrWInvWTW,*ptrMatrixDiagonal,*ptrUqT,*ptrSigma2;
    double *ptrLambdaq,*ptrUq;
    const int *DimMap;
    const int *DimSamples;
    const int *DimW;
    int SpaceDimension,NumSamples,NumRowsMap,NumColsMap,NumVecBasis,NdxVecBasis,Index;
    int NdxCol,NdxRow,NdxColDist,NdxRowDist;
    mxArray *Means,*W,*Uq,*Sigma2,*Lambdaq,*MyMean,*MyW,*Samples;
    double *ptrVectorDif,*ptrTn,*ptrzin,*ptrWTVectorDif,*ptrVectorErrRec,*ptrMatrixDiagzin;
    mpfr_t *ptrDensities;
    double Erec2,Ein2,En2,LogDetC,MyLogDensity,SumNeighbourhoodFunctions;
    double TopologicDistance,*NeighbourhoodFunction;
    mpfr_t DensProbDistribution,MySumNeighbourhoodFunctions,aux;
    
    
    mpfr_set_default_prec(100);
    
    /* Get input data */
    Means=mxGetField(Model,0,"Means");
    W=mxGetField(Model,0,"W");
    Uq=mxGetField(Model,0,"Uq");
    Sigma2=mxGetField(Model,0,"Sigma2");
    DimMap=mxGetDimensions(Sigma2);
    NumRowsMap=DimMap[0];
    NumColsMap=DimMap[1];
    Lambdaq=mxGetField(Model,0,"Lambdaq");
    Samples=mxGetField(Model,0,"Samples");
    DimSamples=mxGetDimensions(Samples);
    SpaceDimension=DimSamples[0];
    
    
    
    /* Create auxiliary matrices */
    ptrVectorDif=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrTn=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrzin=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrWTVectorDif=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrVectorErrRec=mxMalloc(SpaceDimension*1*sizeof(double));
    ptrMatrixDiagzin=mxMalloc(SpaceDimension*1*sizeof(double));
    NeighbourhoodFunction=mxMalloc(NumColsMap*NumRowsMap*sizeof(double));
    
    ptrDensities=mxMalloc(NumColsMap*NumRowsMap*sizeof(mpfr_t));
    for(Index=0;Index<NumColsMap*NumRowsMap;Index++)
    {
        mpfr_init(ptrDensities[Index]);        
    }     
    mpfr_init(DensProbDistribution);
    mpfr_init(aux);
    mpfr_init(MySumNeighbourhoodFunctions);

 
    Index=0;
    /* LogDensityProb = -inf; */
    (*LogDensityProb)=-DBL_MAX;
    
    for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
    {
        for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
        {
            /* Get pointers to this neuron */
            MyMean=mxGetCell(Means,Index);
            MyW=mxGetCell(W,Index);
            DimW=mxGetDimensions(MyW);
            NumVecBasis=DimW[1];
            ptrW=mxGetPr(MyW);
            ptrUq=mxGetPr(mxGetCell(Uq,Index));
            ptrLambdaq=mxGetPr(mxGetCell(Lambdaq,Index));
            ptrSigma2=mxGetPr(mxGetCell(Sigma2,Index));
            
            
            ptrWT=mxMalloc(NumVecBasis*SpaceDimension*sizeof(double));   
            Traspose(ptrW,ptrWT,SpaceDimension,NumVecBasis);
            
            ptrWTW=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
            MatrixProduct(ptrWT,ptrW,ptrWTW,NumVecBasis,SpaceDimension,NumVecBasis);
            
            ptrInvWTW=mxMalloc(NumVecBasis*NumVecBasis*sizeof(double));
            Inverse(ptrWTW,ptrInvWTW,NumVecBasis);
            
            ptrWInvWTW=mxMalloc(SpaceDimension*NumVecBasis*sizeof(double));
            MatrixProduct(ptrW,ptrInvWTW,ptrWInvWTW,SpaceDimension,NumVecBasis,NumVecBasis);
            
            ptrUqT=mxMalloc(NumVecBasis*SpaceDimension*sizeof(double));
            Traspose(ptrUq,ptrUqT,SpaceDimension,NumVecBasis);  
            
            ptrMatrixDiagonal=mxCalloc(NumVecBasis*NumVecBasis,sizeof(double));
            
            LogDetC=(double)(SpaceDimension-NumVecBasis)*log(*ptrSigma2);
            
            for(NdxVecBasis=0;NdxVecBasis<NumVecBasis;NdxVecBasis++)
            {
                ptrMatrixDiagonal[NdxVecBasis+NdxVecBasis*NumVecBasis]=1.0/ptrLambdaq[NdxVecBasis];
                
                LogDetC+=log(ptrLambdaq[NdxVecBasis]);
                
            }    
            
            /* VectorDif=Samples(:,NdxSample) - Model.Mu{NdxRow,NdxCol}; */
            MatrixDifference(ptrSample,mxGetPr(MyMean),
                    ptrVectorDif,SpaceDimension,1);
            
            
            /* Tn =  WInvWTW * (W' * VectorDif); */
            
            MatrixProduct(ptrWT,ptrVectorDif,ptrWTVectorDif,NumVecBasis,SpaceDimension,1);
            MatrixProduct(ptrWInvWTW,ptrWTVectorDif,ptrTn,SpaceDimension,NumVecBasis,1);
            
            /* Erec2 = sum((VectorDif - Tn).^2); */
            MatrixDifference(ptrVectorDif,ptrTn,ptrVectorErrRec,SpaceDimension,1);
            SquaredNorm(ptrVectorErrRec,&Erec2,SpaceDimension);
            
                       
            /* zin=Model.Uq{NdxNeuro}'*VectorDif; */
            MatrixProduct(ptrUqT,ptrVectorDif,ptrzin,NumVecBasis,SpaceDimension,1);
            
            /* Ein2=zin'*MatrixDiagonal*zin; */
            MatrixProduct(ptrMatrixDiagonal,ptrzin,ptrMatrixDiagzin,NumVecBasis,NumVecBasis,1);
            MatrixProduct(ptrzin,ptrMatrixDiagzin,&Ein2,1,NumVecBasis,1);
            
            /* En2=Ein2+Erec2/Model.Sigma2{NdxRow,NdxCol}; */
            En2=Ein2+Erec2/(*ptrSigma2);
                        
            /* MyLogDensity=(-0.5*Dimension)*log(2*pi)-0.5*LogDetC-0.5*En2; */
            
            /* Find log(p(t sub n | i) */
            MyLogDensity=-0.91893853320467*SpaceDimension-0.5*LogDetC-0.5*En2;
            
            /* This is p(t sub n | i) */
            /*ptrDensities[Index]=exp(MyLogDensity);*/
            mpfr_set_d(ptrDensities[Index],MyLogDensity,GMP_RNDN);
            mpfr_exp(ptrDensities[Index], ptrDensities[Index], GMP_RNDN); /* Exponential e ^ datum */
            
            
            /* Index to iterate through the cells */
            Index++;
            
            /* Release memory */
            mxFree(ptrWT);
            mxFree(ptrWTW);
            mxFree(ptrInvWTW);
            mxFree(ptrWInvWTW);
            mxFree(ptrUqT);
            mxFree(ptrMatrixDiagonal);
            
        }    
 
    }    
    
    
    /* LogDensityProb = -inf; */
    (*LogDensityProb)=-DBL_MAX;
    
    /* Compute the probability density of each distribution, p(t sub n | q sub nj) */
    
    for(NdxColDist=0;NdxColDist<NumColsMap;NdxColDist++)
    {
        for(NdxRowDist=0;NdxRowDist<NumRowsMap;NdxRowDist++)
        {
            /* For each neuron in the map */
            Index=0;
            /*DensProbDistribution=0.0;*/
            mpfr_set_si(DensProbDistribution,0,GMP_RNDN);
            SumNeighbourhoodFunctions=0.0;
            for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
            {
                for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
                {
            
                /* Manhattan distance:
                TopologicDistance=abs(NdxRow-NdxWinningRow)+abs(NdxCol-NdxWinningCol); */
                TopologicDistance=(double)(abs(NdxRow-NdxRowDist)+abs(NdxCol-NdxColDist));
                
                /* NeighbourhoodFunction=exp(-0.5*((TopologicDistance/NeighbourhoodRadius)^2)); */
                /* This is q sub nji (unnormalized) */
                NeighbourhoodFunction[Index]=exp(-0.5*((TopologicDistance*TopologicDistance)
                                /(NeighbourhoodRadius*NeighbourhoodRadius)));
                
                /* Computing (q sub nji) * p(t sub n | i)
                 with unnormalized q sub nji */
                /*aux=NeighbourhoodFunction*ptrDensities[Index];*/
                mpfr_set_d(aux,NeighbourhoodFunction[Index],GMP_RNDN);
                mpfr_mul(aux,aux,ptrDensities[Index],GMP_RNDN);
                
                /* Computing p(t sub n | q sub nj)
                 with unnormalized q sub nji*/                
                /*DensProbDistribution+=ptrMysResponsibilities[Index];*/
                mpfr_add(DensProbDistribution,DensProbDistribution,
                                aux,GMP_RNDN);
                              
                SumNeighbourhoodFunctions+=NeighbourhoodFunction[Index];
                
                Index++;                
                }    
            }
            
            /* Finding p(t sub n | q sub nj)
             by normalizing the previous value */
            /*DensProbDistribution/=SumNeighbourhoodFunctions;*/
            mpfr_set_d(MySumNeighbourhoodFunctions,SumNeighbourhoodFunctions,GMP_RNDN);
            mpfr_div(DensProbDistribution,DensProbDistribution,MySumNeighbourhoodFunctions,GMP_RNDN);
            
            
            /* See whether this distribution q sub nj yields the highest probability density,
               that is, see whether this is the winning distribution */
            mpfr_log(aux,DensProbDistribution,GMP_RNDN);
            MyLogDensity=mpfr_get_d(aux,GMP_RNDN);
            if (MyLogDensity>(*LogDensityProb))
            {
                (*LogDensityProb)=MyLogDensity;
                (*NdxWinningRow)=NdxRowDist;
                (*NdxWinningCol)=NdxColDist;
                /* Update the neuron responsibilities for the winning distribution */
                Index=0;
                for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
                {
                    for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
                    {
                        ptrResponsibilities[Index]=NeighbourhoodFunction[Index]/SumNeighbourhoodFunctions;
                        Index++;
                    }
                }  
                
            }   
            /* End of the processing of this distribution */    

        }
    }  
    
    
    /* Release memory */
    mpfr_clear(DensProbDistribution);
    mpfr_clear(aux);
    mpfr_clear(MySumNeighbourhoodFunctions);
    for(Index=0;Index<NumColsMap*NumRowsMap;Index++)
    {
        mpfr_clear(ptrDensities[Index]);        
    } 
    
    mxFree(ptrVectorDif);
    mxFree(ptrTn);
    mxFree(ptrzin);
    mxFree(ptrWTVectorDif);
    mxFree(ptrVectorErrRec);
    mxFree(ptrMatrixDiagzin);
    mxFree(ptrDensities);
    mxFree(NeighbourhoodFunction);
    
}    

