#include "mex.h"
#include "Debugging.h"

#include <stdio.h>
#include <math.h>
#include <time.h>
#include <float.h>
#include <string.h>

/* 
In order to compile this MEX function, type the following at the MATLAB prompt:
mex QDSOMANLLMEX.c Debugging.c

Usage:
      
[ANLL,LogLikelihoods] = QDSOMANLLMEX(Model,Samples);

where
LogLikelihoods(n)=log P( Samples(:,n) )
ANLL=-mean(LogLikelihoods)


*/

#ifndef isfinite
bool isfinite(double x) {
    return ((x - x) == 0);
}
#endif

void mexFunction(int nlhs, mxArray* plhs[],
                 int nrhs, const mxArray* prhs[])
{  
  
    
    mxArray *Marginal,*Intersect,*Graph;    
    double *ptrPi,*ptrMyMarginal,*ptrMyIntersect,*ptrMyGraph,*Pattern,*ptrSamples,*ptrLogProbabilities;
    double *ptrTranslation,*ptrRanges,*ptrANLL;
    int Index,Index2,IndexNeuron,NdxCol,NumColsMap,NdxRow,NumRowsMap,NdxDim,NdxDim2,Dimension;
    int Value,Value2,MaxRange,NumValues,NumSamples,NdxSample,NdxVal,RootNode;
    int *MyRanges,*MyTrans;
    double MyLogLikelihood,MyLikelihood;

     
    /* Get input data */
    Marginal=mxGetField(prhs[0],0,"Marginal");
    Intersect=mxGetField(prhs[0],0,"Intersect");
    Graph=mxGetField(prhs[0],0,"Graph");
    ptrPi=mxGetPr(mxGetField(prhs[0],0,"Pi"));
    ptrTranslation=mxGetPr(mxGetField(prhs[0],0,"Translate"));
    ptrRanges=mxGetPr(mxGetField(prhs[0],0,"Ranges"));

    NumRowsMap=mxGetScalar(mxGetField(prhs[0],0,"NumRowsMap"));
    NumColsMap=mxGetScalar(mxGetField(prhs[0],0,"NumColsMap"));
    Dimension=mxGetScalar(mxGetField(prhs[0],0,"Dimension"));
    MaxRange=mxGetScalar(mxGetField(prhs[0],0,"MaxRange"));
    NumValues=mxGetScalar(mxGetField(prhs[0],0,"NumValues"));   
    RootNode=mxGetScalar(mxGetField(prhs[0],0,"RootNode")); 
    
    NumSamples=mxGetN(prhs[1]);
    ptrSamples=mxGetPr(prhs[1]);
    
    /* Create output matrix */
    plhs[0]=mxCreateDoubleMatrix(1,1,mxREAL);
    ptrANLL=mxGetPr(plhs[0]);
    plhs[1]=mxCreateDoubleMatrix(NumSamples,1,mxREAL);
    ptrLogProbabilities=mxGetPr(plhs[1]);         
    
    
    /* Prepare integer versions of the ranges table and the translation table */
    MyRanges=mxMalloc(Dimension*sizeof(int));
    MyTrans=mxMalloc(MaxRange*Dimension*sizeof(int));    
    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
    {
         MyRanges[NdxDim]=(int)ptrRanges[NdxDim];         
         for(NdxVal=0;NdxVal<MyRanges[NdxDim];NdxVal++)
         {
              MyTrans[NdxVal+MaxRange*NdxDim]=(int)(ptrTranslation[NdxVal+MaxRange*NdxDim]-1.0);
         }
    }
       
    /* Compute log likelihood for all the samples */
    (*ptrANLL)=0.0;
    for(NdxSample=0;NdxSample<NumSamples;NdxSample++)
    {
        /* Point to current sample */
        Pattern=ptrSamples+NdxSample*Dimension;
        
        IndexNeuron=0;
        MyLikelihood=0.0;
        
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                /* Get pointers to this neuron */
                ptrMyMarginal=mxGetPr(mxGetCell(Marginal,IndexNeuron));
                ptrMyIntersect=mxGetPr(mxGetCell(Intersect,IndexNeuron));
                ptrMyGraph=mxGetPr(mxGetCell(Graph,IndexNeuron));            
                    
                MyLogLikelihood=0.0;
                if (RootNode)
                {
                    /* Directed graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         
                         /* See whether this is the root node */                         
                         if (NdxDim==RootNode-1)
                         {
                              /* This is the root node */
                              MyLogLikelihood+=log(ptrMyMarginal[Index]);
                         }
                         /* Explore connections with other nodes */
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2]) 
                              {
                                   /* NdxDim is the parent node of NdxDim2 */
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index]));
                                        
                              }
                             
                              if (ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   /* NdxDim2 is the parent node of NdxDim */
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index2]));                                            
                              }                                                                    
                         } 
                                     
                    }
                }
                else
                {
                    /* Undirected graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         MyLogLikelihood+=log(ptrMyMarginal[Index]);
                         
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2] || ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index])
                                        -log(ptrMyMarginal[Index2]));
                                        
                              }
                         }             
                    }
                }
                /* Now we have MyLogLikelihood==log P(x sub n | i) */
                /* We may have MyLogLikelihood==NaN for previously unseen combinations of values,
                  since ptrMyIntersect[Index+NumValues*Index2]==0.0 or  ptrMyMarginal[Index]==0.0 */
                if (isfinite(MyLogLikelihood))
                {
                    MyLikelihood+=ptrPi[IndexNeuron]*exp(MyLogLikelihood);
                }
                
                /* Index to iterate through the cells */
                IndexNeuron++;
         
            }         
        }  
        /* Obtain the log likelihood for this sample, and accumulate for the ANLL computation */
        (*ptrANLL)-=(ptrLogProbabilities[NdxSample]=log(MyLikelihood));        
     }  
         
     /* Normalize ANLL */
     (*ptrANLL)/=(double)NumSamples;
     
     /* Release memory */
     mxFree(MyRanges);
     mxFree(MyTrans);  
}    
