#include "mex.h"
#include "Debugging.h"

#include <stdio.h>
#include <math.h>
#include <time.h>
#include <float.h>
#include <string.h>

/* 
In order to compile this MEX function, type the following at the MATLAB prompt:
mex QDSOMMissingMEX.c Debugging.c

Usage:
      
[ANLL,LogLikelihoods] = QDSOMMissingMEX(Model,Samples,Present);

where
LogLikelihoods(n)=log P( x(n)^miss | x(n)^pres)
ANLL=-mean(LogLikelihoods)

Note that:
x(n)^pres comprises the present components of the n-th test vector x(n), and it is formed
by the components of Samples(:,n) which have Present(m,n)==1

x(n)^miss comprises the missing components of the n-th test vector x(n), and it is formed
by the components of Samples(:,n) which have Present(m,n)==0


*/

#ifndef isfinite
bool isfinite(double x) {
    return ((x - x) == 0);
}
#endif


/* Compute the sum on n of P(x sub n | i), where x sub n is a possible completion of Pattern  */
double *ComputeSumLikelihoodFullSamples(const mxArray *Model,double *Pattern,double *ptrMyPresent);


void mexFunction(int nlhs, mxArray* plhs[],
                 int nrhs, const mxArray* prhs[])
{  
  
    
    mxArray *Marginal,*Intersect,*Graph;    
    double *ptrPi,*ptrMyMarginal,*ptrMyIntersect,*ptrMyGraph,*Pattern,*ptrSamples,*ptrOutputLogLikelihoods,*ptrSumProbabilities;
    double *ptrTranslation,*ptrRanges,*ptrANLL,*ptrPresent,*ptrMyPresent;
    int Index,Index2,IndexNeuron,NdxCol,NumColsMap,NdxRow,NumRowsMap,NdxDim,NdxDim2,Dimension;
    int Value,Value2,MaxRange,NumValues,NumSamples,NdxSample,NdxVal,RootNode;
    int *MyRanges,*MyTrans;
    double MyLogLikelihood,MyLikelihood,Normalizing;

  
    /* Get input data */
    Marginal=mxGetField(prhs[0],0,"Marginal");
    Intersect=mxGetField(prhs[0],0,"Intersect");
    Graph=mxGetField(prhs[0],0,"Graph");
    ptrPi=mxGetPr(mxGetField(prhs[0],0,"Pi"));
    ptrTranslation=mxGetPr(mxGetField(prhs[0],0,"Translate"));
    ptrRanges=mxGetPr(mxGetField(prhs[0],0,"Ranges"));

    NumRowsMap=mxGetScalar(mxGetField(prhs[0],0,"NumRowsMap"));
    NumColsMap=mxGetScalar(mxGetField(prhs[0],0,"NumColsMap"));
    Dimension=mxGetScalar(mxGetField(prhs[0],0,"Dimension"));
    MaxRange=mxGetScalar(mxGetField(prhs[0],0,"MaxRange"));
    NumValues=mxGetScalar(mxGetField(prhs[0],0,"NumValues"));   
    RootNode=mxGetScalar(mxGetField(prhs[0],0,"RootNode")); 
    
    NumSamples=mxGetN(prhs[1]);
    ptrSamples=mxGetPr(prhs[1]);
    ptrPresent=mxGetPr(prhs[2]);
    
    /* Create output matrix */
    plhs[0]=mxCreateDoubleMatrix(1,1,mxREAL);
    ptrANLL=mxGetPr(plhs[0]);
    plhs[1]=mxCreateDoubleMatrix(NumSamples,1,mxREAL);
    ptrOutputLogLikelihoods=mxGetPr(plhs[1]);         
    
    /* Prepare integer versions of the ranges table and the translation table */
    MyRanges=mxMalloc(Dimension*sizeof(int));
    MyTrans=mxMalloc(MaxRange*Dimension*sizeof(int));    
    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
    {
         MyRanges[NdxDim]=(int)ptrRanges[NdxDim];         
         for(NdxVal=0;NdxVal<MyRanges[NdxDim];NdxVal++)
         {
              MyTrans[NdxVal+MaxRange*NdxDim]=(int)(ptrTranslation[NdxVal+MaxRange*NdxDim]-1.0);
         }
    }
    
    /* Compute log likelihood for all the samples */
    (*ptrANLL)=0.0;
    for(NdxSample=0;NdxSample<NumSamples;NdxSample++)
    {
        /* Point to current pattern, Pattern==x(n)^pres */
        Pattern=ptrSamples+NdxSample*Dimension;
        ptrMyPresent=ptrPresent+NdxSample*Dimension;
        
        
        /* Compute the sum on n of P(x sub n | i), where x sub n is a possible completion of Pattern==x(n)^pres  */        
        ptrSumProbabilities=ComputeSumLikelihoodFullSamples(prhs[0],Pattern,ptrMyPresent);
        /* Compute P( i | x(n)^pres ) */
        Normalizing=0.0;
        IndexNeuron=0;
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                 Normalizing+=ptrPi[IndexNeuron]*ptrSumProbabilities[IndexNeuron];
                 IndexNeuron++;
            }
        }    
        IndexNeuron=0;
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                 ptrSumProbabilities[IndexNeuron]=(ptrPi[IndexNeuron]*ptrSumProbabilities[IndexNeuron])/Normalizing;
                 IndexNeuron++;
            }
        } 
        
        IndexNeuron=0;
        MyLikelihood=0.0;
        
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                /* Get pointers to this neuron */
                ptrMyMarginal=mxGetPr(mxGetCell(Marginal,IndexNeuron));
                ptrMyIntersect=mxGetPr(mxGetCell(Intersect,IndexNeuron));
                ptrMyGraph=mxGetPr(mxGetCell(Graph,IndexNeuron));            
                
                MyLogLikelihood=0.0;
                if (RootNode)
                {
                    /* Directed graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         
                         /* See whether this is the root node */                         
                         if (NdxDim==RootNode-1)
                         {
                              /* This is the root node */
                              if (ptrMyPresent[NdxDim]==0.0)
                              {
                                   MyLogLikelihood+=log(ptrMyMarginal[Index]);
                              }
                         }
                         /* Explore connections with other nodes */
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2]) 
                              {
                                   if (ptrMyPresent[NdxDim2]==0.0)
                                   {                              
                                       /* NdxDim is the parent node of NdxDim2 */
                                       MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                            -log(ptrMyMarginal[Index]));
                                   }
                                        
                              }
                              if (ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   if (ptrMyPresent[NdxDim]==0.0)
                                   {
                                        /* NdxDim2 is the parent node of NdxDim */
                                        MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                             -log(ptrMyMarginal[Index2]));                                            
                                   }
                              }                                                                    
                         } 
                                     
                    }
                }
                else
                {
                    /* Undirected graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         if (ptrMyPresent[NdxDim]==0.0)
                         {
                              MyLogLikelihood+=log(ptrMyMarginal[Index]);
                         }                         
                         
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2] || ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   if ((ptrMyPresent[NdxDim]==0.0) && (ptrMyPresent[NdxDim2]==0.0))
                                   {
                                        MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                             -log(ptrMyMarginal[Index])
                                             -log(ptrMyMarginal[Index2]));
                                   }
                              }
                         }             
                    }
                }
                /* Now we have MyLogLikelihood==log P( x(n)^miss | i, x(n)^pres ) */
                /* We may have MyLogLikelihood==NaN for previously unseen combinations of values,
                  since ptrMyIntersect[Index+NumValues*Index2]==0.0 or  ptrMyMarginal[Index]==0.0 */
                if (isfinite(MyLogLikelihood))
                {
                    MyLikelihood+=ptrSumProbabilities[IndexNeuron]*exp(MyLogLikelihood);
                }
                
                /* Index to iterate through the cells */
                IndexNeuron++;
         
            }         
        }  
        /* Obtain the log likelihood for this sample, and accumulate for the ANLL computation */
        (*ptrANLL)-=(ptrOutputLogLikelihoods[NdxSample]=log(MyLikelihood));        
        /* Release memory*/
        mxFree(ptrSumProbabilities);
     }  
     
     /* Normalize ANLL */
     (*ptrANLL)/=(double)NumSamples;
     
     /* Release memory */
     mxFree(MyRanges);
     mxFree(MyTrans);  
}    




/*------------------------------------------*/

/* Compute the sum on n of P(x sub n | i), where x sub n is a possible completion of Pattern */
double *ComputeSumLikelihoodFullSamples(const mxArray *Model,double *MyPattern,double *ptrMyPresent)
{  
  
    
    mxArray *Marginal,*Intersect,*Graph;    
    double *ptrPi,*ptrMyMarginal,*ptrMyIntersect,*ptrMyGraph,*Pattern;
    double *ptrTranslation,*ptrRanges,*ptrANLL;
    int Index,Index2,IndexNeuron,NdxCol,NumColsMap,NdxRow,NumRowsMap,NdxDim,NdxDim2,Dimension;
    int Value,Value2,MaxRange,NumValues,NdxSample,NdxVal,RootNode;
    int *MyRanges,*MyTrans,*MyAbsent;
    double MyLogLikelihood;
    int NumSamples,NumAbsent,Cnt,NdxAbsent;
    double *ptrSumProbabilities;

  
    /* Get input data */
    Marginal=mxGetField(Model,0,"Marginal");
    Intersect=mxGetField(Model,0,"Intersect");
    Graph=mxGetField(Model,0,"Graph");
    ptrPi=mxGetPr(mxGetField(Model,0,"Pi"));
    ptrTranslation=mxGetPr(mxGetField(Model,0,"Translate"));
    ptrRanges=mxGetPr(mxGetField(Model,0,"Ranges"));

    NumRowsMap=mxGetScalar(mxGetField(Model,0,"NumRowsMap"));
    NumColsMap=mxGetScalar(mxGetField(Model,0,"NumColsMap"));
    Dimension=mxGetScalar(mxGetField(Model,0,"Dimension"));
    MaxRange=mxGetScalar(mxGetField(Model,0,"MaxRange"));
    NumValues=mxGetScalar(mxGetField(Model,0,"NumValues"));   
    RootNode=mxGetScalar(mxGetField(Model,0,"RootNode")); 
        
    
    /* Prepare output */     
    ptrSumProbabilities=mxMalloc(NumRowsMap*NumColsMap*sizeof(double));
    memset(ptrSumProbabilities,0,NumRowsMap*NumColsMap*sizeof(double));
    
    /* Prepare integer versions of the ranges table and the translation table */
    MyRanges=mxMalloc(Dimension*sizeof(int));
    MyTrans=mxMalloc(MaxRange*Dimension*sizeof(int));    
    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
    {
         MyRanges[NdxDim]=(int)ptrRanges[NdxDim];         
         for(NdxVal=0;NdxVal<MyRanges[NdxDim];NdxVal++)
         {
              MyTrans[NdxVal+MaxRange*NdxDim]=(int)(ptrTranslation[NdxVal+MaxRange*NdxDim]-1.0);
         }
    }
    
    
    /* Prepare samples */
    MyAbsent=mxMalloc(Dimension*sizeof(int));
    NumAbsent=0;
    NumSamples=1;
    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
    {
         /* If this component is missing, we need more samples */
         if (ptrMyPresent[NdxDim]==0.0)
         {
              MyAbsent[NumAbsent]=NdxDim;
              NumAbsent++;
              NumSamples*=MyRanges[NdxDim];
         }
    }
    Pattern=mxMalloc(Dimension*sizeof(double));
    memcpy(Pattern,MyPattern,Dimension*sizeof(double));

    
    /* Compute log likelihood for all the samples */
    for(NdxSample=0;NdxSample<NumSamples;NdxSample++)
    {
        /* Get current sample */
        Cnt=NdxSample;
        for(NdxAbsent=0;NdxAbsent<NumAbsent;NdxAbsent++)
        {
             Pattern[MyAbsent[NdxAbsent]]=(double)(Cnt%MyRanges[MyAbsent[NdxAbsent]]);
             Cnt/=MyRanges[MyAbsent[NdxAbsent]];
        }                
        IndexNeuron=0;

        
        for(NdxCol=0;NdxCol<NumColsMap;NdxCol++)
        {
            for(NdxRow=0;NdxRow<NumRowsMap;NdxRow++)
            {
                MyLogLikelihood=0.0;
                /* Get pointers to this neuron */
                ptrMyMarginal=mxGetPr(mxGetCell(Marginal,IndexNeuron));
                ptrMyIntersect=mxGetPr(mxGetCell(Intersect,IndexNeuron));
                ptrMyGraph=mxGetPr(mxGetCell(Graph,IndexNeuron));            
                
                if (RootNode)
                {
                    /* Directed graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         
                         /* See whether this is the root node */                         
                         if (NdxDim==RootNode-1)
                         {
                              /* This is the root node */
                              MyLogLikelihood+=log(ptrMyMarginal[Index]);
                         }
                         /* Explore connections with other nodes */
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2]) 
                              {
                                   /* NdxDim is the parent node of NdxDim2 */
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index]));
                                        
                              }
                              if (ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   /* NdxDim2 is the parent node of NdxDim */
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index2]));                                            
                              }                                                                    
                         } 
                                     
                    }
                }
                else
                {
                    /* Undirected graph */
                    for(NdxDim=0;NdxDim<Dimension;NdxDim++)
                    {
                         Value=(int)Pattern[NdxDim];
                         Index=MyTrans[Value+MaxRange*NdxDim];
                         MyLogLikelihood+=log(ptrMyMarginal[Index]);
                         
                         for(NdxDim2=NdxDim+1;NdxDim2<Dimension;NdxDim2++)
                         {
                              Value2=(int)Pattern[NdxDim2];
                              Index2=MyTrans[Value2+MaxRange*NdxDim2];
                              /* Only if it belongs to the Chow-Liu tree of this neuron */
                              if (ptrMyGraph[NdxDim+Dimension*NdxDim2] || ptrMyGraph[NdxDim2+Dimension*NdxDim])
                              {
                                   MyLogLikelihood+=(log(ptrMyIntersect[Index+NumValues*Index2])
                                        -log(ptrMyMarginal[Index])
                                        -log(ptrMyMarginal[Index2]));
                                        
                              }
                         }             
                    }
                }
                /* Now we have MyLogLikelihood==log P(x sub n | i) */
                /* We may have MyLogLikelihood==NaN for previously unseen combinations of values,
                  since ptrMyIntersect[Index+NumValues*Index2]==0.0 or  ptrMyMarginal[Index]==0.0 */
                if (isfinite(MyLogLikelihood))
                {
                    ptrSumProbabilities[IndexNeuron]+=exp(MyLogLikelihood);
                }
                
                /* Index to iterate through the cells */
                IndexNeuron++;
         
            }         
        }  
     }  
     
     
     /* Release memory */
     mxFree(MyRanges);
     mxFree(MyTrans);  
     mxFree(MyAbsent);
     mxFree(Pattern);
     
     return ptrSumProbabilities;
} 
