function [Model]= TrainQDSOM(Samples,Ranges,NumRowsMap,NumColsMap,NumSteps,RootNode)
% Inputs
%   Samples: Matrix [Dimension,NumSamples] with the training data
%   Ranges: Matrix [Dimension,1] which stores the number of possible values
%       in each dimension. The possible values in the i-th dimension are
%       assumed to be 0,...,Ranges(i)-1
%   NumRowsMap,NumColsMap: size of the self organizing map
%   NumSteps: Number of steps of the training phase
%   RootNode: Root node of the Chow-Liu spanning tree. This is a scalar
%        value in the range 1,...,Dimension. If this parameter is absent,
%        it defaults to zero, which means that the tree is undirected
% Output:
%   Model=Trained QDSOM model, with fields:
%       Model.NumValues=The number of different discrete values, i.e.,
%           sum(Ranges)
%       Model.Translate(a,b)=k iff (X_b==a) corresponds to index k in
%           Model.Marginal and Model.Intersect. Note that Model.Translate
%           has size [Model.MaxRange,Model.Dimension]
%       Model.Marginal(i,j}(k)=The marginal probability P( X_b==a | (i,j)),
%           given the unit {i,j} of the map. Note that
%           Model.Translate(a,b)=k
%           The size of Model.Marginal{i,j} is [Model.NumValues].
%       Model.Intersect(i,j}(k,k')=The intersection probability 
%           P( (X_b==a) && (X_d==c) | (i,j)),
%           given the unit {i,j} of the map. Note that
%           Model.Translate(a,b)=k, Model.Translate(c,d)=k'
%           The size of Model.Intersect{i,j} is [Model.NumValues,Model.NumValues]
%       Model.Mutual{i,j}(b,d)=Estimated mutual information between X_b and
%           X_d measured in bits, according to unit {i,j} of the map.
%           The matrix Model.Mutual{i,j} is symmetric with zero diagonal
%           and size [Model.Dimension,Model.Dimension]
%       Model.Graph{i,j}(b,d)= 1 iff X_b is the parent node of X_d in the 
%           Chow-Liu tree corresponding to unit {i,j} of the map; 0
%           otherwise.
%           Please note that Model.Graph{i,j} is a non symmetric matrix
%           with zeros in the main diagonal. It has size
%           [Model.Dimension,Model.Dimension]

[Dimension, NumSamples] = size(Samples);

% Map initialization
Model.Dimension=Dimension;
Model.Samples=Samples;
Model.NumRowsMap=NumRowsMap;
Model.NumColsMap=NumColsMap;
Model.Ranges=Ranges;
Model.MaxRange=max(Ranges);
Model.NumValues=sum(Ranges);
Model.NumVictories=zeros(NumRowsMap,NumColsMap);

if nargin<6
    Model.RootNode=0;
else
    Model.RootNode=RootNode;
end

% Initialize the index translation table
CurrentIndex=1;
for NdxDim=1:Dimension
    for NdxVal=1:Ranges(NdxDim)
        Model.Translate(NdxVal,NdxDim)=CurrentIndex;
        CurrentIndex=CurrentIndex+1;
    end
end


% Initialize the marginals with globally estimated probabilities
for NdxDim=1:Dimension
    for NdxVal=1:Ranges(NdxDim)
        Marginal(Model.Translate(NdxVal,NdxDim))=sum(Samples(NdxDim,:)==(NdxVal-1))/NumSamples;
    end
end
% In case there are some absent attribute values
if numel(Marginal)<Model.NumValues
    Marginal(Model.NumValues)=0;
end
for NdxRow=1:NumRowsMap
    for NdxCol=1:NumColsMap
        Model.Marginal{NdxRow,NdxCol}=Marginal;
    end
end

% Initialize the intersection probabilites with globally estimated
% probabilities, and the mutual information
Mutual=zeros(Dimension);
for NdxDim1=1:Dimension
    for NdxVal1=1:Ranges(NdxDim1)
        for NdxDim2=(NdxDim1+1):Dimension
            for NdxVal2=1:Ranges(NdxDim2)
                % If we assumed independence:
                %Intersect(Model.Translate(NdxVal1,NdxDim1),Model.Translate(NdxVal2,NdxDim2))=...
                %    Marginal(Model.Translate(NdxVal1,NdxDim1))*Marginal(Model.Translate(NdxVal2,NdxDim2)); 
                MyIntersectProb=sum( (Samples(NdxDim1,:)==(NdxVal1-1)) & (Samples(NdxDim2,:)==(NdxVal2-1)) )/NumSamples;
                Intersect(Model.Translate(NdxVal1,NdxDim1),Model.Translate(NdxVal2,NdxDim2))=...
                    MyIntersectProb;
                Intersect(Model.Translate(NdxVal2,NdxDim2),Model.Translate(NdxVal1,NdxDim1))=...
                    MyIntersectProb;
                Logarithm2=log2(MyIntersectProb/...
                    (Marginal(Model.Translate(NdxVal1,NdxDim1))*Marginal(Model.Translate(NdxVal2,NdxDim2))));
                if isfinite(Logarithm2)                    
                    Mutual(NdxDim1,NdxDim2)=Mutual(NdxDim1,NdxDim2)+...
                        MyIntersectProb*Logarithm2;
                    Mutual(NdxDim2,NdxDim1)=Mutual(NdxDim1,NdxDim2);
                end
            end
        end
    end
end
% In case there are some absent attribute values
if (size(Intersect,1)<Model.NumValues) || (size(Intersect,2)<Model.NumValues)
    Intersect(Model.NumValues,Model.NumValues)=0;
end

for NdxRow=1:NumRowsMap
    for NdxCol=1:NumColsMap        
        Model.Intersect{NdxRow,NdxCol}=Intersect;
        Model.Mutual{NdxRow,NdxCol}=Mutual;
    end
end
%Mutual

% Initialize the spanning tree by Chow-Liu method (bioinformatics toolbox
% required for graphminspantree)
Weights=sparse(2*max(Mutual(:))-Mutual);
Weights=Weights-diag(diag(Weights));
if Model.RootNode==0
    % Undirected
    [SpanningTree,pred] = graphminspantree(Weights,1);
else
    % Directed
    [SpanningTree,pred] = graphminspantree(Weights,RootNode);
    
end
Graph=double(full(SpanningTree)>0);
%view(biograph(SpanningTree,[],'ShowArrows','off','ShowWeights','on'))

for NdxRow=1:NumRowsMap
    for NdxCol=1:NumColsMap        
        Model.Graph{NdxRow,NdxCol}=Graph;
    end
end



% Equal a priori probabilities
Model.Pi=(1/(NumRowsMap*NumColsMap))*ones(NumRowsMap,NumColsMap);
fprintf('The neurons are initialized\n')


%----------------------------------------------------
% Training phase
%----------------------------------------------------

fprintf('Training the map\n')

Model=TrainQDSOMMEX(Model,NumSteps);



fprintf('Training finished\n')
        
                
