function [Results]=UKR(OrigImg,varargin)
% Unbiased Kernel Regression
% Reference:
% E. Lpez-Rubio, M.N. Florentn-Nez, Kernel regression based feature extraction for 3D MR image denoising
% Medical Image Analysis. DOI:10.1016/j.media.2011.02.006

t0 = clock;

%% RICIAN NOISE LEVEL ESTIMATION FROM SKEWNESS
disp('Estimating noise...')
for ZSlice=1:size(OrigImg,3)
    SigmaEstimation(ZSlice)=NsigmaestFast(1000*OrigImg(:,:,ZSlice),7)/1000;
end    
Results.EstimatedSigma=median(SigmaEstimation);
Results.Estimated2Sigma2=2*Results.EstimatedSigma^2;

% The maximum likelihood estimator for the Rayleigh distribution of the background
% (to be used only on images with background regions)
%Results.EstimatedSigma=sqrt(mean(OrigImg(find(OrigImg<0.1)).^2)/2);
%Results.Estimated2Sigma2=2*Results.EstimatedSigma^2;

%% MODEL PARAMETERS MANAGEMENT

% If no model parameters are supplied, then select them automatedly
% These parameter selections can be further optimized
if nargin<2
    
    if Results.EstimatedSigma<0.021
        Parameters.FourierSmoothing=3.79597; 
        Parameters.h=0.98827; 
        Parameters.ksize=7;  
        Parameters.ssize=7; 
        Parameters.wsize = 7; 
        Parameters.lambda = 1.63609; 
        Parameters.NonLocalMeansSmoothing = 0.000287814; 
    end

    if (Results.EstimatedSigma>=0.021) && (Results.EstimatedSigma<0.041)
        Parameters.FourierSmoothing=3.90586; 
        Parameters.h=0.60711; 
        Parameters.ksize=7;  
        Parameters.ssize=7; 
        Parameters.wsize = 7; 
        Parameters.lambda = 0.110266; 
        Parameters.NonLocalMeansSmoothing = 0.00077781; 
    end
    
    if (Results.EstimatedSigma>=0.041) && (Results.EstimatedSigma<0.067)
        Parameters.FourierSmoothing=5.17508; 
        Parameters.h=0.79454; 
        Parameters.ksize=7;  
        Parameters.ssize=7; 
        Parameters.wsize = 7; 
        Parameters.lambda = 0.13355; 
        Parameters.NonLocalMeansSmoothing = 0.0013252; 
    end

    if (Results.EstimatedSigma>=0.067) && (Results.EstimatedSigma<0.089)
        Parameters.FourierSmoothing=4.54708; 
        Parameters.h=1.07039; 
        Parameters.ksize=7;  
        Parameters.ssize=7; 
        Parameters.wsize = 7; 
        Parameters.lambda = 0.3008; 
        Parameters.NonLocalMeansSmoothing = 0.0021564; 
    end

    if (Results.EstimatedSigma>=0.089)
        Parameters.FourierSmoothing=4.8871; 
        Parameters.h=0.7803; 
        Parameters.ksize=9;  
        Parameters.ssize=7; 
        Parameters.wsize = 7; 
        Parameters.lambda = 0.0209; 
        Parameters.NonLocalMeansSmoothing = 0.0030891; 
    end
else
    Parameters=varargin{1};    
end

%% KERNEL REGRESSION
% Consider the squared magnitude image
OrigImg=OrigImg.^2;

% Computing features
disp('Median filtering...')
Vmedian = median3D(OrigImg,'replicate');

disp('Fourier filtering...')

[LowPass,zx1c,zx2c,zx3c]=EdgeDetectFourier(double(Vmedian),Parameters.FourierSmoothing);

zc=LowPass;
zx1c=0.5*zx1c/max(abs(zx1c(:)));
zx2c=0.5*zx2c/max(abs(zx2c(:)));
zx3c=0.5*zx3c/max(abs(zx3c(:)));

% Estimating voxel error
EstimatedErrors=OrigImg-double(Vmedian);
EstimatedErrors=EstimatedErrors.^2;


% Fuzzy membership (clean set)
IsClean=1-EstimatedErrors;


Results.zc=zc;
Results.zx1c=zx1c;
Results.zx2c=zx2c;
Results.zx3c=zx3c;

% Compute steering matrices
disp('Computing steering matrices...')

C = SteeringMatrix3D(zx1c, zx2c, zx3c, IsClean, Parameters.wsize, Parameters.lambda);


Results.C=C;

% Apply second order steering kernel regression
disp('Computing second order kernel regression...')

[zs, zx1s, zx2s, zx3s] = SteeringKernelRegression3D(OrigImg, IsClean, Parameters.h, C, Parameters.ksize);

zx1s(find(isnan(zx1s)))=0;
zx2s(find(isnan(zx2s)))=0;
zx3s(find(isnan(zx3s)))=0;

%% Rician corrections to kernel regression

Results.zs=zs;
Results.zx1s=zx1s;
Results.zx2s=zx2s;
Results.zx3s=zx3s;
Results.zc=sqrt(max(0,Results.zc-Results.Estimated2Sigma2));
Results.zs=sqrt(max(0,Results.zs-Results.Estimated2Sigma2));

Results.Vmedian=sqrt(max(0,Vmedian-Results.Estimated2Sigma2));

Results.IsClean=IsClean;




%% Zeroth order kernel regression, gradient is included into the feature vector
disp('Computing zeroth order kernel regression...')

Offset=floor(Parameters.ksize/2);

zkr=ZerothOrderKernelRegression3D(OrigImg,Results.zs,Results.zx1s,Results.zx2s,Results.zx3s,...
    Parameters.NonLocalMeansSmoothing,Parameters.ssize);


Results.zkr=sqrt(max(0,zkr-Results.Estimated2Sigma2));

Results.ElapsedTime=etime(clock,t0);



