﻿-- Pepe Gallardo,   2003

-- Generación de calendarios

import AP.Cgi
import Time


type Día       = Int -- entre 1 y 31 
type Mes       = Int -- entre 1 y 12
type Año       = Int 
type DíaSemana = Int -- entre 0 (Lunes) y 6 (Domingo)

-- Comprueba si un año es bisiesto
esBisiesto :: Año -> Bool
esBisiesto a = if (a `mod` 100 == 0) then (a `mod` 400 == 0) else (a `mod` 4 == 0)

-- Números de días de cada mes del año
totalesMeses :: Año -> [Int]
totalesMeses a = [31, feb, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
  where
    feb = if esBisiesto a then 29 else 28

-- Calcula el día de la semana correspondiente al uno de Enero de un año (0=Lunes, 1=Martes, ...)
unoDeEnero :: Año -> DíaSemana
unoDeEnero a = (365 * x + x `div` 4 - x `div` 100 + x `div` 400) `mod` 7
 where x = a - 1

-- Calcula el día de la semana correspondiente al primero de un mes para un año (0=Lunes, 1=Martes, ...)
primerDía :: Mes -> Año -> DíaSemana
primerDía m a = primerosDías !! (m-1)
 where
   acumMeses    = scanl (+) (unoDeEnero a) (totalesMeses a)
   primerosDías = map (`mod` 7) acumMeses
   

-- Devuelve una lista con todos los días del mes. Rellena los huecos
-- iniciales de la primera semana con números <= 0
listaMes :: Mes -> Año -> [Día]
listaMes m a = [1 - primerDía m a .. totalesMeses a !! (m-1)]


type Celda = Html ()  -- celda de una tabla
type Fila  = Html ()  -- fila de una tabla

-- Formatea una celda para un día
formateaDía :: Día -> Día -> Celda
formateaDía hoy d = withCss dayCSS celda
 where
 celda
  | d <= 0    = hTd (hText "")
  | d == hoy  = withCss todayCSS $ hTd (hShow d)
  | otherwise = hTd (hShow d)


-- Estilos (ver abajo)
tableCSS  = [ pBorder         =: pt 1 |+| vSolid
            , pBorderCollapse =: vCollapse
	    , pFontFamily     =: value "Arial"
	    , pFontSize       =: percent 80
            , pTextAlign      =: vCenter
	    ]

headerCSS = [ pColor           =: vYellow
	    , pBackgroundColor =: vBlue
            , pFontWeight      =: vBold
	    , pBorder          =: pt 1 |+| vSolid
	    ]

				 
dayCSS = [ pBorder   =: pt 1 |+| vSolid
         , pPadding  =: pt 3 
	 ]

todayCSS = [ pColor           =: vBlack
	   , pBackgroundColor =: vAqua
	   , pFontWeight      =: vBold
	   ]


-- Crea una tabla de cierto ancho a partir de una lista de celdas
tablaDe :: Int -> [Celda] -> Html ()
tablaDe ancho = hTable . sequence_ . enFilasDe ancho
  where
    -- Agrupa las celdas en filas de cierta longitud
    enFilasDe :: Int -> [Celda] -> [Fila]
    enFilasDe _ [] = []
    enFilasDe n xs = hTr (sequence_ ys) : enFilasDe n zs
     where
      (ys,zs) = splitAt n xs



calendarioDe :: Día -> Mes -> Año -> Html ()
calendarioDe hoy m a =
 do { withCss [ pFontWeight =: vBold
              , pColor =: vRed
	      ] $ hSpan $ hText (nombreMes m) 
    ; hHr
    ; withCss tableCSS $ tablaDe 7 $
           -- Cabecera: Nombres de los días de la semana
           [ withCss headerCSS $ hTd $ hText $ d | d <- nombresDías ]
           ++  
           -- El calendario
	   [ formateaDía hoy d | d <- listaMes m a ]
    }		        


página :: Html () -> Html () -> Html ()
página título cuerpo = 
  hHtml $
   do { hHead (hTitle título)
      ; hBody $ hCenter $ do { hB título
                             ; hBr
                             ; cuerpo
                             }
      }


-- Calendario para un mes
calendarioDeFecha :: Día -> Mes -> Año -> Html ()
calendarioDeFecha hoy m a = 
  página (hText $ "Calendario de " ++ nombreMes m ++ " de " ++ show a)
         (calendarioDe hoy m a)


-- Calendario para todo un año
calendarioDeAño :: Día -> Mes -> Año -> Html ()
calendarioDeAño hoy mes año = 
  página ( hText $ "Calendario del año " ++ show año )
         ( tablaDe 3 
            [ withCss monthCSS $ hTd $ calendarioDe (if m==mes then hoy else 0) m año 
	    | m <- [1..12] 
	    ]
	 )
 where	
   monthCSS = [ pPadding       =: pt 10
              , pTextAlign     =: vCenter	         
              , pVerticalAlign =: vTop
              ]


	
nombreMes :: Mes -> String	
nombreMes m = nombresMeses !! (m-1)
 where
  nombresMeses =
   [ "Enero"
   , "Febrero"
   , "Marzo"
   , "Abril"
   , "Mayo"
   , "Junio"
   , "Julio"
   , "Agosto"
   , "Septiembre"
   , "Octubre"
   , "Noviembre"
   , "Diciembre"
   ]
 	

nombresDías :: [String]
nombresDías = ["L","M","X","J","V","S","D"]
 


main :: IO ()
main = 
 do { clock <- getClockTime
    ; c     <- toCalendarTime clock
    ; let d = ctDay c
    ; let m = 1 + fromEnum (ctMonth c)
    ; let a = ctYear c
    ; runCgi (tell $ calendarioDeAño d m a)
    }


