﻿--------------------------------------------------------------------
-- Haskell source HTML pretty printer
--
-- Pepe Gallardo, 2003
--------------------------------------------------------------------


module HaskellPretty (haskellPrettyFile, haskellPrettyString) where
	
import AP.Cgi 
import Text.ParserCombinators.Parsec
import HaskellParser
import Monad (when)


-- tokens de Haskell 
data Token = Spaces String
           | Comment String
           | Delimiter String
           | Identifier String
           | Operator String
           | Number String
           | Literal String
           | Other String
           deriving Show


reservedIds =
  [ "case", "class", "data", "default", "deriving", "do"
  , "else", "hiding", "if", "import", "in", "infix", "infixl", "infixr"
  , "instance", "let", "module", "newtype", "of", "primitive", "qualified"
  , "then", "type", "where"
  ]  --, "as", ,  ]

reservedOps = 
  [ "->", "..", ":", "::", "<-", "=", "=>"
  , "@", "\\", "|", "~"
  ] 

predefinedTypes =
  [ "Bool", "Char", "Double", "Either", "FilePath", "Float", "Int", "Integer"
  , "IO", "IOError", "IOResult", "Maybe", "Ordering", "Ratio", "Rational"
  , "ReadS", "ShowS", "String"
  ]

predefinedValues = 
  [ "EQ", "False", "GT", "Just", "Left", "LT"
  , "Nothing", "Right", "True"
  ]

ordElem x [] = False
ordElem x (y:ys)
 | x == y    = True
 | x <  y    = False
 | otherwise = ordElem x ys


token2Html :: Token -> Html ()
token2Html (Spaces str)           = hText str
token2Html (Comment str)          = withAttr [pClass =: value "comment"] $ hSpan $ hText str
token2Html (Delimiter str)        = withAttr [pClass =: value "delimiter"] $ hSpan $ hText str
token2Html (Identifier str)
 | str `ordElem` reservedIds      = withAttr [pClass =: value "reservedId"] $ hSpan $ hText str
 | str `ordElem` predefinedTypes  = withAttr [pClass =: value "predefinedType"] $ hSpan $ hText str
 | str `ordElem` predefinedValues = withAttr [pClass =: value "value"] $ hSpan $ hText str
 | otherwise                      = hText str
token2Html (Operator str)         = withAttr [pClass =: value "reservedOp"] $ hSpan $ hText str
token2Html (Number str)           = withAttr [pClass =: value "number"] $ hSpan $ hText str
token2Html (Literal str)          = withAttr [pClass =: value "literal"] $ hSpan $ hText str
token2Html (Other str)            = hText str

tokens2Html :: [Token] -> Html ()
tokens2Html xs = hPre $ sequence_ [token2Html x | x <- xs ]


haskellPrettyFile :: FilePath -> Html ()
haskellPrettyFile file =
  do { str <- unsafeIO $ readFile file
     ; haskellPrettyAux file str
     }

haskellPrettyString :: String -> Html ()
haskellPrettyString = haskellPrettyAux "" 

haskellPrettyAux :: FilePath -> String -> Html ()
haskellPrettyAux file str =
  do { let eitherV =  runParser parseProgram () file str
     ; either (\parseError -> page file $ hText $ show parseError)
              (\tokens     -> page file $ tokens2Html tokens)
              eitherV
     }



-- Genera una página HTML dados su título y cuerpo
page :: String -> Html () -> Html ()
page title body =
  hHtml $
   do { hHead $ 
         do { hTitle $ hText title
            ; withAttr [ pHRef =: value "haskellStyle.css"
                       , pRel  =: value "stylesheet"
                       , pType =: value "text/css"
                       ] hLink 
            }     
      ; hBody $ withCss bodyCss $ hDiv $ 
       do { when (not.null $ title) $
            withCss [ pFontFamily =: value "Arial" 
                    , pFontSize =: pt 12 
                    , pFontWeight =: vBold
                    ] $ hSpan $ hText $ "Fichero \""++title++"\""
          ; body 
          }
      }
    where     
     bodyCss = [ pFontFamily  =: value "Courier New"
   	       , pFontSize    =: percent 90
	       ]



main = runCgi (tell $ haskellPrettyFile "FormularioSimple.hs")


test file = runCgi (tell $ haskellPrettyFile file)


parseToken :: Parser Token
parseToken = 
 Spaces `fmap` whiteSpace 
 <|>
 Comment `fmap` (oneLineComment <|> multiLineComment) 
 <|>
 (\x -> Delimiter [x]) `fmap` (oneOf "()[]{},;`")
 <|>
 Identifier `fmap` identifier 
 <|>
 Operator `fmap` operator 
 <|>
 (Number . show) `fmap` integer
 <|>
 (Number . show) `fmap` float
 <|>
 (Literal . show) `fmap` charLiteral 
 <|> 
  Literal `fmap` stringLiteral
 <|> 
 (\x -> Other [x]) `fmap` anyChar 

parseProgram :: Parser [Token]
parseProgram = 
 do { tks <- many1 parseToken
    ; eof 
    ; return tks
    }
 


 


