-- #hide

-- Sergio Jimenez Gonzalez  (ult. modificacin: 10/6/2003)
-- Titulacin: Ing. Tcnico en Informtica de Sistemas
-- Proyecto fin de carrera.


-- |Este mdulo define las propiedades y atributos de HTML 4.0 y CSS2
module AP.Cgi.Lib.Attributes(
{- | Los atributos, propiedades y valores han sido definidos conforme a las siguientes reglas:
	
	* Todos los atributos simples comienzan por \'a\' (por ejemplo, 'aMultiple').

	* Todas las propiedades empiezan por \'p\' (por ejemplo, 'pWidth').

	* Todos los valores definidos para las propiedades empiezan por \'v\' (por ejemplo, 'vButton').

	* Los nombres compuestos conservan la primera letra de cada palabra en maysculas (por ejemplo, 'pBorderRight').
-}

-- *** Tipos bsicos
	Attr,
	Property,
	property,		-- property  :: String -> Property
	Value,
	value,			-- value     :: String -> Value
-- *** Clases y funciones relacionadas
	IsAttr(..),
	Plus((|+|)),
-- *** Algunas funciones tiles
	none,			-- none        :: Attr
	simpleAttr,		-- simpleAttr  :: String -> Attr
	complexAttr,		-- complexAttr :: Property -> Value -> Attr
	(=:),			-- (=:)      :: Property -> Value -> Attr
--	listAttr,		-- listAttr    :: [Attr] -> Attr
--	(<+>),			-- (<+>)     :: Attr -> Attr -> Attr
	attrToDoc,		-- attrToDoc :: Attr -> Doc
	cssToDoc,		-- cssToDoc  :: Attr -> Doc
-- *** Algunos atributos tiles para \<DOCTYPE\>
	strictDocAttribute,		-- strictDocAttribute :: Attr,
	transDocAttribute,		-- transDocAttribute :: Attr
	frameSetDocAttribute,		-- frameSetDocAttribute :: Attr
-- *** Atributos simples
	aChecked,aCompact,aDeclare,aDefer,aDisabled,aIsMap,aMultiple,aNoHRef,aNoResize,aNoShade,
	aNoWrap,aReadOnly,aSelected,		  -- (Todos tienen tipo :: Attr)

--(todas las propiedades tienen tipo :: Property)
-- *** Propiedades 
  -- **** Core Attributes
	pClass,pId,pStyle,pTitle,
  -- **** Internationalization Attributes
	pDir,pLang,
  -- **** Common Scripting Events
	pOnClick,pOnDblClick,pOnKeyDown,pOnKeyPress,pOnKeyUp,pOnMouseDown,pOnMouseMove,pOnMouseOut,
	pOnMouseOver,pOnMouseUp,
  -- **** Propiedades normales (No son propiedades tan comunes como las anteriores, dependen de las etiquetas)
	pAbbr,pAccesKey,pAccept,pAcceptCharset,pAction,pAlign,pALink,pAlt,pArchive,pAxis,pBackground,
	pBgColor,pBorder,pCellSpacing,pCellPadding,pChar,pCharoff,pCharset,pCite,pClassId,pClear,
	pCode,pCodeBase,pCodeType,pColor,pCols,pColSpan,pContent,pCoords,pData,pDateTime,pEncType,
	pFace,pFor,pFrame,pFrameBorder,pHeaders,pHeight,pHRef,pHRefLang,pHSpace,pHttpEquiv,pLabel,
	pLanguage,pLink,pLongDesc,pMarginHeight,pMarginWidth,pMaxLength,pMedia,pMethod,pName,pObject,
	pOnBlur,pOnChange,pOnFocus,pOnLoad,pOnReset,pOnSelect,pOnSubmit,pOnUnLoad,pProfile,pPrompt,
	pRel,pRev,pRows,pRowSpan,pRules,pScheme,pScope,pScrolling,pShape,pSize,pSpan,pSrc,pStandBy,
	pStart,pSummary,pTabIndex,pTarget,pText,pType,pUseMap,pVAlign,pValue,pValueType,pVersion,
	pVLink,pVSpace,pWidth,

--   (Todos los valores tienen tipo :: Value)
-- *** Valores para algunas de las propiedades listadas anteriormente
  	vAbove,vAll,vAuto,vBaseline,vBelow,vBorder,vBottom,vBox,vButton,vCenter,vChar,vCheckBox,vCircle,
	vCol,vColGroup,vCols,vData,vDefault,vDisc,vFile,vGet,vGroups,vHidden,vHSides,vImage,vJustify,
	vLeft,vLhs,vLtr,vMiddle,vNo,vNone,vObject,vPassword,vPoly,vPost,vRadio,vRect,vRef,vReset,vRhs,
	vRight,vRow,vRowGroup,vRows,vRtl,vSquare,vSubmit,vText,vTop,vVoid,vVSides,vYes,

-- (Todas las propiedades CSS2 tienen tipo :: Property)
-- *** Propiedades CSS2 
  -- **** Box model
	pMarginTop,pMarginRight,pMarginBottom,pMarginLeft,pMargin,pPaddingTop,pPaddingRight,pPaddingBottom,
	pPaddingLeft,pPadding,pBorderTopWidth,pBorderRightWidth,pBorderBottomWidth,pBorderLeftWidth,pBorderWidth,
	pBorderTopColor,pBorderRightColor,pBorderBottomColor,pBorderLeftColor,pBorderColor,pBorderTopStyle,
	pBorderRightStyle,pBorderBottomStyle,pBorderLeftStyle,pBorderStyle,pBorderTop,pBorderRight,
	pBorderBottom,pBorderLeft,
  -- **** Visual formatting model 
	pDisplay,pPosition,pTop,pRight,pBottom,pLeft,pFloat,pZIndex,pDirection,pUnicodeBidi,
  -- **** Visual formatting model details
	pMinWidth,pMaxWidth,pMinHeight,pMaxHeight,pLineHeight,pVerticalAlign,
  -- **** Visual effects
	pOverflow,pClip,pVisibility,
  -- **** Generated content, automatic numbering and lists
	pQuotes,pCounterReset,pCounterIncrement,pMarkerOffset,pListStyleType,pListStyleImage,pListStylePosition,
	pListStyle,
  -- **** Paged media
	pMarks,pPageBreakBefore,pPageBreakAfter,pPageBreakInside,pPage,pOrphans,pWidows,
  -- **** Colors and backgrounds
	pBackgroundColor,pBackgroundImage,pBackgroundRepeat,pBackgroundAttachment,pBackgroundPosition,
  -- **** Fonts
	pFontFamily,pFontStyle,pFontVariant,pFontWeight,pFontStretch,pFontSize,pFontSizeAdjust,pFont,
  -- **** Text 
	pTextIndent,pTextAlign,pTextDecoration,pTextShadow,pLetterSpacing,pWordSpacing,pTextTransform,
	pWhiteSpace,
  -- **** Tables
	pCaptionSide,pTableLayout,pBorderCollapse,pBorderSpacing,pEmptyCells,pSpeakHeader,
  -- **** User interface 
	pCursor,pOutline,pOutlineWidth,pOutlineStyle,pOutlineColor,
  -- **** Aural style sheets
	pVolume,pSpeak,pPauseBefore,pPauseAfter,pPause,pCueBefore,pCueAfter,pCue,pPlayDuring,pAzimuth,
	pElevation,pSpeechRate,pVoiceFamily,pPitch,pPithRange,pStress,pRichness,pSpeakPunctuation,pSpeakNumeral,

-- *** Valores predeterminados para algunas de las propiedades anteriores (CSS2)
	vAbsolute,vActiveBorder,vActiveCaption,vAlways,vAppWorkspace,vArmenian,vAvoid,vBackground,
	vBehind,vBidiOverride,vBlink,vBlock,vBold,vBolder,vBoth,vButtonFace,vButtonHighlight,
	vButtonShadow,vButtonText,vCapitalize,vCaption,vCaptionText,vCenterLeft,vCenterRight,vCjkIdeoGraphic,
	vCloseQuote,vCode,vCollapse,vCompact,vCondensed,vContinuous,vCrop,vCross,vCrosshair,vDashed,vDecimal,
	vDecimalLeadingZero,vDigits,vDotted,vDouble,vEmbed,vEResize,vExpanded,vExtraCondensed,vFarLeft,
	vFarRight,vFast,vFaster,vFixed,vGeorgian,vGreyText,vGroove,vHebrew,vHelp,vHide,vHigh,vHigher,
	vHighlight,vHighlightText,vHiragana,vHiraganaIroha,vIcon,vInactiveBorder,vInactiveCaption,vInactiveCaptionText,
	vInfoBackground,vInfoText,vInherit,vInLine,vInLineTable,vInset,vInSide,vItalic,vKatakana,vKatakanaIroha,
	vLandscape,vLarge,vLarger,vLeftSide,vLeftwards,vLevel,vLight,vLighter,vLineTrough,vListItem,vLoud,vLow,
	vLower,vLowerAlpha,vLowerFase,vLowerGreek,vLowerLatin,vLowerRoman,vMarker,vMedium,vMessageBox,vMenu,
	vMenuText,vMix,vMove,vNarrower,vNResize,vNeResize,vNwResize,vNoCloseQuote,vNoOpenQuote,vNoRepeat,vNormal,
	vNoWrap,vOblique,vOnce,vOpenQuote,vOverLine,vOutset,vOutside,vPointer,vPortrait,vPre,vRelative,vRepeat,
	vRepeatx,vRepeaty,vRidge,vRightSide,vRightwards,vRunIn,vScroll,vScrollBar,vSemiCondensed,vSemiExpanded,
	vSeparate,vShow,vSilent,vSlow,vSlower,vSmall,vSmaller,vSmallCaps,vSamllCaption,vSoft,vSolid,vSpellOut,
	vSResize,vSeResize,vSwResize,vStatic,vStatusBar,vSub,vSuper,vTable,vTableCaption,vTableCell,vTableColumn,
	vTableColumnGroup,vTableFooterGroup,vTableHeaderGroup,vTableRow,vTableRowGroup,vTextBottom,vTextTop,vThick,
	vThin,vThreeDDarkShadow,vThreeDFace,vThreeDHighlight,vThreeDLightShadow,vThreeDShadow,vTransparent,vUltraCondensed,
	vUltraExpanded,vUnderLine,vUpperAlpha,vUpperCase,vUpperLatin,vUpperRoman,vVisible,vWait,vWider,vWindow,
	vWindowFrame,vWindowText,vWResize,vXFast,vXHigh,vXLarge,vXLoud,vXLow,vXSoft,vXSlow,vXXLarge,vXSmall,vXXSmall,
-- *** VARIOS
  -- **** Unidades de longitud
    -- ***** Relativos
  	em,ex,px,
    -- ***** Absolutos
	inch,cm,mm,pt,pc,	-- :: Num a => a -> Value
  -- **** Porcentajes
  	percent,
  -- **** Construccin de colores
  	rgb, 				-- rgb  :: Int -> Int -> Int -> Value
	rgbP,				-- rgbP :: Float -> Float -> Float -> Value
  -- **** Construccin de URI's
	url, 				-- url  :: String -> Value
  -- **** Grados
	deg,grad,rad,			-- :: Num a => a -> Value
  -- **** Tiempos
	ms,s,				-- :: Num a => a -> Value
  -- **** Frecuencias
	hz,kHz,				-- :: Num a => a -> Value
-- *** Valores para algunos colores (nombres pertenecientes a la paleta VGA de Windows)
	vAqua,vBlack,vBlue,vFuchsia,vGray,vGreen,vLime,vMaroon,vNavy,vOlive,vPurple,vRed,vSilver,vTeal,
	vWhite,vYellow
)where

import Char
import List

import AP.Cgi.Lib.Doc
--------------------------------------------------------------------------------------------------------
-- Tipos bsicos para los atributos

{- | Este tipo representa a un atributo HTML 4.0 o un atributo de estilo (CSS 2). La nica diferencia entre ambos
 es la forma de visualizarlos. Hay varios tipos de atributos y disponemos de funciones especficas para crear
 cada uno de ellos.

 	* Atributos simples
	
	* Atributos compuestos (de la forma \"Propiedad = Valor\")

	* Secuencia de atributos
-}
data Attr = None			-- Sin atributos
	  | SAttr String		-- Atributo simple (booleano... tal como MULTIPLE en las etiquetas <select>)
	  | CAttr Property Value	-- Atributo Compuesto
	  | LAttr [Attr]		-- Lista de atributos

-- | Esta funcin se puede utilizar cuando no queremos aadir ningn atributo y necesitamos algn argumento de tipo 'Attr'.
none :: Attr
none = None

{- | Esta funcin crea un atributo simple. Son aquellos atributos que se representan slo con una palabra clave.
	Ejemplo:

	@
		aMultiple :: 'Attr'
		aMultiple = simpleAttr \"MULTIPLE\"
	@
-}
simpleAttr :: String -> Attr
simpleAttr = SAttr

{- | Esta funcin crea un atributo compuesto, que son los ms usuales. Sigue una sintaxis de \"Propiedad = Valor\", por lo
 que necesitamos proporcionarle un argumento de tipo 'Property' y un argumento de tipo 'Value'.
 
	Ejemplo:

	@
		atributo :: 'Attr'
		atributo = complexAttr 'pBgColor' 'vBlue'
	@
 
 Alternativamente podemos utilizar el siguiente operador para crear atributos a partir de una propiedad y un valor.

-}
complexAttr :: Property -> Value -> Attr
complexAttr = CAttr

-- Podemos crear un atributo a partir de una lista de ellos gracias a esta funcin (se secuenciarn todos en la etiqueta
-- donde se aada).
listAttr :: [Attr] -> Attr
listAttr = LAttr

-- Propiedades

-- | Este tipo permite representar una propiedad para un atributo de una etiqueta (tanto propiedades HTML 4.0 como CSS2). 
-- Para crear propiedades podemos utilizar la funcin 'property'.
data Property = Property String

-- Valores, para poder asginar valores directamente, sin necesidad de cadenas de caracteres

{- | Este tipo nos permite representar los valores que se asignarn a las propiedades. Para ello, podemos utilizar la
 funcin 'value' para crear un valor a partir de una cadena de caracteres, o utilizar numeros gracias a las instancias 
 definidas. 
 Algunos propiedades CSS2 pueden tener valores mltiples, por lo que disponemos de un operador especial que se encarga
 de esta tarea '|+|'.
-}
data Value = Value String       -- valores cadena
           | ValueInt Int       -- valores enteros
	   | ValueFloat Float   -- valores reales
	   | ValueL [Value]     -- lista de valores
	   deriving (Eq,Show,Read)

--------------------------------------------------------------------------------------------------------
-- instancias de Show para los tipos de arriba
instance Show Property where
  show (Property s) = s

-- La siguiente funcin genera el tipo Doc asociado a un valor (Para la representacin de los atributos)
showV :: Value -> Doc 
showV (Value s)        = texto s
showV (ValueInt n)     = texto $ show n
showV (ValueFloat f)   = texto $ show f
showV (ValueL (x:xs) ) = (showV x) <> (foldl (\y ys -> y <> texto " " <> showV ys) vacio xs)

-- (No hacemos instancia para el tipo atributo porque su representacin depende si son atributos normales o de estilo)
--------------------------------------------------------------------------------------------------------
{- | Esta funcin permite crear un dato de tipo 'Property' a partir de una cadena de caracteres.
 	
	Ejemplo:

	@
		pBackground :: 'Property'
		pBackground = property \"background\"
	@
-}
property :: String -> Property
property = Property

{- | Esta funcion permite crear valores para una propiedad.
 	Ejemplo:
	
	@
		vBorder :: 'Value'
		vBorder = value \"border\"
	@
-}
value :: String -> Value
value = Value

-- Para poder usar enteros como Values
instance Num Value where
--  fromInt                 = ValueInt
  fromInteger             = ValueInt . fromInteger
  ValueInt x + ValueInt y = ValueInt (x+y)
  ValueInt x - ValueInt y = ValueInt (x-y)
  ValueInt x * ValueInt y = ValueInt (x*y)
  negate (ValueInt x)     = ValueInt (negate x)
  abs (ValueInt x)        = ValueInt (abs x)
  signum (ValueInt x)     = ValueInt (signum x)

-- Para poder usar reales como Values
instance Fractional Value where
  fromRational = ValueFloat . fromRational
  recip (ValueFloat x) = ValueFloat (recip x)
--------------------------------------------------------------------------------------------------------
infixl 3 |+|
class Plus a where
 (|+|) :: a -> a -> a

{- ^ Esta clase nos permite secuenciar datos del mismo tipo, a partir del operador |+|.
    El tipo 'Value' pertenece a esta clase.

	Ejemplo:

	@
		atributo 'Attr'
		atributo = 'pBorder' =: 'px' 5 |+| 'vSolid' |+| 'vYellow'
	@
-} 

-- para poder unir varios valores en CSS
instance Plus Value where
  v1 |+| v2 = ValueL [v1,v2]
--------------------------------------------------------------------------------------------------------
-- Para asignar un valor a un atributo
infix 2 =:
(=:) :: Property -> Value -> Attr

{- ^ Este operador se comporta igual que la funcin 'complexAttr'. Construye un atributo a partir de una propiedad
 y un valor (pero su uso es ms cmodo e intuitivo, puesto que se trata de un operador infijo).

 	Ejemplo:
	
	@
		atributo :: 'Attr'
		atributo = 'pBgColor' =: 'vBlue'
	@

-}
p =: v = CAttr p v



-- Creamos una clase para poder usar un attr o una lista de ellos
infixl 1 <+>

-- | Esta clase define varias funciones para obtener un dato de tipo 'Attr' a partir de otro instanciado. De esta forma, 
-- los tipos instanciados en esta clase pueden ser utilizados en las funciones de adicin de atributos 
-- ('HTML.withAttr', 'HTML.withCss').
class IsAttr a where
  toAttr     :: a -> Attr 	
  
  -- ^ Obtiene un dato de tipo 'Attr' a partir del tipo instanciado.

  listToAttr :: [a] -> Attr
  
  -- ^ Obtiene un dato de tipo 'Attr' a partir de una lista de valores del tipo instanciado.
  
  {- | Este operador permite secuenciar varios atributos, a partir del tipo instanciado.

  	Ejemplo:

	@
		atributos :: 'Attr'
		atributos = 'pColor' =: 'vBlue' \<+\> 'pSize' =: 'px' 10
	@
  -} 
  (<+>)      :: a -> a -> a
  

instance IsAttr Attr where
  toAttr     = id
  listToAttr = LAttr 
  
  v1   <+> None = v1		-- Si alguno de los dos es "None", nos quedamos con el otro
  None <+> v2   = v2
  v1   <+> v2   = LAttr [v1,v2]
  

instance (IsAttr a) => IsAttr [a] where
  toAttr     = listToAttr
  listToAttr = listToAttr . concat
  xs <+> ys = xs ++ ys 

--------------------------------------------------------------------------------------------------------

-- attrToDoc slo hace la conversin de los atributos a tipo Doc. Los atributos simples se representan 
-- tal cual (sin modificacin alguna), los pares "a" =:"v" se pasan a la forma A="v" (ATRIBUTO="valor")

attrToDoc :: Attr -> Doc	-- Lo comento para que no salga en la ayuda
attrToDoc None 		     = vacio
attrToDoc (SAttr s)          = texto " " <> texto s
attrToDoc (CAttr prop valor) = texto " " <> texto (fmap toUpper $ show prop) <> texto "=\'" <> showV valor <> texto "\'"
attrToDoc (LAttr as)         = foldr (<>) vacio (fmap attrToDoc as)

-- cssToDoc tiene el mismo cometido que "attrToDoc", pero siguiendo la sintaxis de hojas de estilo

cssToDoc :: Attr -> Doc	-- Lo comento para que no salga en la ayuda
cssToDoc None      = vacio
cssToDoc (SAttr s) = vacio  -- En atributos de estilo no deberia haber ningun atributo simple, lo ignoramos
cssToDoc at        = texto " STYLE=\'" <> cssToDoc' at <> texto "\'"
  where
    cssToDoc' :: Attr -> Doc
    cssToDoc' None 		 = vacio
    cssToDoc' (SAttr s)          = vacio  -- En atributos de estilo no deberia haber ningun atributo simple, lo ignoramos
    cssToDoc' (CAttr prop valor) = texto (show prop) <> texto ": " <> showV valor
    cssToDoc' (LAttr as)         = foldr (<>) vacio . intersperse (texto "; ") . fmap cssToDoc' $ as

--------------------------------------------------------------------------------------------------------

-- Algunos atributos tiles para <DOCTYPE>
-- | Este atributo es especfico para pginas HTML \"strict\".
strictDocAttribute :: Attr
strictDocAttribute = simpleAttr "HTML" <+> simpleAttr "PUBLIC" <+> simpleAttr "\"-//W3C//DTD HTML 4.0//EN\""
		   -- <+> simpleAttr "\"http://www.w3.org/TR/REC-html40/strict.dtd\""

-- | Este atributo es especfico para pginas HTML \"transitional\".
transDocAttribute :: Attr
transDocAttribute = simpleAttr "HTML" <+> simpleAttr "PUBLIC" <+> simpleAttr "\"-//W3C//DTD HTML 4.0 Transitional//EN\""
		  -- <+> simpleAttr "\"http://www.w3.org/TR/REC-html40/loose.dtd\""

-- | Este atributo es especfico para pginas HTML con marcos (\"frames\"). 
frameSetDocAttribute :: Attr
frameSetDocAttribute = simpleAttr "HTML" <+> simpleAttr "PUBLIC" <+> simpleAttr "\"-//W3C//DTD HTML 4.0 Frameset//EN\""
		     -- <+> simpleAttr "\"http://www.w3.org/TR/REC-html40/frameset.dtd\""
--------------------------------------------------------------------------------------------------------
-- Para las dems definiciones, la nomenclatura seguida es:
--       aXXXX -> para los atributos, donde XXXX es el nombre del atributo.	ej: aCompact => atributo COMPACT
--       pXXXX -> para las propiedades, donde XXXX es el nombre de la propiedad.ej: pClass   => propiedad CLASS
--  Se suprimen los guiones. ej: HTTP-EQUIV => pHttpEquiv
--------------------------------------------------------------------------------------------------------
-- Atributos simples  (Todos tienen tipo :: Attr)
--------------------------------------------------------------------------------------------------------
aChecked  :: Attr
aChecked  = simpleAttr "CHECKED"
aCompact  :: Attr
aCompact  = simpleAttr "COMPACT"
aDeclare  :: Attr
aDeclare  = simpleAttr "DECLARE"
aDefer    :: Attr
aDefer    = simpleAttr "DEFER"
aDisabled :: Attr
aDisabled = simpleAttr "DISABLED"
aIsMap    :: Attr
aIsMap    = simpleAttr "ISMAP"
aMultiple :: Attr
aMultiple = simpleAttr "MULTIPLE"
aNoHRef   :: Attr
aNoHRef   = simpleAttr "NOHREF"
aNoResize :: Attr
aNoResize = simpleAttr "NORESIZE"
aNoShade  :: Attr
aNoShade  = simpleAttr "NOSHADE"
aNoWrap   :: Attr
aNoWrap   = simpleAttr "NOWRAP"
aReadOnly :: Attr
aReadOnly = simpleAttr "READONLY"
aSelected :: Attr
aSelected = simpleAttr "SELECTED"

-- Los posibles valores que se pueden asignar a estas propiedades para formar los atributos compuestos pueden variar
-- dependiendo de la etiqueta donde se usen. Para ello ver la especificacin de HTML.

-- A continuacin se listan propiedades de atributos comunes a la mayoria de los tags HTML 4.0
-- (Todos tienen tipo :: Property)

-- Core Attributes
pClass :: Property
pClass = property "class"
pId    :: Property
pId    = property "id"
pStyle :: Property
pStyle = property "style"
pTitle :: Property
pTitle = property "title"

-- Internationalization Attributes
pDir  :: Property
pDir  = property "dir"
pLang :: Property
pLang = property "lang"

-- Common Scripting Events
pOnClick     :: Property
pOnClick     = property "onclick"
pOnDblClick  :: Property
pOnDblClick  = property "ondblclick"
pOnKeyDown   :: Property
pOnKeyDown   = property "onkeydown"
pOnKeyPress  :: Property
pOnKeyPress  = property "onkeypress"
pOnKeyUp     :: Property
pOnKeyUp     = property "onkeyup"
pOnMouseDown :: Property
pOnMouseDown = property "onmousedown"
pOnMouseMove :: Property
pOnMouseMove = property "onmousemove"
pOnMouseOut  :: Property
pOnMouseOut  = property "onmouseout"
pOnMouseOver :: Property
pOnMouseOver = property "onmouseover"
pOnMouseUp   :: Property
pOnMouseUp   = property "onmouseup"

------------------------------
-- Propiedades normales (No son propiedades comunes)
pAbbr          :: Property
pAbbr          = property "abbr" 
pAccesKey      :: Property
pAccesKey      = property "acceskey"
pAccept        :: Property
pAccept        = property "accept"
pAcceptCharset :: Property
pAcceptCharset = property "accept-charset"
pAction        :: Property
pAction        = property "action"
pAlign         :: Property
pAlign         = property "align"
pALink         :: Property
pALink         = property "alink"
pAlt           :: Property
pAlt           = property "alt"
pArchive       :: Property
pArchive       = property "archive"
pAxis          :: Property
pAxis          = property "axis"
pBackground    :: Property
pBackground    = property "background"
pBgColor       :: Property
pBgColor       = property "bgcolor"
pBorder	       :: Property
pBorder	       = property "border"
pCellSpacing   :: Property
pCellSpacing   = property "cellspacing"
pCellPadding   :: Property
pCellPadding   = property "cellpadding"
pChar          :: Property
pChar          = property "char"
pCharoff       :: Property
pCharoff       = property "charoff"
pCharset       :: Property
pCharset       = property "charset"
pCite          :: Property
pCite          = property "cite"
pClassId       :: Property
pClassId       = property "classid"
pClear	       :: Property
pClear	       = property "clear"
pCode          :: Property
pCode          = property "code"
pCodeBase      :: Property
pCodeBase      = property "codebase"
pCodeType      :: Property
pCodeType      = property "codetype"
pColor         :: Property
pColor         = property "color"
pCols          :: Property
pCols          = property "cols"
pColSpan       :: Property
pColSpan       = property "colspan"
pContent       :: Property
pContent       = property "content"
pCoords        :: Property
pCoords        = property "coords"
pData          :: Property
pData          = property "data"
pDateTime      :: Property
pDateTime      = property "datetime"
pEncType       :: Property
pEncType       = property "enctype"
pFace          :: Property
pFace          = property "face"
pFor           :: Property
pFor           = property "for"
pFrame         :: Property
pFrame         = property "frame"
pFrameBorder   :: Property
pFrameBorder   = property "frameborder"
pHeaders       :: Property
pHeaders       = property "headers"
pHeight        :: Property
pHeight        = property "height"
pHRef          :: Property
pHRef          = property "href"
pHRefLang      :: Property
pHRefLang      = property "hreflang"
pHSpace        :: Property
pHSpace        = property "hspace"
pHttpEquiv     :: Property
pHttpEquiv     = property "http-equiv"
pLabel         :: Property
pLabel         = property "label"
pLanguage      :: Property
pLanguage      = property "language"
pLink          :: Property
pLink          = property "link"
pLongDesc      :: Property
pLongDesc      = property "longdesc"
pMarginHeight  :: Property
pMarginHeight  = property "marginheight"
pMarginWidth   :: Property
pMarginWidth   = property "marginwidth"
pMaxLength     :: Property
pMaxLength     = property "maxlength"
pMedia         :: Property
pMedia         = property "media"
pMethod        :: Property
pMethod        = property "method"
pName          :: Property
pName          = property "name"
pObject        :: Property
pObject        = property "object"
pOnBlur        :: Property
pOnBlur        = property "onblur"
pOnChange      :: Property
pOnChange      = property "obchange"
pOnFocus       :: Property
pOnFocus       = property "onfocus"
pOnLoad        :: Property
pOnLoad        = property "onload"
pOnReset       :: Property
pOnReset       = property "onreset"
pOnSelect      :: Property
pOnSelect      = property "onselect"
pOnSubmit      :: Property
pOnSubmit      = property "onsubmit"
pOnUnLoad      :: Property
pOnUnLoad      = property "onunload"
pProfile       :: Property
pProfile       = property "profile"
pPrompt        :: Property
pPrompt        = property "prompt"
pRel           :: Property
pRel           = property "rel"
pRev           :: Property
pRev           = property "rev"
pRows          :: Property
pRows          = property "rows"
pRowSpan       :: Property
pRowSpan       = property "rowspan"
pRules         :: Property
pRules         = property "rules"
pScheme        :: Property
pScheme        = property "scheme"
pScope         :: Property
pScope         = property "scope"
pScrolling     :: Property
pScrolling     = property "scrolling"
pShape         :: Property
pShape         = property "shape"
pSize          :: Property
pSize          = property "size"
pSpan          :: Property
pSpan          = property "span"
pSrc           :: Property
pSrc           = property "src"
pStandBy       :: Property
pStandBy       = property "standby"
pStart         :: Property
pStart         = property "start"
pSummary       :: Property
pSummary       = property "summary"
pTabIndex      :: Property
pTabIndex      = property "tabindex"
pTarget        :: Property
pTarget        = property "target"
pText          :: Property
pText          = property "text"
pType          :: Property
pType          = property "type"
pUseMap        :: Property
pUseMap        = property "usemap"
pVAlign        :: Property
pVAlign        = property "valign"
pValue         :: Property
pValue         = property "value"
pValueType     :: Property
pValueType     = property "valuetype"
pVersion       :: Property
pVersion       = property "version"
pVLink         :: Property
pVLink         = property "vlink"
pVSpace        :: Property
pVSpace        = property "vspace"
pWidth         :: Property
pWidth         = property "width"

--------------------------------------------------------------------------------------------------------
-- Valores para algunas de las propiedades listadas arriba (Valores para propiedades de etiquetas HTML 4.0)
vAbove    :: Value
vAbove    = value "above"
vAll      :: Value
vAll      = value "all"
vAuto     :: Value
vAuto     = value "auto"
vBaseline :: Value
vBaseline = value "baseline"
vBelow    :: Value
vBelow    = value "below"
vBorder   :: Value
vBorder   = value "border"
vBottom   :: Value
vBottom   = value "bottom"
vBox      :: Value
vBox      = value "box"
vButton   :: Value
vButton   = value "button"
vCenter   :: Value
vCenter   = value "center"
vChar     :: Value
vChar     = value "char"
vCheckBox :: Value
vCheckBox = value "checkbox"
vCircle   :: Value
vCircle   = value "circle"
vCol      :: Value
vCol      = value "col"
vColGroup :: Value
vColGroup = value "colgroup"
vCols     :: Value
vCols     = value "cols"
vData     :: Value
vData     = value "data"
vDefault  :: Value
vDefault  = value "default"
vDisc     :: Value
vDisc     = value "disc"
vFile     :: Value
vFile     = value "file"
vGet      :: Value
vGet      = value "get"
vGroups   :: Value
vGroups   = value "groups"
vHidden   :: Value
vHidden   = value "hidden"
vHSides   :: Value
vHSides   = value "hsides"
vImage    :: Value
vImage    = value "image"
vJustify  :: Value
vJustify  = value "justify"
vLeft     :: Value
vLeft     = value "left"
vLhs      :: Value
vLhs      = value "lhs"
vLtr      :: Value
vLtr      = value "ltr"
vMiddle   :: Value
vMiddle   = value "middle"
vNo       :: Value
vNo       = value "no"
vNone     :: Value
vNone     = value "none"
vObject   :: Value
vObject   = value "object"
vPassword :: Value
vPassword = value "password"
vPoly     :: Value
vPoly     = value "poly"
vPost     :: Value
vPost     = value "post"
vRadio    :: Value
vRadio    = value "radio"
vRect     :: Value
vRect     = value "rect"
vRef      :: Value
vRef      = value "ref"
vReset    :: Value
vReset    = value "reset"
vRhs      :: Value
vRhs      = value "rhs"
vRight    :: Value
vRight    = value "right"
vRow      :: Value
vRow      = value "row"
vRowGroup :: Value
vRowGroup = value "rowgroup"
vRows     :: Value
vRows     = value "rows"
vRtl      :: Value
vRtl      = value "rtl"
vSquare   :: Value
vSquare   = value "square"
vSubmit   :: Value
vSubmit   = value "submit"
vText     :: Value
vText     = value "text"
vTop      :: Value
vTop      = value "top"
vVoid     :: Value
vVoid     = value "void"
vVSides   :: Value
vVSides   = value "vsides"
vYes      :: Value
vYes      = value "yes"

--------------------------------------------------------------------------------------------------------
-- A continuacin se listan propiedades pertenecientes a la especificacin CSS2 (hojas de estilo)
--------------
-- BOX MODEL
  -- Margin properties
pMarginTop         :: Property
pMarginTop         = property "margin-top"
pMarginRight       :: Property
pMarginRight       = property "margin-right"
pMarginBottom       :: Property
pMarginBottom       = property "margin-bottom"
pMarginLeft        :: Property
pMarginLeft        = property "margin-left"
pMargin            :: Property
pMargin            = property "margin"
  -- Padding properties
pPaddingTop        :: Property
pPaddingTop        = property "padding-top"
pPaddingRight      :: Property
pPaddingRight      = property "padding-right"
pPaddingBottom     :: Property
pPaddingBottom     = property "padding-bottom"
pPaddingLeft       :: Property
pPaddingLeft       = property "padding-left"
pPadding           :: Property
pPadding           = property "padding"
  -- Border width
pBorderTopWidth    :: Property
pBorderTopWidth    = property "border-top-width"
pBorderRightWidth  :: Property
pBorderRightWidth  = property "border-right-width"
pBorderBottomWidth :: Property
pBorderBottomWidth = property "border-bottom-width"
pBorderLeftWidth   :: Property
pBorderLeftWidth   = property "border-left-width"
pBorderWidth       :: Property
pBorderWidth       = property "border-width"
  -- Border color
pBorderTopColor    :: Property
pBorderTopColor    = property "border-top-color"
pBorderRightColor  :: Property
pBorderRightColor  = property "border-right-color"
pBorderBottomColor :: Property
pBorderBottomColor = property "border-bottom-color"
pBorderLeftColor   :: Property
pBorderLeftColor   = property "border-left-color"
pBorderColor       :: Property
pBorderColor       = property "border-color"
  -- Border styles
pBorderTopStyle    :: Property
pBorderTopStyle    = property "border-top-style"
pBorderRightStyle  :: Property
pBorderRightStyle  = property "border-right-style"
pBorderBottomStyle :: Property
pBorderBottomStyle = property "border-bottom-style"
pBorderLeftStyle   :: Property
pBorderLeftStyle   = property "border-left-style"
pBorderStyle       :: Property
pBorderStyle       = property "border-style"
  -- Border shorthand properties
pBorderTop         :: Property
pBorderTop         = property "border-top"
pBorderRight       :: Property
pBorderRight       = property "border-right"
pBorderBottom      :: Property
pBorderBottom      = property "border-bottom"
pBorderLeft        :: Property
pBorderLeft        = property "border-left"
	-- pBorder            = property "border"        ya est definida
--------------
-- VISUAL FORMATTING MODEL
pDisplay     :: Property
pDisplay     = property "display"
pPosition    :: Property
pPosition    = property "position"
  -- Box offsets
pTop         :: Property
pTop         = property "top"
pRight       :: Property
pRight       = property "right"
pBottom      :: Property
pBottom      = property "bottom"
pLeft        :: Property
pLeft        = property "left"

pFloat       :: Property
pFloat       = property "float"
	-- pClear             :: Property
	-- pClear             = property "clear"         ya est definida
pZIndex      :: Property
pZIndex      = property "z-index"
  -- Text direction
pDirection   :: Property
pDirection   = property "direction"
pUnicodeBidi :: Property
pUnicodeBidi = property "unicode-bidi"

--------------
-- VISUAL FORMATTING MODEL DETAILS
  -- Content width
	-- pWidth             :: Property
	-- pWidth             = property "width"         ya est definida
  -- Maximum and minimum widths
pMinWidth       :: Property
pMinWidth       = property "min-width"
pMaxWidth       :: Property
pMaxWidth       = property "max-width"
  -- Content heigth
	-- pHeight            :: Property
	-- pHeight            = property "height"        ya est definida
pMinHeight      :: Property
pMinHeight      = property "min-height"
pMaxHeight      :: Property
pMaxHeight      = property "max-height"
  -- Line height calculations
pLineHeight     :: Property
pLineHeight     = property "line-height"
pVerticalAlign  :: Property
pVerticalAlign  = property "vertical-align"

--------------
-- VISUAL EFFECTS
pOverflow   :: Property
pOverflow   = property "overflow"
pClip       :: Property
pClip       = property "clip"
pVisibility :: Property
pVisibility = property "visibility"

--------------
-- GENERATED CONTENT, AUTOMATIC NUMBERING AND LISTS
-- pContent          = property "content"
pQuotes            :: Property
pQuotes            = property "quotes"
pCounterReset      :: Property
pCounterReset      = property "counter-reset"
pCounterIncrement  :: Property
pCounterIncrement  = property "counter-increment"
  -- Markers and lists
pMarkerOffset      :: Property
pMarkerOffset      = property "marker-offset"
pListStyleType     :: Property
pListStyleType     = property "list-style-type"
pListStyleImage    :: Property
pListStyleImage    = property "list-style-image"
pListStylePosition :: Property
pListStylePosition = property "list-style-position"
pListStyle         :: Property
pListStyle         = property "list-style"

--------------
-- PAGED MEDIA
pMarks           :: Property
pMarks           = property "marks"
  -- Page breaks
pPageBreakBefore :: Property
pPageBreakBefore = property "page-break-before"
pPageBreakAfter  :: Property
pPageBreakAfter  = property "page-break-after"
pPageBreakInside :: Property
pPageBreakInside = property "page-break-inside"
pPage            :: Property
pPage            = property "page"
pOrphans         :: Property
pOrphans         = property "orphans"
pWidows          :: Property
pWidows          = property "widows"

--------------
-- COLORS AND BACKGROUNDS
	-- pColor  :: Property
	-- pColor  = property "color"        ya est definida
  -- Background properties
pBackgroundColor      :: Property
pBackgroundColor      = property "background-color"
pBackgroundImage      :: Property
pBackgroundImage      = property "background-image"
pBackgroundRepeat     :: Property
pBackgroundRepeat     = property "background-repeat"
pBackgroundAttachment :: Property
pBackgroundAttachment = property "background-attachment"
pBackgroundPosition   :: Property
pBackgroundPosition   = property "background-position"
	-- pBackground           :: Property
	-- pBackground           = property "background"        ya est definida

--------------
-- FONTS
pFontFamily  :: Property
pFontFamily  = property "font-family"
  -- Font styling
pFontStyle   :: Property
pFontStyle   = property "font-style"
pFontVariant :: Property
pFontVariant = property "font-variant"
pFontWeight  :: Property
pFontWeight  = property "font-weight"
pFontStretch :: Property
pFontStretch = property "font-stretch"
pFontSize    :: Property
pFontSize    = property "font-size"
pFontSizeAdjust :: Property
pFontSizeAdjust = property "font-size-adjust"
pFont        :: Property
pFont        = property "font"

--------------
-- TEXT
pTextIndent     :: Property
pTextIndent     = property "text-indent"
pTextAlign      :: Property
pTextAlign      = property "text-align"
  -- Text decoration
pTextDecoration :: Property
pTextDecoration = property "text-decoration"
pTextShadow     :: Property
pTextShadow     = property "text-shadow"
  -- Letter and word spacing
pLetterSpacing  :: Property
pLetterSpacing  = property "letter-spacing"
pWordSpacing    :: Property
pWordSpacing    = property "word-spacing"

pTextTransform  :: Property
pTextTransform  = property "text-transform"

pWhiteSpace     :: Property
pWhiteSpace     = property "white-space"

--------------
-- TABLES
pCaptionSide    :: Property
pCaptionSide    = property "caption-side"
pTableLayout    :: Property
pTableLayout    = property "table-layout"
  -- Borders
pBorderCollapse :: Property
pBorderCollapse = property "border-collapse"
pBorderSpacing  :: Property
pBorderSpacing  = property "border-spacing"
pEmptyCells     :: Property
pEmptyCells     = property "empty-cells"
  -- Audio rendering of tables
pSpeakHeader    :: Property
pSpeakHeader    = property "speak-header"

--------------
-- USER INTERFACE
pCursor       :: Property
pCursor       = property "cursor"
pOutline      :: Property
pOutline      = property "outline"
pOutlineWidth :: Property
pOutlineWidth = property "outline-width"
pOutlineStyle :: Property
pOutlineStyle = property "outline-style"
pOutlineColor :: Property
pOutlineColor = property "outline-color"

--------------
-- AURAL STYLE SHEETS
pVolume      :: Property
pVolume      = property "volume"
pSpeak       :: Property
pSpeak       = property "speak"
pPauseBefore :: Property
pPauseBefore = property "pause-before"
pPauseAfter  :: Property
pPauseAfter  = property "pause-after"
pPause       :: Property
pPause       = property "pause"
pCueBefore   :: Property
pCueBefore   = property "cue-before"
pCueAfter    :: Property
pCueAfter    = property "cue-after"
pCue         :: Property
pCue         = property "cue"
pPlayDuring  :: Property
pPlayDuring  = property "play-during"
pAzimuth     :: Property
pAzimuth     = property "azimuth"
pElevation   :: Property
pElevation   = property "elevation"
pSpeechRate  :: Property
pSpeechRate  = property "speech-rate"
pVoiceFamily :: Property
pVoiceFamily = property "voice-family"
pPitch       :: Property
pPitch       = property "pitch"
pPithRange   :: Property
pPithRange   = property "pitch-range"
pStress      :: Property
pStress      = property "stress"
pRichness    :: Property
pRichness    = property "richness"
pSpeakPunctuation :: Property
pSpeakPunctuation = property "speak-punctuation"
pSpeakNumeral     :: Property
pSpeakNumeral     = property "speak-numeral"


--------------
--------------------------------------------------------------------------------------------------------
-- Valores predeterminados para algunas de las propiedades anteriores (CSS2)
	--vAbove           :: Value
	--vAbove           = value "above"
vAbsolute        :: Value
vAbsolute        = value "absolute"
vActiveBorder    :: Value
vActiveBorder    = value "ActiveBorder"
vActiveCaption   :: Value
vActiveCaption   = value "ActiveCaption"
vAlways          :: Value
vAlways          = value "always"
vAppWorkspace    :: Value
vAppWorkspace    = value "AppWorkspace"
vArmenian        :: Value
vArmenian        = value "armenian"
	-- vAuto :: Value
	-- vAuto = value "auto"
vAvoid           :: Value
vAvoid           = value "avoid"
vBackground      :: Value
vBackground      = value "Background"
	-- vBaseline :: Value
	-- vBaseline = value ""
vBehind          :: Value
vBehind          = value "behind"
	--vBelow           :: Value
	--vBelow           = value "below"
vBidiOverride    :: Value
vBidiOverride    = value "bidi-override"
vBlink           :: Value
vBlink           = value "blink"
vBlock           :: Value
vBlock           = value "block"
vBold            :: Value
vBold            = value "bold"
vBolder          :: Value
vBolder          = value "bolder"
vBoth            :: Value
vBoth            = value "both"
-- vBottom = value ""
vButtonFace      :: Value
vButtonFace      = value "ButtonFace"
vButtonHighlight :: Value
vButtonHighlight = value "ButtonHighlight"
vButtonShadow    :: Value
vButtonShadow    = value "ButtonShadow"
vButtonText      :: Value
vButtonText      = value "ButtonText"
vCapitalize      :: Value
vCapitalize      = value "capitalize"
vCaption         :: Value
vCaption         = value "caption"
vCaptionText     :: Value
vCaptionText     = value "CaptionText"
-- vCenter = value ""
vCenterLeft      :: Value
vCenterLeft      = value "center-left"
vCenterRight     :: Value
vCenterRight     = value "center-right"
-- vCircle = value "circle"
vCjkIdeoGraphic  :: Value
vCjkIdeoGraphic  = value "cjk-ideographic"
vCloseQuote      :: Value
vCloseQuote      = value "close-quote"
vCode            :: Value
vCode            = value "code"
vCollapse        :: Value
vCollapse        = value "collapse"
vCompact         :: Value
vCompact         = value "compact"
vCondensed       :: Value
vCondensed       = value "condensed"
vContinuous      :: Value
vContinuous      = value "continuous"
vCrop            :: Value
vCrop            = value "crop"
vCross           :: Value
vCross           = value "cross"
vCrosshair       :: Value
vCrosshair       = value "crosshair"
vDashed          :: Value
vDashed          = value "dashed"
vDecimal         :: Value
vDecimal         = value "decimal"
vDecimalLeadingZero :: Value
vDecimalLeadingZero = value "decimal-leading-zero"
vDigits          :: Value
vDigits          = value "digits"
-- vDisc = value "disc"
vDotted          :: Value
vDotted          = value "dotted"
vDouble          :: Value
vDouble          = value "double"
vEmbed           :: Value
vEmbed           = value "embed"
vEResize         :: Value
vEResize         = value "e-resize"
vExpanded        :: Value
vExpanded        = value "expanded"
vExtraCondensed  :: Value
vExtraCondensed  = value "extra-condensed"
vFarLeft         :: Value
vFarLeft         = value "far-left"
vFarRight        :: Value
vFarRight        = value "far-right"
vFast            :: Value
vFast            = value "fast"
vFaster          :: Value
vFaster          = value "faster"
vFixed           :: Value
vFixed           = value "fixed"
vGeorgian        :: Value
vGeorgian        = value "georgian"
vGreyText        :: Value
vGreyText        = value "GreyText"
vGroove          :: Value
vGroove          = value "groove"
vHebrew          :: Value
vHebrew          = value "hebrew"
vHelp            :: Value
vHelp            = value "help"
-- vHidden      = value "hidden"
vHide            :: Value
vHide            = value "hide"
vHigh            :: Value
vHigh            = value "high"
vHigher          :: Value
vHigher          = value "higher"
vHighlight       :: Value
vHighlight       = value "Highlight"
vHighlightText   :: Value
vHighlightText   = value "HighlightText"
vHiragana        :: Value
vHiragana        = value "hiragana"
vHiraganaIroha   :: Value
vHiraganaIroha   = value "hiragana-iroha"
vIcon            :: Value
vIcon            = value "icon"
vInactiveBorder  :: Value
vInactiveBorder  = value "InactiveBorder"
vInactiveCaption :: Value
vInactiveCaption = value "InactiveCaption"
vInactiveCaptionText  :: Value
vInactiveCaptionText  = value "InactiveCaptionText"
vInfoBackground  :: Value
vInfoBackground  = value "InfoBackground"
vInfoText        :: Value
vInfoText        = value "InfoText"
vInherit         :: Value
vInherit         = value "inherit"
vInLine          :: Value
vInLine          = value "inline"
vInLineTable     :: Value
vInLineTable     = value "inline-table"
vInset           :: Value
vInset           = value "inset"
vInSide          :: Value
vInSide          = value "inside"
vItalic          :: Value
vItalic          = value "italic"
-- vJustify = value ""
vKatakana        :: Value
vKatakana        = value "katakana"
vKatakanaIroha   :: Value
vKatakanaIroha   = value "katakana-iroha"
vLandscape       :: Value
vLandscape       = value "landscape"
vLarge           :: Value
vLarge           = value "large"
vLarger          :: Value
vLarger          = value "larger"
-- vLeft = value ""
vLeftSide        :: Value
vLeftSide        = value "left-side"
vLeftwards       :: Value
vLeftwards       = value "leftwards"
vLevel           :: Value
vLevel           = value "level"
vLight           :: Value
vLight           = value "light"
vLighter         :: Value
vLighter         = value "lighter"
vLineTrough      :: Value
vLineTrough      = value "line-trough"
vListItem        :: Value
vListItem        = value "list-item"
vLoud            :: Value
vLoud            = value "loud"
vLow             :: Value
vLow             = value "low"
vLower           :: Value
vLower           = value "lower"
vLowerAlpha      :: Value
vLowerAlpha      = value "lower-alpha"
vLowerFase       :: Value
vLowerFase       = value "lowercase"
vLowerGreek      :: Value
vLowerGreek      = value "lower-greek"
vLowerLatin      :: Value
vLowerLatin      = value "lower-latin"
vLowerRoman      :: Value
vLowerRoman      = value "lower-roman"
vMarker          :: Value
vMarker          = value "marker"
vMedium          :: Value
vMedium          = value "medium"
vMessageBox      :: Value
vMessageBox      = value "message-box"
vMenu            :: Value
vMenu            = value "menu"
vMenuText        :: Value
vMenuText        = value "MenuText"
-- vMiddle = value ""
vMix             :: Value
vMix             = value "mix"
vMove            :: Value
vMove            = value "move"
vNarrower        :: Value
vNarrower        = value "narrower"
vNResize         :: Value
vNResize         = value "n-resize"
vNeResize        :: Value
vNeResize        = value "ne-resize"
vNwResize        :: Value
vNwResize        = value "nw-resize"
-- vNone = value ""
vNoCloseQuote     :: Value
vNoCloseQuote     = value "no-close-quote"
vNoOpenQuote      :: Value
vNoOpenQuote      = value "no-open-quote"
vNoRepeat         :: Value
vNoRepeat         = value "no-repeat"
vNormal           :: Value
vNormal           = value "normal"
vNoWrap           :: Value
vNoWrap           = value "nowrap"
vOblique          :: Value
vOblique          = value "oblique"
vOnce             :: Value
vOnce             = value "once"
vOpenQuote        :: Value
vOpenQuote        = value "open-quote"
vOverLine         :: Value
vOverLine         = value "overline"
vOutset           :: Value
vOutset           = value "outset"
vOutside          :: Value
vOutside          = value "outside"
vPointer          :: Value
vPointer          = value "pointer"
vPortrait         :: Value
vPortrait         = value "portrait"
vPre              :: Value
vPre              = value "pre"
vRelative     	  :: Value
vRelative     	  = value "relative"
vRepeat       	  :: Value
vRepeat       	  = value "repeat"
vRepeatx      	  :: Value
vRepeatx      	  = value "repeat-x"
vRepeaty      	  :: Value
vRepeaty      	  = value "repeat-y"
vRidge        	  :: Value
vRidge        	  = value "ridge"
-- vRight = value ""
vRightSide        :: Value
vRightSide        = value "right-side"
vRightwards       :: Value
vRightwards       = value "rightwards"
vRunIn        	  :: Value
vRunIn        	  = value "run-in"
vScroll       	  :: Value
vScroll       	  = value "scroll"
vScrollBar        :: Value
vScrollBar        = value "ScrollBar"
vSemiCondensed    :: Value
vSemiCondensed    = value "semi-condensed"
vSemiExpanded     :: Value
vSemiExpanded     = value "semi-expanded"
vSeparate         :: Value
vSeparate         = value "separate"
vShow             :: Value
vShow             = value "show"
vSilent           :: Value
vSilent           = value "silent"
vSlow             :: Value
vSlow             = value "slow"
vSlower           :: Value
vSlower           = value "slower"
vSmall        	  :: Value
vSmall        	  = value "small"
vSmaller      	  :: Value
vSmaller      	  = value "smaller"
vSmallCaps    	  :: Value
vSmallCaps    	  = value "small-caps"
vSamllCaption     :: Value
vSamllCaption     = value "small-caption"
vSoft             :: Value
vSoft             = value "soft"
vSolid        	  :: Value
vSolid        	  = value "solid"
vSpellOut         :: Value
vSpellOut         = value "spell-out"
-- vSquare = value "square"
vSResize          :: Value
vSResize          = value "s-resize"
vSeResize         :: Value
vSeResize         = value "se-resize"
vSwResize         :: Value
vSwResize         = value "sw-resize"
vStatic       	  :: Value
vStatic       	  = value "static"
vStatusBar        :: Value
vStatusBar        = value "status-bar"
vSub          	  :: Value
vSub          	  = value "sub"
vSuper        	  :: Value
vSuper        	  = value "super"
vTable        	  :: Value
vTable        	  = value "table"
vTableCaption     :: Value
vTableCaption     = value "table-caption"
vTableCell        :: Value
vTableCell        = value "table-cell"
vTableColumn      :: Value
vTableColumn      = value "table-column"
vTableColumnGroup :: Value
vTableColumnGroup = value "table-column-group"
vTableFooterGroup :: Value
vTableFooterGroup = value "table-footer-group"
vTableHeaderGroup :: Value
vTableHeaderGroup = value "table-header-group"
vTableRow         :: Value
vTableRow         = value "table-row"
vTableRowGroup    :: Value
vTableRowGroup    = value "table-row-group"
	--vText             :: Value
	--vText             = value "text"
vTextBottom       :: Value
vTextBottom       = value "text-bottom"
vTextTop          :: Value
vTextTop          = value "text-top"
vThick            :: Value
vThick            = value "thick"
vThin             :: Value
vThin             = value "thin"
vThreeDDarkShadow :: Value
vThreeDDarkShadow = value "ThreeDDarkShadow"
vThreeDFace       :: Value
vThreeDFace       = value "ThreeDFace"
vThreeDHighlight  :: Value
vThreeDHighlight  = value "ThreeDHighlight"
vThreeDLightShadow :: Value
vThreeDLightShadow = value "ThreeDLightShadow"
vThreeDShadow     :: Value
vThreeDShadow     = value "ThreeDShadow"
-- vTop = value ""
vTransparent      :: Value
vTransparent      = value "transparent"
vUltraCondensed   :: Value
vUltraCondensed   = value "ultra-condensed"
vUltraExpanded    :: Value
vUltraExpanded    = value "ultra-expanded"
vUnderLine        :: Value
vUnderLine        = value "underline"
vUpperAlpha  	  :: Value
vUpperAlpha  	  = value "upper-alpha"
vUpperCase   	  :: Value
vUpperCase   	  = value "uppercase"
vUpperLatin       :: Value
vUpperLatin       = value "upper-latin"
vUpperRoman  	  :: Value
vUpperRoman  	  = value "upper-roman"
vVisible     	  :: Value
vVisible     	  = value "visible"
vWait             :: Value
vWait             = value "wait"
vWider            :: Value
vWider            = value "wider"
vWindow           :: Value
vWindow           = value "Window"
vWindowFrame      :: Value
vWindowFrame      = value "WindowFrame"
vWindowText       :: Value
vWindowText       = value "WindowText"
vWResize          :: Value
vWResize          = value "w-resize"
vXFast            :: Value
vXFast            = value "x-fast"
vXHigh            :: Value
vXHigh            = value "x-high"
vXLarge      	  :: Value
vXLarge      	  = value "x-large"
vXLoud            :: Value
vXLoud            = value "x-loud"
vXLow             :: Value
vXLow             = value "x-low"
vXSoft            :: Value
vXSoft            = value "x-soft"
vXSlow            :: Value
vXSlow            = value "x-slow"
vXXLarge     	  :: Value
vXXLarge     	  = value "xx-large"
vXSmall      	  :: Value
vXSmall      	  = value "x-small"
vXXSmall     	  :: Value
vXXSmall     	  = value "xx-small"

------------------------------------------
-- UNIDADES
-- LONGITUD
_unit :: Num a => String -> a -> Value
_unit str x = Value (show x ++ str)

   -- Relativos
-- | Especifica el tamao de forma relativa a la propiedad 'pFontSize' (el tamao final se obtiene multiplicando 
-- ambos valores).
em      :: Num a => a -> Value
em      = _unit "em" 
-- | Tamao relativo a la altura de la letra \'x\'.
ex      :: Num a => a -> Value
ex      = _unit "ex" 
-- | Tamao en pixels, dependiente del dispositivo de visualizacin.
px      :: Num a => a -> Value
px      = _unit "px" 

   -- Absolutos
-- | Tamao en pulgadas. (1 pulgada = 2.54 centmetros).
inch    :: Num a => a -> Value
inch    = _unit "in" 
-- | Tamao en centmetros.
cm      :: Num a => a -> Value
cm      = _unit "cm" 
-- | Tamao en milmetros.
mm      :: Num a => a -> Value
mm      = _unit "mm" 

-- | Tamao en puntos. Los puntos utilizados en CSS2 equivalen a la 72-ava parte de una pulgada.
pt      :: Num a => a -> Value
pt      = _unit "pt" 
-- | Tamao en picas. Una pica es igual a 12 puntos.
pc      :: Num a => a -> Value
pc      = _unit "pc" 

-- | Tamao en porcentaje.
percent :: Num a => a -> Value
percent = _unit "%"

-- | Esta funcin nos permite especificar un color (Red,Green,Blue).
-- Rango: 0 - 255 (Incluidos).
rgb :: Int -> Int -> Int -> Value
rgb r g b = value $ "rgb("++show r++","++show g++","++show b++")"

-- | /rgbP/ tiene el mismo objetivo que 'rgb', pero nos permite determinar el color con porcentajes en vez de nmeros
-- enteros. Rango: 0.0 - 100.0 (Incluidos).
rgbP :: Float -> Float -> Float -> Value
rgbP r g b = value $ "rgb("++show r++"%,"++show g++"%,"++show b++"%)"

-- | Para aadir una direccin (URI's).
url :: String -> Value
url s = value $ "url("++s++")"

-- ngulos
-- | Especificacin de un ngulo mediante grados centgrados.
deg  :: Num a => a -> Value
deg  = _unit "deg"
-- | Especificacin de un ngulo mediante gradianes.
grad :: Num a => a -> Value
grad = _unit "grad"
-- | Especificacin de un ngulo mediante radianes.
rad  :: Num a => a -> Value
rad  = _unit "rad"

-- Tiempos
-- | Medida del tiempo en milisegundos.
ms :: Num a => a -> Value
ms = _unit "ms"
-- | Medida del tiempo en segundos.
s  :: Num a => a -> Value
s  = _unit "s"

-- Frecuencias
-- | Medida de frecuencia en hertzios.
hz  :: Num a => a -> Value
hz  = _unit "Hz"
-- | Medida de frecuencia en kilohertzios.
kHz :: Num a => a -> Value
kHz = _unit "kHz"

--------------------------------------------------------------------------------------------------------
-- Valores para algunos colores (los nombres son de la paleta VGA de Windows)
vAqua    :: Value
vAqua    = value "#00FFFF"
vBlack   :: Value
vBlack   = value "#000000"
vBlue    :: Value
vBlue    = value "#0000FF"
vFuchsia :: Value
vFuchsia = value "#FF00FF"
vGray    :: Value
vGray    = value "#808080"
vGreen   :: Value
vGreen   = value "#008000"
vLime    :: Value
vLime    = value "#00FF00"
vMaroon  :: Value
vMaroon  = value "#800000"
vNavy    :: Value
vNavy    = value "#000080"
vOlive   :: Value
vOlive   = value "#808000"
vPurple  :: Value
vPurple  = value "#800080"
vRed     :: Value
vRed     = value "#FF0000"
vSilver  :: Value
vSilver  = value "#C0C0C0"
vTeal    :: Value
vTeal    = value "#008080"
vWhite   :: Value
vWhite   = value "#FFFFFF"
vYellow  :: Value
vYellow  = value "#FFFF00"
