﻿-- #hide

-- Sergio Jimenez Gonzalez  (ult. modificación: 21/7/2003)
-- Titulación: Ing. Técnico en Informática de Sistemas
-- Proyecto fin de carrera.

-- Implementación modulo HTML


-- El módulo HTML proporciona las funciones básicas para creación de etiquetas Html y adición de atributos
-- (tanto Html como CSS2 en línea).
module AP.Cgi.Lib.Html(
-- Módulos
--	module Cgi.Attributes,
--	module Cgi.Codification,
--	module Cgi.Var,
--	module Cgi.Lock,
-- Clases
	CgiInput(..),
	IOAction(..),
-- Tipos básicos
	HTML,
	CGI_, 
	Cgi(..),
	Html,
	cgiToCGI_,
	_CGItoHtml,
	FormInfo(..),
	StateType(..),
	MethodType(..),
	SelectOption,
-- Algunas funciones útiles
	withAttr,		
	withDocAttr,		
	withCss,
	getFieldName,
	getSubmitName,
-- getHTML se exporta para ser utilizada en el módulo Cgi. No usar en los Scripts.
	getHTML, 	
-- Funciones para las etiquetas
  -- Para encapsular el texto	
	hText, 		
	hEmpty,		
	hShow,
  -- Elementos más externos
	hDocType,	
	hHtml, 		
	hHead,		
	hBody,		
	hFrameSet,	
  -- Elementos Head
	hBase,		
	hIsIndex,	
	hLink, 		
	hMeta, 		
	hScript,	
	hStyle,		
	hTitle,			
  -- Elementos de bloque genéricos
	hAddress,		
	hBlockQuote,		
	hCenter,		
	hDiv,			
	hH1,			
	hH2,			
	hH3,			
	hH4,			
	hH5,			
	hH6,			
	hHr,			
	hNoScript,		
	hP,			
	hPre,			
  -- Elementos de Listas
	hDir,			
	hDl,			
	hDt,			
	hDd,			
	hLi,			
	hMenu,			
	hOl,			
	hUl,			
  -- Elementos de tablas
	hTable,			
	hCaption,		
	hColGroup,		
	hColGroupS,		
	hCol,			
	hTHead,			
	hTFoot,			
	hTBody,			
	hTr,			
	hTd,			
	hTh,			
  -- Formularios
    -- Funciones y variables de control de formularios
	-- Estas dos funciones son utilizadas en el módulo Cgi.
	locate,			
	formState,
	tempFilePath,
	cgiCont,
	formFileState,
	ask,
	askExt,
	tell,
	tellExt,
	raiseCGI_, 		
	catchCGI_Ext, 		
	catchCGI_,		
	errorPage_,		
    ------------  Busqueda de datos de entrada y campos de información
	hForm,
	hFormExt,
	hTextInput,		
	hPasswordInput,		
	hCheckBoxInput,		
	hRadioGroupInput,	
	hRadioGroupInputExt,	
	hFileInput,		
	hHiddenInput,		
	hPrimButtonInput,	
	hImageInput, 		
	hResetInput,		
	hButtonInput,	
	hSubmitInput,		
	hSelectInput,
	hSelectInputExt,	
	hTextAreaInput,		
        hTextAreaInputExt,
    ------------  Otros elementos
	hButton,		
	hFieldSet,		
	hLegend,		
	hLabel,			
	hOptGroup,		
	hOption,		
  -- Elementos especiales "en linea"
	hA,			
	hApplet,		
	hBaseFont,		
	hBdo,			
	hBr,			
	hFont,			
	hIFrame,		
	hImg,			
	hMap,			
	hArea,			
	hObject,		
	hParam,			
	hQ,			
	hSpan,			
	hSub,			
	hSup,			
  -- Elementos de frase
	hAbbr,		
	hAcronym,	
	hCite,		
	hCode,		
	hDel,		
	hDfn,		
	hEm,		
	hIns,		
	hKbd,		
	hSamp,		
	hStrong,	
	hVar,		
  -- Elementos de estilo de fuente
	hB,			
	hBig,			
	hI,			
	hS,			
	hSmall,			
	hStrike,		
	hTt,			
	hU,			
  -- Frames
	hFrame,			
	hNoFrames		
) where

-- Importación de modulos
import AP.Cgi.Lib.M
import AP.Cgi.Lib.Doc
import AP.Cgi.Lib.Attributes  
import AP.Cgi.Lib.Codification
import AP.Cgi.Lib.Var
import AP.Cgi.Lib.Lock

import List
import Maybe
import System
import IO
-----------------------------------------------------------------------------------------------------------------
-- Tipos básicos para HTML
type Tag  = String -- Nos servirá para representar las diferentes etiquetas HTML

data HTML = Htext String  		-- Texto normal del documento
	  | Htag {tag       :: Tag,	-- Nombre de la etiqueta
	          attr      :: Attr,	-- Atributos normales
		  css       :: Attr,	-- Atributos de estilo
		  child     :: [HTML],	-- Lista de "hijos" HTML... (subetiquetas anidadas)
		  tabulated :: Bool}	-- Indica si se tabula (True) o no (False) su contenido (sus hijos) 
		  		             -- a la hora de representar el tipo HTML
	  | Htags {tagS   :: Tag,	-- Etiqueta simple (tipo <br>...)
	  	   attrS  :: Attr,	-- Atributos normales
		   cssS   :: Attr}	-- Atributos de estilo
	  
	  | HDoctype {attrD     :: Attr,   -- los atributos para el elemento <DOCTYPE>
	  	      docChilds :: [HTML]} -- solo contiene el elemento <HTML>, pero ponemos una lista para no
		      			   -- introducir un nuevo constructor "vacio" en el tipo HTML para la generación
					   -- de la etiqueta y asi aprovechar la implementación de las demás etiquetas
					   -- (se introducen solas en la lista de hijos de la etiqueta padre)
-----------------------------------------------------------------------------------------------------------------
-- Representación del tipo HTML
-- Constantes
tabulation :: Integer
tabulation = 2

htmlToDoc :: HTML -> Doc
htmlToDoc = htmlToDoc'' True True 0

-- htmlToDoc' toma un HTML (documento a representar), un booleano (que nos indica si debemos tabular el texto final o no) y 
-- un número que nos indica la tabulación actual (para poder restarla y representar el texto sin tabulación alguna en esos
-- casos concretos)

-- Con las etiquetas normales, para saber si debemos tabular, hacemos una AND lógica entre los valores booleanos de 
-- la tabulación de la propia etiqueta y la que estamos propagando. En ese caso, si la etiqueta por sí misma debe tabularse, 
-- pero ésta se encuentra dentro de otra que no debe hacerlo (imagínese <pre> <p>... </p> </pre>) el texto final 
-- no se tabularía, pero las demás etiquetas si (en este caso <p> saldría tabulado con respecto a <pre>, pero el 
-- texto de p, saldria con tabulación 0)
-- versión de pepe

nl' b = if b then nl else vacio

htmlToDoc' :: Bool -> Integer -> HTML -> Doc
htmlToDoc' b n (Htext s) = if b then (nl' True <> texto s') else (tab (-n) (nl' False <> texto s'))  
  where							    --  /^\ Si no queremos tabulación, restamos la tabulación actual
    s' = code s

htmlToDoc' b n (HDoctype at c) = texto ("<!DOCTYPE") <> (attrToDoc at ) <> texto ">" <> (childsToDoc' c True n)  
				--nl' b		-- Si la etiqueta es DOCTYPE, la representamos como si fuera
						-- una etiqueta simple (pero con hijos). Es la única excepción
						-- de entre todas las etiquetas
htmlToDoc' b n (Htag t at ac c tb) = 
  tab (if tb then 0 else (-n)) $ nl' (b && tb) <> texto ("<" ++ t) <> (attrToDoc at ) <> (cssToDoc ac) <> texto ">" 
				  <> tab tabulation (childsToDoc' c (tb && b) (n + tabulation)) 
				<> nl' (b && tb) <> texto ("</" ++ t ++ ">")

htmlToDoc' b n (Htags t at ac) = nl' b <> texto ("<" ++ t) <> (attrToDoc at) <> (cssToDoc ac) <> texto ">" 

-- childsToDoc' pasa a tipo Doc una lista de HTML (los hijos). Propaga los valores booleanos de tabulación y el valor actual, 
-- para que los hijos finales (Htext) puedan restar en caso de no querer tabular.
childsToDoc' :: [HTML] -> Bool -> Integer -> Doc
childsToDoc' [] _ _ = vacio
childsToDoc' cs b n = foldr (<>) vacio (fmap (htmlToDoc' b n) cs)

-----------------------------------------------------------------------------------------------------------------
-- versión de pepe


htmlToDoc'' :: Bool -> Bool -> Integer -> HTML -> Doc
htmlToDoc'' nlb b n (Htext s) = if b then (nl' False <> texto s') else (tab (-n) (nl' False <> texto s'))  
  where							    --  /^\ Si no queremos tabulación, restamos la tabulación actual
    s' = code s

htmlToDoc'' nlb b n (HDoctype at c) = nl' nlb <> texto ("<!DOCTYPE") <> (attrToDoc at ) <> texto ">" <> (childsToDoc'' c True True n)  
						-- Si la etiqueta es DOCTYPE, la representamos como si fuera
						-- una etiqueta simple (pero con hijos). Es la única excepción
						-- de entre todas las etiquetas
htmlToDoc'' nlb b n (Htag t at ac c tb) = 
  tab (if tb then 0 else (-n)) $ nl' nlb <> texto ("<" ++ t) <> (attrToDoc at ) <> (cssToDoc ac) <> texto ">" 
				  <> tab tabulation (childsToDoc'' c (tb && b) (tb && b) (n + tabulation)) 
				<> nl' (nlb && tb) <> texto ("</" ++ t ++ ">")

htmlToDoc'' nlb b n (Htags t at ac) = nl' nlb <> texto ("<" ++ t) <> (attrToDoc at) <> (cssToDoc ac) <> texto ">" 

-- childsToDoc'' pasa a tipo Doc una lista de HTML (los hijos). Propaga los valores booleanos de tabulación y el valor actual, 
-- para que los hijos finales (Htext) puedan restar en caso de no querer tabular.
childsToDoc'' :: [HTML] -> Bool -> Bool -> Integer -> Doc
childsToDoc'' [] _   _ _ = vacio
childsToDoc'' cs nlb b n = foldr (<>) vacio (fmap (htmlToDoc'' nlb b n) cs)


-----------------------------------------------------------------------------------------------------------------
-- Hacemos instancia de Show para poder visualizar las etiquetas
instance Show HTML where
  -- show :: HTML -> String 
  show = show. htmlToDoc

-- Hacemos instancia de Pretty para hacer notar que el tipo HTML es formateable
instance Pretty HTML where
  -- pp     :: HTML -> Doc
  -- ppShow :: HTML -> String  (por defecto ppShow = docAString. pp)
  pp = htmlToDoc

-----------------------------------------------------------------------------------------------------------------
-- Ahora utilizaremos el tipo HTML camuflado en el transformador de estados junto a la información necesaria para
-- procesar la información de los formularios

type FormEntry = [(String,String)]	-- lista de tuplas (nombre,valor) decodificadas. (datos del formulario)
-- |Forma de guardar el estado (parámetro de la función 'AP.Cgi.Lib.CgiBase.runCgiExt').
data StateType = InClient		
		-- ^ Guarda el estado en un campo oculto del formulario.
	       | InServer			
	       	-- ^ Crea un fichero temporal en el servidor, junto con un password aleatorio para controlar el acceso
		-- de los usuarios a dichos archivos (el nombre del archivo se incluye en un campo oculto del formulario,
		-- pero el estado se almacena en el servidor).
	       | Extern_ String	deriving Eq	-- Campo para guardar el password una vez generado

data FormInfo  = FI {dataEntry     :: FormEntry,  -- Entrada de datos (por entrada estandar ó variable de entorno)
		     stateType     :: StateType,  -- Forma de guardar el estado
		     totalValues ,
		     currentValues :: ([String],[String]), -- Listas de resultados acumulados (valores devueltos por formularios), y valores de entrada salida (segunda lista)
		     totalAsk ,
		     currentAsk    :: Int,	-- Contador del número de askExt's ejecutados (total y parcial)
		     idCount       :: (Int,Int) -- Contadores identificativos para los campos de los formularios
		 }				-- El primero, campos normales; El segundo, exclusivo para los "submit"
type CgiState  = (HTML,FormInfo)
type CGI_ a     = M CgiState CgiState a	-- Devuelve una página cuando se produce un error

-- | Un valor de tipo /Cgi a/ representa un cómputo realizado en un programa Cgi que devuelve un valor de tipo /a/.
-- (Cualquier página se forma con las funciones 'ask' ó 'tell' que devuelven tipo /Cgi a/).
newtype Cgi a = Cgi_ (CGI_ a)

-- Esta función nos permite desencapsular el CGI_
cgiToCGI_ :: Cgi a -> CGI_ a
cgiToCGI_ (Cgi_ cgi) = cgi


instance Monad Cgi where
  return x       = Cgi_ (return x)
  (Cgi_ c) >>= f = Cgi_ (do { x <- c 
                            ; cgiToCGI_ (f x)
                            })  


-- El siguiente tipo lo utilizamos para camuflar el tipo CGI_ y forzar el uso de las funciones definidas en el modulo
-- CgiBase para el diálogo entre formularios.
-- | Un valor de tipo /Html a/ representa un cómputo que crea una etiqueta de una página Html y devuelve un 
-- dato de tipo /a/.
newtype Html a = H (CGI_ a)

-- Esta función nos permite desencapsular el CGI_
htmlToCGI_ :: Html a -> CGI_ a
htmlToCGI_ (H cgi) = cgi


-----------------------------------------------------------------------------------------------------------------
instance Monad Html where
  -- return :: a -> Html a
  return x = H (return x)

  -- (>>=) :: Html a -> (a -> Html b) -> Html b
  (H cgi) >>= f = H (do { value <- cgi
  		        ; htmlToCGI_ (f value)
		        })

-- Las siguientes funciones permiten utilizar las que están definidas en la mónada CGI_ pero con tipo Html
readStateH :: Html CgiState
readStateH = H readState

writeStateH :: CgiState -> Html ()
writeStateH s = H (writeState s)

_CGItoHtml :: CGI_ a -> Html a
_CGItoHtml a = H a


-----------------------------------------------------------------------------------------------------------------
-- Algunas funciones útiles para añadir atributos y estilos a una etiqueta

-- La siguiente función es parecido a una función de orden superior. Le pasamos unos atributos, un CGI_ y una función.
-- Dicha función tiene el tipo :: Attr -> HTML -> Bool -> HTML
-- Esta función la utilizamos para no repetir el código que se necesita para definir las funciones withAttr,withDocAttr,
-- withCss
generalWith :: (IsAttr attr) => (Attr -> HTML -> Bool -> HTML) -> Bool -> attr -> Html a -> Html a
generalWith f b attr h = generalWith' f b (toAttr attr) h
  where
    generalWith' :: (Attr -> HTML -> Bool -> HTML) -> Bool -> Attr -> Html a -> Html a
    generalWith' f b atrs (H h) = H $
    			      do {valor <- h	-- ejecutamos el cuerpo para que se creen los hijos
    				 ;padre <- readState		-- leemos el resultado (obtenemos la etiqueta padre)
				 ;writeState ((f atrs (fst padre) b),snd padre)
				 ;return valor
			      }  -- añadimos los atributos al último hijo del padre... O a la propia etiqueta
				   -- (Depende de la función f). Y dependiendo del parámetro booleano, se añaden
				   -- como atributos normales o de estilo

{- |/withAttr/ toma un parámetro que puede ser traducido en atributos HTML y los 
 añade a la etiqueta /Html/ que toma como segundo argumento.
	Ejemplo:

	@
  		withAttr atributos $ 'hHtml' (...)
  		  where
  		    atributos :: 'Attributes.Attr'
  		    atributos = ...
	@

		Para profundizar sobre la forma de crear atributos ver 'AP.Cgi.Lib.Attributes.Attr'.
-}

-- (los añade al último hijo de la etiqueta padre).
withAttr :: (IsAttr attr) => attr 	-- ^ Atributos HTML
			  -> Html a 	-- ^ Etiqueta donde se incluirán los atributos.
			  -> Html a 	-- ^ Etiqueta con atributos añadidos.
withAttr = generalWith addAttribute True
-- NO DEBE USARSE PARA AÑADIR ATRIBUTOS AL ELMENTO DOCTYPE, para ello, hay que utilizar la funcion siguiente a esta
-- (/withDocAttr/).

-- withDocAttr tiene la misma finalidad que withAttr, pero es especial para el elemento \<DOCTYPE\>. 
-- La única diferencia radica en el primer parámetro de generalWith (putAttribute, en vez de addAttribute)
withDocAttr :: (IsAttr attr) => attr -> Html a -> Html a
withDocAttr = generalWith putAttribute True

-- | /withCss/ toma un parámetro que puede ser traducido en atributos de estilo (/CSS/) y los añade a etiqueta
-- /Html/ que toma como segundo argumento. El ejemplo presentado para 'withAttr' también es válido aquí, pues
-- los atributos de estilo son también de tipo 'Attributes.Attr'.

--(la única diferencia con  withAttr es el último parámetro, False, en vez de True)
withCss :: (IsAttr attr) => attr  	-- ^ Atributos CSS
			 -> Html a 	-- ^ Etiqueta donde se incluirán los atributos.
			 -> Html a	-- ^ Etiqueta con atributos añadidos.
withCss a h = generalWith addAttribute False a h

-- El caso de atributos CSS para la etiqueta \<DOCTYPE\> no tiene sentido, 
-- por lo que no proporcionamos función para añadirlos.
-----------------------------------------------------------------------------------------------------------------

partir [x]    = ([],x)
partir (x:xs) = (x:ys, z)
 where
   (ys,z) = partir xs
    
-- Estas son funciones de apoyo para las funciones anteriores
-- addAttribute toma un parámetro atributo y una página HTML, y los añade al último hijo de ésta
-- El último argumento (booleano) indica si hay que añadirlo a los atributos normales (True) o a los de estilo (False)
addAttribute :: Attr -> HTML -> Bool -> HTML
addAttribute a (HDoctype at c) b  = let (init,last) = partir c in HDoctype at (init ++ [putAttribute a (last ) b]) 
			-- deberia tener solo un hijo (<HTML>) 
			-- Es el único que no se crea como los demás, y al utilizar "using"
			-- obtendremos la etiqueta <doctype> en vez del padre (porque no tiene padre), asi que no tenemos que buscar el ultimo hijo
addAttribute a (Htag t at ac c tb) b = let (init,last) = partir c in Htag t at ac (init  ++ [putAttribute a (last ) b]) tb
addAttribute _ (Htags t at ac) _     = Htags t at ac   -- No tiene hijos
addAttribute _ h _		     = h   	   -- El texto no puede tener atributos, así que ignoramos esta posibilidad

-- Esta es la función que finalmente asigna los atributos a una etiqueta. Toma los atributos y la etiqueta en cuestión.
-- El último argumento (booleano) indica si hay que añadirlo a los atributos normales (True) o a los de estilo (False)
putAttribute :: Attr -> HTML -> Bool -> HTML
putAttribute a (HDoctype at c) _        = HDoctype (at <+> a) c  -- Aqui aprovechamos el operador :-D

putAttribute a (Htag t at ac c tb) True = Htag t (at <+> a) ac c tb
putAttribute a (Htags t at ac) True     = Htags t (at <+> a) ac

putAttribute a (Htag t at ac c tb) False = Htag t at (ac <+> a) c tb
putAttribute a (Htags t at ac) False     = Htags t at (ac <+> a)

putAttribute _ h _		        = h 	-- El texto no puede tener atributos, así que ignoramos esta posibilidad

---------------------------------------------------------------------------------------------------------
-- Las funciones siguientes nos ayudan a crear las etiquetas HTML y añadirlas como hijos a otro elemento Hmtl

-- htmlElement toma una cadena de caracteres y un booleano (para saber si se tabula o no) y crea una etiqueta HTML, 
-- con la lista de hijos vacia, y sin ningún atributo
htmlElement :: String -> Bool -> HTML
htmlElement etiq tb = Htag etiq none none [] tb

-- simpleHTMLElement toma una cadena de caracteres y crea una etiqueta simple, sin atributos y con la lista de hijos vacia
simpleHTMLElement :: String -> HTML
simpleHTMLElement etiq = Htags etiq none none

-- docTypeHTMLElement crea la etiqueta más externa (<DOCTYPE>) que es especial.
docTypeHTMLElement :: Attr -> HTML
docTypeHTMLElement at = HDoctype at []

-- getHTML devuelve la pagina HTML encapsulada en la mónada IO. Es la pagina contenida en el elemento CGI_ que toma como 
--argumento
getHTML :: Html a -> IO HTML
getHTML t = do {state <- getMonad (htmlToCGI_ t)
	       ;case (isReturn state) of
	          True  -> return $ fst $ getState state
		  False -> return $ fst $ getError state
	    }

getMonad :: CGI_ a -> IO (Status CgiState CgiState CgiState)
getMonad t = runM (saca t) (error "Falla la función getState")
  where
    saca :: CGI_ a -> CGI_ CgiState
    saca t = do {t
		;s <- readState
		;return s
	     }

-- similar a la anterior, pero la devuelve camuflada en la mónada
getHTML' :: Html a -> Html HTML
getHTML' t = H $
	     do {(htmlToCGI_ t)
	     	;state <- readState
		;return (fst state)
	     }

--  addToChilds toma la pagina actual y el estado del hijo que se va a añadir.Añade éste a la lista de hijos de la página 
-- padre. El resto de información camuflada en el estado del hijo se conserva (puede estar más actualizada que la del padre)
addToChilds :: HTML -> CgiState -> Html ()
addToChilds pagina (hijoHTML,informacion) =
  case pagina of
    (Htag n at ac h tb) -> writeStateH (elem,informacion)
	where
	  elem = Htag n at ac (h++[hijoHTML]) tb
    (HDoctype at c)     -> writeStateH (elem,informacion)
	where
	  elem = HDoctype at (c++[hijoHTML])  -- Realmente solo deberia haber un hijo (el elemento <HTML>)

-- addChild toma un elemento CGI_ y añade la página camuflada en él, en la lista de hijos de la página global
addChild :: Html a -> Html a
addChild t = do {(padre,_) <- readStateH
		;value <- t
		;estadoHijo <- readStateH
		;addToChilds padre estadoHijo
		;return value
	     }

-- La siguiente función crea elementos Cgi's de forma genérica a partir del nombre de la etiqueta, sus hijos (tb Cgi) y el valor
-- booleano de la tabulación.
--  (Esta función será la que utilicemos para crear las diferentes funciones de tags HTML)
normalTag :: String -> Bool -> Html a -> Html a
normalTag nombre tb body = elem
	where
	  hijo = do {(_,info) <- readStateH
	  	    ;writeStateH ((htmlElement nombre tb),info)-- Hacemos que el estado actual sea la etiqueta que nos han pasado
	  	    ;body		   -- Los hijos se incluirán en esta etiqueta
	  	    }
	  elem = addChild hijo

simpleTag :: String -> Html ()
simpleTag etiq = elem
	where
	  hijo = do {(_,info) <- readStateH
	  	    ;writeStateH ((simpleHTMLElement etiq),info)
		    }
	  elem = addChild hijo

-----------------------------------------------------------------------------------------------------------------
-- A continuación se declaran funciones y variables necesarias para la creación correcta de formularios y procesamiento
-- de los datos devueltos por los mismos.

---------------------
-- Identificador para el campo oculto (incluido automáticamente en todos los formularios)
formState :: String
formState = "FORM_STATE"

-- Para los casos en que el estado es guardado en un fichero, éste será el identificador del campo oculto y FORM_STATE 
-- seguirá almacenando el estado (leido del fichero esta vez).
formFileState :: String
formFileState = "FORM_FILE_STATE"

----------------
-- Identificadores para campos de entrada y botones "submit"
formId = "FORM_ID_"
submitId = "SUBMIT_ID_"

-- El campo oculto almacenará 2 datos, las listas de resultados completa y el contador total de askExt's ejecutados.
-- La siguiente función nos devuelve una tupla con esos datos a partir de la información total
getFormState :: FormInfo -> (([String],[String]),Int)
getFormState info = (totalValues info, totalAsk info)

----------------
-- "getName" compone un nombre a partir de las variables "fieldId" y el contador actual (No actualiza el contador)
getName :: Bool -> FormInfo -> String
getName True  info = formId   ++ (show $ fst $ idCount info) -- Nombre para campo normal
getName False info = submitId ++ (show $ snd $ idCount info) -- Nombre para campo "submit"

-- | Esta función nos permite obtener el nombre que se asignará al siguiente campo de formulario.
getFieldName :: Html String
getFieldName = do {(pag,info) <- readStateH	-- leemos el estado actual
		  ;return $ getName True info
	       }

-- | Esta función nos permite obtener el nombre que se asignará al siguiente campo /submit/ de un formulario.
getSubmitName :: Html String
getSubmitName = do {(pag,info) <- readStateH	-- leemos el estado actual
		   ;return $ getName False info
	        }
--------------------------------

-- "nextId" toma un parámetro booleano para alternar entre el incremento del contador de campos normales o el de "submit"
nextId :: Bool -> FormInfo -> FormInfo
nextId True  info = let 
		      (f,s) = idCount info 
		    in 
		      info {idCount = (f+1,s)}

nextId False info = let 
		      (f,s) = idCount info 
		    in 
		      info {idCount = (f,s+1)}
-------------------------------

-- Locate busca el nombre que toma como argumento y lo busca en la lista de tuplas (primera componente). Si lo encuentra, 
-- devuelve la lista que le precede como primera componente, y la tupla que coincide con el nombre será la cabeza de la
-- lista segunda componente de la tupla devuelta. (Util para buscar algún valor en la entrada de datos)
locate :: String -> [(String,String)] -> ([(String,String)],[(String,String)])
locate nombre lista = span (\(x,y) -> (x /= nombre)) lista
-----------------------------------

-- "askCgi" tiene el siguiente comportamiento:
-- comprueba el valor del segundo contador (currentAsk), si es mayor que cero, significa que ya hemos ejecutado este 
-- formulario en un diálogo anterior. Ahora no nos toca mostrarlo, decrementamos el contador de los askExt's actuales.
--   Si el segundo contador es 0, es nuestro turno, incrementamos el valor de los askExt's totales, y ejecutamos los hijos.
--  En cualquier caso, se resetea previamente el estado de la página.
-- Cuando detectamos que el askExt actual debe mostrar la página, elevamos una excepción con la página como error.
-- La función startCGI_ del módulo CGI_ se encargará de "cazar" esta excepción.
-----------------------
-- El primer argumento son los atributos para el elemento <DOCTYPE>
---------
askCgi :: Attr -> CGI_ a -> Cgi a
askCgi at body = 
          Cgi_ $ do {(pag,info) <- readState
	            ;if (currentAsk info > 0) then  -- ya hemos ejecutado esto antes... lo que hagamos ahora será reseteado posteriormente
	             let
		       info' = info {currentAsk = (currentAsk info)-1}
		     in
		       do {writeState (newPag, info')	-- Decrementamos el contador actual de askCgi's ejecutados
		          ;valor <- body		-- Tomamos el valor devuelto por la mónada
		          ;return valor			-- lo devolvemos para que el valor pueda ser usado con posterioridad
		       }
	             else -- Es la primera vez que se ejecuta este askCgi, debemos actualizar el estado con la ejecución de los hijos
	             let
		       info' = info {totalAsk = (totalAsk info)+1, currentAsk = (currentAsk info)-1}
		     in
		       do {writeState (newPag, info')
		          ;raiseCGI_ body  		-- elevamos la página resultante
		       }
	         }
  where
    newPag :: HTML
    newPag = docTypeHTMLElement at

-- | Esta función tiene exactamente el mismo objetivo que 'ask', presentar al cliente un formulario. 
-- Pero esta versión extendida nos permite determinar el atributo del elemento \<DOCTYPE\>. 
-- (En la función 'ask' se utiliza por defecto el atributo 'Attributes.transDocAttribute').
askExt :: Attr 		-- ^ Atributos para la etiqueta \<DOCTYPE\>
       -> Html a 	-- ^ Página a mostrar
       -> Cgi a		-- ^ Valor devuelto
askExt at h = askCgi at (htmlToCGI_ h)

{- | Esta función permite mostrar al cliente un formulario (cuando utilizamos /ask/, se supone que estamos esperando
 alguna información del cliente). Para los casos en los que no queramos obtener valores de formulario (sólo queramos mostrar
 una página normal) utilizaremos 'tell'.

	Ejemplo:
	
	@
		main :: IO ()
	 	main = 'AP.Cgi.Lib.CgiBase.runCgi' principal
		  where
	 	    principal :: 'Cgi' ()
		    principal = ask primeraPágina
	 	    primeraPágina :: 'Html' a
		    primeraPágina = 'hHtml' $ do { 'hHead'(...)
					      ; 'hBody'(...)
					      }
        @

	En los programas que codifiquemos, cada /ask/ representa una interacción con el cliente, por lo que algo como esto:

	@
			do { ask pagina1
			   ; ask pagina2
			   ; ask pagina3
			   }
	@

	No mostrará las 3 páginas al ejecutarse, sino que automáticamente determinará cuáles de ellas han sido 
	ejecutadas ya.
	En la primera ejecución, sólo se mostrará /pagina1/. En la segunda, no se mostrará /pagina1/ y se mostrará
	/pagina2/. En la tercera ejecución, no se mostrará ni /pagina1/ ni /pagina2/, y se mostrará /pagina3/.

	Hay un método alternativo para secuenciar varios formularios y permitir la presentación de diferentes páginas
	dependiendo del botón pulsado en un formulario (podemos incluir más de uno en cada formulario). Esto está explicado
	en detalle en la definición de la función 'hButtonInput'.
-}
ask :: Html a -- ^ Página a mostrar
    -> Cgi a  -- ^ Valor devuelto
ask = askExt transDocAttribute
---------------------------------
-- tell muestra una página al usuario. Toma como argumento los atributos del <DOCTYPE> y el CGI_ en cuestión.
tellCGI_ :: Attr -> CGI_ a -> CGI_ a
tellCGI_ at body = do {state <- readState
		     ;writeState (newPag,snd state)
		     ;raiseCGI_ body
	          }
  where
    newPag :: HTML
    newPag = docTypeHTMLElement at

-- | Esta función tiene el mismo objetivo que 'tell', presentar al cliente una página sin formularios.
-- Al igual que 'askExt', es una versión extendida de su predecesora. Permite determinar el atributo para
-- la etiqueta \<DOCTYPE\> en la página generada, siendo utilizado por defecto el atributo 'Attributes.transDocAttribute'
-- en 'tell'.
tellExt :: Attr 	-- ^ Atributos para la etiqueta \<DOCTYPE\>
        -> Html a 	-- ^ Página a mostrar
	-> Cgi a	-- ^ Valor devuelto
tellExt at h = Cgi_ $ tellCGI_ at (htmlToCGI_ h)

{- | Esta función muestra una página al cliente en la que no tenemos posibilidad de recoger información de ningún tipo.
 
	Ejemplo:
	
	@
		main :: IO ()
		main = 'AP.Cgi.Lib.CgiBase.runCgi' páginaSinForm
		  where
		    páginaSinForm :: 'Cgi' ()
		    páginaSinForm = tell $ do { 'hHtml' (...) }
	@

	Con tell no tenemos posibilidad de presentar las páginas progresivamente (sería un error secuenciar
	varias acciones /tell/, ya que sólo se mostraría una de ellas, la primera).
-}
tell :: Html a 	-- ^ Página a mostrar
     -> Cgi a   -- ^ Valor devuelto
tell c = tellExt transDocAttribute c

-----------------------------------------------------------------------------------------------------------------	  

-- |Los tipos pertenecientes a esta clase son cómputos que pueden realizar operaciones de entrada\/salida. Los tipos
-- 'Html' y 'Cgi' son instancias de esta clase.
class (Monad m) => IOAction m where

  unsafeIO :: IO a -> m a 
  {- ^ Esta función permite ejecutar un cómputo de entrada\/salida en la mónada /m/.
  
     Ejemplo:

	@
     		do { valor \<- unsafeIO $ getClockTime
     		   ; ... (utilizamos valor)...
     		   }
	@

     Este trozo de código realizará la acción de entrada\/salida cada vez que se ejecute.
  -}

  safeIO   :: (Show a,Read a) => IO a -> m a
  {- ^ Esta función, al igual que 'unsafeIO', permite ejecutar un cómputo de entrada\/salida en la mónada /m/,
  pero tiene en cuenta los valores devueltos por la acción para ejecuciones futuras, es decir, garantiza que el valor
  devuelto siempre es el mismo (el primero que obtuvimos).

     Ejemplo:

     >		do { valor <- safeIO $ randomRIO (minV,maxV)
     >		   ; ... (utilizamos valor)...
     >		   }

     Este trozo de código sólo realizará la acción de entrada\/salida la primera vez que se ejecute (las demás veces
     recogerá el valor almacenado de la primera ejecución).
   -}


-----------------------------------------------------------------------------------------------------------------	  
-- Varias funciones útiles relacionadas con el manejo de errores de la mónada y acciones IO
instance IOAction Html where
  -- unsafeIO :: IO a -> Html a
  unsafeIO = H . ioM

  -- safeIO :: (Show a, Read a) => IO a -> Html a
  safeIO accion =          do {(pag,info) <- readStateH
				  ;if (null $ snd $ currentValues info) then	-- No valores de E/S almacenados
				     do {res <- unsafeIO accion			-- ejecutamos la accion
					;let
					   leido    = show res
					   (tv,tio) = totalValues info
					   info'    = info {totalValues = (tv,tio++[leido])}
					;writeStateH (pag,info')
					;return res
				     }
				   else	 -- hay valores en la lista, leemos el primero que es el que nos corresponde
				     do {let 
				     	   (cv,cio) = currentValues info
					   leido    = read $ head cio
				           info'    = info {currentValues = (cv,tail cio)}
					;writeStateH (pag,info')
					;return leido
				     }
			       }

instance IOAction Cgi where
  -- unsafeIO : IO a -> Cgi a
  unsafeIO accion = Cgi_ cgi
    where
      (H cgi) = unsafeIO accion

  -- safeIO :: (Show a, Read a) => IO a -> Cgi a
  safeIO accion = Cgi_ cgi
    where
      (H cgi) = safeIO accion

-----------------------------------------------------------------------------------------------------------------	  
-- Esta función eleva una excepción cuyo página de error es la que toma como argumento camuflada en el CGI_ (junto con su
-- estado)
raiseCGI_ :: CGI_ a -> CGI_ a
raiseCGI_ m = do {m				-- ejecutamos la/s accion/es
		 ;state' <- readState		-- tomamos el nuevo estado
	         ;raiseM state'			-- elevamos excepción con nuevo estado
	      }

-- catchCGI_Ext toma dos Cgi's. Si el primero es una excepcion (un Raise) ignora el resultado y ejecuta
-- el segundo. Si el primero no es una excepción, devuelve su resultado. (Es como un manejador de excepciones)
catchCGI_Ext :: CGI_ a -> CGI_ a -> CGI_ a
catchCGI_Ext m1 m2 = catchM m1 (\_ -> m2)

-- Esta función nos permite capturar un error y devolver la página asociada con su estado (transforma la pagina de error
-- en una pagina normal)
catchCGI_ :: CGI_ () -> CGI_ ()
catchCGI_ m = catchM m (\e -> writeState e)

-----------------------------------------------------------------------------------------------------------------	  
-- A continuación vamos a definir una función que crea una página un poco genérica que puede ser utilizada para mostrar 
-- mensajes de error
-- Los dos primeros parámetros son el título y el texto a mostrar respectivamente.
errorPage_ :: String -> String -> Html ()
errorPage_ tit msg = hHtml $
			do {hHead (hTitle $ hText tit)
			   ;hBody (
			      do { hCenter $ withCss [ pColor =: vRed ] $ hDiv $ 
			          do { hH1 $ hB $ hText "Error"
 				     ; hH1 $ hText msg
 				     ; hH2 $ withAttr (pHRef =: value "javascript:history.go(-1)") $ hA $ hText "[Back]"
 				}
			      }
			   )
			}

-----------------------------------------------------------------------------------------------------------------	  
-- Las funciones que se definen ahora crean los campos HTML necesarios en los formularios y la comprobación 
-- de los datos de entrada.

-- A partir de la entrada recibida (cadenas de caracteres) haremos una conversión para que la entrada sea tipada.
-- Para ello vamos a definir una clase llamada CgiInput. 

-- |Esta clase permite mostrar mensajes de error cuando no sea posible una lectura correcta de la cadena de caracteres 
-- para su conversión a un tipo instanciado de la clase. Así mismo, define las funciones de análisis y visualización de un valor para
-- un tipo específico (el tipo instanciado).
-- Si queremos mostrar de una forma específica un dato de dicho tipo, o cambiar la forma en la que se lee, para realizar
-- la conversión de tipos, sólo debemos definir las funciones pertenecientes a la clase.
-- Todas las funciones, excepto 'msg', tienen algún método por defecto definido (que coincide con las funciones predefinidas
-- de Haskell para lectura y visualización).
class (Show a, Read a) => CgiInput a where

  -- |Visualización de un valor.
  toString :: a -> String	
  toString = show
  
  -- |Visualización de una lista de valores.
  listToString :: [a] -> String
  listToString = show
  
  -- |Lectura de un valor (a partir de una cadena de caracteres).
  parse :: ReadS a
  parse = reads		-- método por defecto
  
  -- |Lectura de una lista de valores.
  parseList :: ReadS [a]
  parseList = readList  -- método por defecto

  {-| Mensaje de error que se mostrará en los casos en los que haya problemas de lectura (si la conversión de tipos no se
   realiza con éxito).
   Esta es la única función que no tiene métodos por defecto definidos. Por lo tanto, cualquier instancia de un tipo
   debe proporcionar, al menos, la definición de esta función.

   Ejemplo:
   
   >             instance CgiInput Int where
   > 		  msg _ = "Int expected from data input." 
   -}
  msg       :: a -> String
  msg _  = "error parsing data"

instance (CgiInput a) => CgiInput [a] where
  parse    = parseList
  toString = listToString
  msg _ = "List expected from data input."

instance CgiInput Char where
  listToString xs = xs	
  parse str = if (null str) then [] else [(head str,"")] -- Eleva página de error si no introducimos ningún dato.
  parseList str = [(str,"")]
  msg _ = "Char expected from data input."

instance CgiInput Int where
  msg _ = "Int expected from data input."

instance CgiInput Integer where
  msg _ = "Integer expected from data input."

instance CgiInput Float where
  msg _ = "Float expected from data input."

instance CgiInput Double where
  msg _ = "Double expected from data input."

instance CgiInput Bool where
  msg _ = "Boolean expected from data input."


-- Versión copiada del "readIO" del Prelude. En vez de elevar una excepción devolvemos un "Maybe a"
readIO'   :: CgiInput a => String -> IO (Maybe a)
readIO' s = case [x | (x,t) <- parse s, ("","") <- lex t] of
              [x] -> return (Just x)
              _   -> return Nothing

read' :: CgiInput a => String -> a
read' s = case [x | (x,t) <- parse s, ("","") <- lex t] of
              [x] -> x
              _   -> error "Html.read' no parse"

-- Podemos observar que a diferencia de la versión original que hacía un "reads s", nosotros hacemos aqui un 
-- "parse s". De esta forma se utiliza el método definido en la clase (que si no lo cambiamos, conincidirá con reads).

-----------------------------------------------------------------------------------------------------------------	  
-- Definición de los tags
-- La nomenclatura utilizada es:
--		hXXXX -> para las etiquetas HTML, donde XXXX es el nombre de la etiqueta. ej: hBody => etiqueta <BODY>
-- (La única excepción es la función "hText", que no representa ninguna etiqueta, y se utiliza para escribir el texto 
--  normal)

-- Funcion que encapsula texto normal en el transformador de estados
-- La filosofía es parecida a la de "normalTag", excepto que en este caso, no tenemos "hijos" que ejecutar.
-- | Función que muestra en la página, la cadena de caracteres que se le proporcione como argumento.
hText :: String -> Html ()
hText s = elem
	where
	  hijo = do {(_,info) <- readStateH
	  	    ;writeStateH ((Htext s),info)
		    }
	  elem = addChild hijo

-- | Esta función es útil cuando no queremos mostrar ningún elemento, pero necesitamos una función de tipo /Html a/.
-- (Esta función no incluye ninguna información en la página resultante).
hEmpty :: Html ()
hEmpty = hText ""

-- | Mostramos la representación en cadenas de caracteres, del argumento recibido (se hace un /show/ del valor).
hShow :: (Show a) => a -> Html ()
hShow x = hText (show x)

-- A continuación mostramos una organización de los elementos HTML dependiendo de la zona donde pueden aparecer.

-- El primer tag es diferente, pues aún siendo simple, debe proporcionar la lista de elementos HTML a los que se incluirán
-- los demás tags. Así pues, toma como parámetro un elemento CGI_ (que será normalmente el "HTML") y lo ejecuta para que se añada 
-- a su lista de hijos. El estado inicial de la mónada depende de la entrada de datos procesada en el módulo Cgi.hs
hDocType :: FormInfo -> Html () -> Html ()
hDocType info body = do {writeStateH (docTypeHTMLElement none,info)
			;body
			}

{-
Elementos mas externos
    * HTML - documento HTML
      * HEAD - Cabecera del documento
      * BODY - Cuerpo del documento
      * FRAMESET
-}
hHtml :: Html a -> Html a
hHtml = normalTag "HTML" True

hHead :: Html a -> Html a
hHead = normalTag "HEAD" True

hBody :: Html a -> Html a
hBody = normalTag "BODY" True

hFrameSet :: Html a -> Html a
hFrameSet = normalTag "FRAMESET" True

{-
Elementos Head
    * BASE - URI base del documento
    * ISINDEX - Input prompt
    * LINK - Enlaces del documento
    * META - Metadatos
    * SCRIPT - Script del lado del cliente
    * STYLE - Hoja de estilo incrustada
    * TITLE - Título del documento
-}
hBase :: Html ()
hBase = simpleTag "BASE"

hIsIndex :: Html ()
hIsIndex = simpleTag "ISINDEX"

hLink :: Html ()
hLink = simpleTag "LINK"

hMeta :: Html ()
hMeta = simpleTag "META"

hScript :: Html a -> Html a
hScript = normalTag "SCRIPT" True

hStyle :: Html a -> Html a
hStyle = normalTag "STYLE" True

hTitle :: Html a -> Html a
hTitle = normalTag "TITLE" True

{-
Elementos de bloque genericos
    * ADDRESS - Información de contacto
    * BLOCKQUOTE - Block quotation
    * CENTER - Bloque centrado
    * DEL - Texto borrado
    * DIV - Contenedor genérico de bloques
    * H1 - Cabecera de nivel 1
    * H2 - Cabecera de nivel 2
    * H3 - Cabecera de nivel 3
    * H4 - Cabecera de nivel 4
    * H5 - Cabecera de nivel 5
    * H6 - Cabecera de nivel 6
    * HR - Separador horizontal
    * INS - Texto insertado
 * ISINDEX - Input prompt    -- Este elemento  ya ha sido definido en "Elementos de Head"
    * NOSCRIPT - Contenido alternativo al script
    * P - Párrafo
    * PRE - Texto preformateado
-}
hAddress :: Html a -> Html a
hAddress = normalTag "ADDRESS" True

hBlockQuote :: Html a -> Html a
hBlockQuote = normalTag "BLOCKQUOTE" True

hCenter :: Html a -> Html a
hCenter = normalTag "CENTER" True

hDel :: Html a -> Html a
hDel = normalTag "DEL" True

hDiv :: Html a -> Html a
hDiv = normalTag "DIV" True

hH1 :: Html a -> Html a
hH1 = normalTag "H1" True

hH2 :: Html a -> Html a
hH2 = normalTag "H2" True

hH3 :: Html a -> Html a
hH3 = normalTag "H3" True

hH4 :: Html a -> Html a
hH4 = normalTag "H4" True

hH5 :: Html a -> Html a
hH5 = normalTag "H5" True

hH6 :: Html a -> Html a
hH6 = normalTag "H6" True

hHr :: Html ()
hHr = simpleTag "HR"

hIns :: Html a -> Html a
hIns = normalTag "INS" True

hNoScript :: Html a -> Html a
hNoScript = normalTag "NOSCRIPT" True

hP :: Html a -> Html a
hP = normalTag "P" True

hPre :: Html a -> Html a
hPre = normalTag "PRE" False

{-
Lists

    * DIR - Lista de directorio
    * DL - Lista de definición
          o DT - Término de definición
          o DD - Descripción de la definición
    * LI - Objeto de lista (List Item)
    * MENU - Lista Menu
    * OL - Lista Ordenada
    * UL - Lista desordenada (unordered list)
-}

hDir :: Html a -> Html a
hDir = normalTag "DIR" True

hDl :: Html a -> Html a
hDl = normalTag "DL" True

hDt :: Html a -> Html a
hDt = normalTag "DT" True

hDd :: Html a -> Html a
hDd = normalTag "DD" True

hLi :: Html a -> Html a
hLi = normalTag "LI" True

hMenu :: Html a -> Html a
hMenu = normalTag "MENU" True

hOl :: Html a -> Html a
hOl = normalTag "OL" True

hUl :: Html a -> Html a
hUl = normalTag "UL" True

{-
 Elementos de Tablas

    * TABLE - Tabla
          o CAPTION - Cabecera de la tabla
          o COLGROUP - Grupo de columnas de tabla  (Puede ser normalTag simple o normal)
	  	+ COL - Columna de tabla
          o THEAD - Cabeza de la tabla
	  o TFOOT - Pie de la tabla
          o TBODY - Cuerpo de la tabla
          o TR - Fila de la tabla
                + TD - Celda de datos de tabla 
                + TH - Celda cabecera de tabla
-}
hTable :: Html a -> Html a
hTable = normalTag "TABLE" True

hCaption :: Html a -> Html a
hCaption = normalTag "CAPTION" True

hColGroup :: Html a -> Html a
hColGroup = normalTag "COLGROUP" True

hColGroupS :: Html ()
hColGroupS = simpleTag "COLGROUP"

hCol :: Html ()
hCol = simpleTag "COL"

hTHead :: Html a -> Html a
hTHead = normalTag "THEAD" True

hTFoot :: Html a -> Html a
hTFoot = normalTag "TFOOT" True

hTBody :: Html a -> Html a
hTBody = normalTag "TBODY" True

hTr :: Html a -> Html a
hTr = normalTag "TR" True

hTd :: Html a -> Html a
hTd = normalTag "TD" True

hTh :: Html a -> Html a
hTh = normalTag "TH" True


{-
Formularios

    * FORM - Formulario interactivo
          o BUTTON - Boton
          o FIELDSET - Grupo de control de formulario
                + LEGEND - Titulo del grupo
          o INPUT - Entrada de formulario
          o LABEL - Etiqueta de campo de formulario
          o SELECT - Selector de opciones
                + OPTGROUP - Grupo de opciones
                      # OPTION - Opcion de menu
          o TEXTAREA - Entrada de texto multi-linea
-}
--------------------------
-- algunas variables útiles...
noEntryInput = "NO_ENTRY_INPUT"

-- |Path del directorio temporal donde se guardarán los archivos que guardan el estado (en caso de que 'StateType' 
-- sea 'InServer'). Cuando se utiliza este método, automáticamente se comprueba si este directorio existe o no. 
-- De no existir, se crean automáticamente tanto el directorio como el archivo que guarda el contador 'cgiCont'. 
-- Si el directorio existe, entonces sólo se comprueba la existencia de dicho archivo.
tempFilePath :: String
tempFilePath = ".\\cgitemp\\"	-- Tendremos un directorio llamado "cgiTemp" donde se guardaran los archivos temporales.

-- |Nombre del archivo que contiene la cuenta para generar los diferentes nombres de los archivos temporales.
cgiCont :: String
cgiCont = "cgi_cont.txt"

-- Nombre para los archivos temporales  (se forman del modo: tempFile++"cont"++".txt", donde cont se obtiene de "cgiCont")
tempFile :: String
tempFile = "tmp_cgi_"
-----------------------------------------------------------------------------------------------------------
-- | Este tipo nos permite determinar el método que vamos a utilizar en los formularios (parámetro de la función 'hFormExt').
data MethodType = Get 	-- ^ Método /Get/ (la información es recibida a través de la variable de entorno 'queryString').
		| Post  -- ^ Método /Post/ (la información es recibida a través de la entrada estándar).


-- hForm toma un CGI_ que es el cuerpo del formulario. Una vez ejecutado, lee el estado y añade un campo oculto con 
-- la información necesaria para saber en qué parte del "diálogo" nos encontramos.
-- En el caso de que el estado se guarde en un fichero del servidor, debemos buscar el nombre asociado al fichero en la 
-- entrada de datos. Si no lo encontramos, lo creamos con el identificador "tempFile" y el contador de "cgiCont".
-- Si lo encontramos, utilizamos el mismo fichero.

-- | /hForm/ crea las etiquetas necesarias para incluir un formulario en la página. Automáticamente añade los valores
-- para la acción (atributo 'Attributes.pAction'), que será el mismo archivo que se está ejecutando actualmente, y 
-- el método ('Post'). Recibe como argumento el contenido que se mostrará en el formulario.
hForm :: Html () 		-- ^ Página a mostrar
      -> Html ()		-- Utiliza método Post por defecto
hForm = hFormExt Post

-----------------------------------------------------------------------------------------------------------
-- | Esta función funciona exactamente igual que 'hForm', pero permite especificar el método a utilizar en su 
-- primer argumento
hFormExt :: MethodType 		-- ^ Método a utilizar en el envío de datos al servidor.
         -> Html () 		-- ^ Página a mostrar
	 -> Html ()
hFormExt met body = do {servidor <- leeDatos serverUrl
		       ;archivo  <- leeDatos scriptName
		       ;let formulario = value ((init servidor) ++ archivo)
		       ;withAttr (pMethod =: choice met <+> pAction =: formulario) $ elem	-- añadimos metodo y el nombre del archivo por defecto
	            }
  where
    choice :: MethodType -> Value
    choice met = case met of
    		   Get  -> vGet
		   Post -> vPost
    hijo = do {(_,info) <- readStateH		-- Hacemos que el estado actual sea la etiqueta FORM
	      ;let padre = htmlElement "FORM" True
	      ;writeStateH (padre,info)
	      ;body			-- Los hijos se incluirán en esta etiqueta
	      ;(_,info') <- readStateH	-- leemos de nuevo el estado (por si lo han cambiado los hijos)
	      ;case (stateType info') of
	         InClient -> withAttr (pName =: value formState <+> pType =: vHidden <+>   -- Incluimos el campo oculto con el estado actual
	                               pValue =: value (show $ getFormState info')) $ hInput 
		 Extern_ pass -> do {nombre <- unsafeIO creaNombre
		 		    ;let
				       archivo = tempFilePath ++ nombre
				       estado  = show $ (getFormState info', pass)
				    ;unsafeIO $ writeFile archivo estado
				    ;withAttr (pName =: value formFileState <+> pType =: vHidden <+>
				               pValue =: (value $ show $ (nombre,pass)) ) $ hInput
			   }
	   }				
    elem   = addChild hijo
    leeDatos :: String -> Html String
    leeDatos s = unsafeIO $ getEnv s

    creaNombre :: IO String -- Accede en exclusión mutua al fichero q tiene la cuenta para diferenciar los archivos de estado
    creaNombre = do {let archivo = tempFilePath ++ cgiCont
	  	    ;handle <- lockFile archivo
          	    ;str <- strictReadFile archivo :: IO String
	  	    ;let cont = (read str)+1 :: Integer
	  	    ;writeFile archivo (show cont)
	  	    ;unlockFile handle
		    ;return (tempFile++str++".txt")
       		 }
	where
	  strictReadFile path =
	    do {str <- readFile path
	       ;return $! str
	    }

-- El primer parametro sirve para poder añadir un elemento HTML simple o compuesto, dependiendo del parámetro enviado. De
-- esta forma podemos utilizar la funcion input para crear campos de entrada y select's.
-- "valor" es un valor por defecto que será devuelto si no se encuentra el resultado asociado a la entrada
-- Los atributos se añaden cuando se añada la entrada del campo a la página actual
input :: HTML -> String -> String -> Attr -> Html (Maybe String)
input elem nombreCampo valor atrs = 
			       do {(pag,info) <- readStateH
				  ;if (null $ fst $ currentValues info) then	-- No hay resultados almacenados 
				     if (null $ dataEntry info) then	-- No hay datos de entrada
				       if (currentAsk info >= 0) then	-- ya nos hemos ejecutado al menos 1 vez, nuestro campo no esta en la entrada, devolvemos un valor por defecto y lo almacenamos en la lista de resultados)
				         do {let leido     = Just valor
					 	 (tv, tio) = totalValues info
					         info'     = info {totalValues = (tv++[valor], tio)}
					    ;writeStateH (pag,info')
					    ;return leido
					 }
				       else 	-- Añadimos el campo y devolvemos Nothing
					 do {let hijo = (writeStateH (elem,info))  -- elem es el campo que tenemos que añadir
					    ;withAttr (pName =: value nombreCampo <+> atrs) $ addChild hijo
					    ;return Nothing
					 }
				     else
				       do {let (lista1,lista2) = locate nombreCampo (dataEntry info)	-- Buscamos la tupla correspondiente
					       leido = if (null lista2) then
							 Just valor
						       else
						         Just (snd $ head lista2)
					       info'  = if (null lista2) then	-- No hemos encontrado ninguna tupla con ese nombre (1a componente)
					       		  let
							    (tv, tio) = totalValues info
							  in
							    info {totalValues = (tv++[valor],tio)}
							else			-- Hemos encontrado la tupla, es la cabeza de la segunda lista
							  let
							    (tv, tio) = totalValues info
							  in
							    info {dataEntry = lista1 ++ (tail lista2), totalValues = (tv++[fromJust leido],tio)}
					  ;writeStateH (pag,info')
					  ;return leido
				       }
				   else		-- hay valores en la lista, leemos el primero que es el que nos corresponde
				     do {let (cv, cio) = currentValues info
				             leido = Just (head cv)
				     	     info' = info {currentValues = (tail cv,cio)}
					;writeStateH (pag,info')
					;return leido
				     }
			       }

-- "makeInput" recibe un booleano que le indica si es un campo normal (true) o un campo submit (false)
-- La cadena es un valor por defecto que se devuelve si no se ha encontrado la entrada o aún no ha llegado el resultado
-- Los parámetros se añaden para determinar el tipo de entrada
-- Su objetivo es llamar a la función "input" con el nombre apropiado de campo y devolver el resultado devuelto por éste
makeInput :: Bool -> String -> Attr -> Html String
makeInput  b defecto atrs = do {(pag,info) <- readStateH
			       ;let nombreCampo = getName b info
			            info' = nextId b info
			       ;writeStateH (pag,info')
			       ;valor <- input (simpleHTMLElement "INPUT") nombreCampo defecto atrs
			       ;if (isNothing valor) then return defecto
			        else return (fromJust valor)
			    }

{- | Función utilizada para incluir un campo de entrada de texto. Sin embargo, el dato recogido es automáticamente
 comprobado y convertido al tipo adecuado para su utilización. Por lo tanto, es obligatorio que el tipo al que debe
 convertirse el valor recogido sea instancia de la clase 'CgiInput' (para asegurarnos de que disponemos de las funciones
 adecuadas de conversión y visualización de dicho tipo).

 	Ejemplo:
	
	@
		do {...
		   ;edad \<- 'hTextInput'
		   ;guardaEdad edad
		}
		  where 
		    guardaEdad :: Int -> 'Html' a
	@

	En este ejemplo, la variable /edad/ debe ser de tipo /Int/ obligatoriamente, ya que la definición de /guardaEdad/
	así lo define. En caso de que el dato recogido no pueda ser convertido al tipo esperado, 
	obtendremos una página de error (para más información ver 'CgiInput').
-}
hTextInput :: (CgiInput a) => Html a	-- ^ Devuelve el valor introducido en el formulario
hTextInput= do {(pag,info) <- readStateH
	       ;let nombreCampo = getName True info
	            info' = nextId True info
	       ;writeStateH (pag,info')
	       ;cadena <- input (simpleHTMLElement "INPUT") nombreCampo "" (pType =: vText)
	       ;if (isNothing cadena) then 	-- No hay datos de entrada
	          return undefined
	        else				-- Hay datos de entrada.... Intentamos hacer la conversión de tipos
		  do {valor <- unsafeIO(readIO' $ fromJust cadena)
		     ;if (isNothing valor) then   -- No ha sido posible la conversión de tipos, mostramos página de error.
			do {H $ cgiToCGI_ $ tellExt transDocAttribute (errorPage_ "Página de error" $ msg (fromJust valor))
		           ;return undefined
			}
	              else  -- Devolvemos el valor obtenido
		        unsafeIO(return $ fromJust valor)
		  }
	    }

{- | Creación de campo de /passwords/ (no se visualizan los caracteres al introducirlos).
 	Ejemplo:
	
	@
		do {nombre \<- 'hTextInput'
		   ;pass   \<- 'hPasswordInput'
		   ;registra nombre pass
		   ...
		}
		  where
		    registra :: String -\> String -\> 'Html' a
		    registra n p = ...
	@
-}
hPasswordInput :: Html String	-- ^ Cadena de caracteres introducida en el formulario
hPasswordInput = makeInput True "" (pType =: vPassword)

{- | Creación de una caja de selección.
	Ejemplo:

	@
		do{...
		  ;pulsado \<- hCheckBoxInput
		  ;procesaPulsado pulsado
		}
		  where
		    procesaPulsado :: Bool -\> 'Html' a
		    procesaPulsado p = ...
	@

-}
hCheckBoxInput :: Html Bool 	-- ^ Devuelve /True/ si fue pulsada. /False/ en otro caso.
hCheckBoxInput = do {res <- (makeInput True noEntryInput (pType =: vCheckBox))
                   ;if (res /= noEntryInput) then return True -- No podemos hacer directamente un "read res" porque el
		    else return False			 -- valor devuelto depende de los atributos
		 }

{- | Esta función crea un conjunto de botones radiales relacionados (sólo uno de ellos puede 
 estar marcado en un momento dado). Para cada uno de ellos se puede dar una definición o crear una etiqueta especial
 (notar que el parámetro es de tipo /Html ()/ por lo que podríamos incluir una imagen, por ejemplo, como descripción
 del botón radial). Si sólo queremos dar una definición escrita, nos basta con utilizar 'hText'.

	Ejemplo:
	
	@
 		do {...
 		   ;'hText' \"Elige una opción\"
 		   ;valor \<- hRadioGroupInput [('hText' \"Opción 1\",1),('hText' \"Opción 2\",2),('hText' \"Opción 3\",3)]
 		   ;guardaValor valor
 			...
 		}
 		  where
 		    guardaValor :: Int -\> ...
	@

	En este ejemplo, podemos observar que el valor devuelto por la función /hRadioGroupInput/ no es una cadena de 
	caracteres sino un entero, ya que se hace la conversión de tipos de forma automática.
-}
hRadioGroupInput :: (CgiInput a) => [(Html (), a)] -- ^ Una lista de \"definiciones\" y el valor asociado al botón.
                                 -> Html a	   -- ^ Devolvemos el valor asociado al botón pulsado (valor proporcionado en
				 		   -- el parámetro anterior).
hRadioGroupInput xs = hRadioGroupInputExt xs (error "hRadioGroupInput read too soon")

-- radioGroupInput recibe una lista de tuplas con Html () (los datos que se mostrarán al lado de cada radial)
-- y los valores asociados a cada radial, y un valor por defecto que se devolverá cuando no haya entradas o 
-- no se haya seleccionado ninguna
hRadioGroupInputExt :: (CgiInput a) => [(Html (), a)] -> a -> Html a
hRadioGroupInputExt lista defecto = do {val <- getRadioValue True lista
			           ;if (isNothing val) then return defecto
				    else return (fromJust val)
			        }
  where
    getRadioValue :: (CgiInput a) => Bool -> [(Html (), a)] -> Html (Maybe a)
    getRadioValue check ((x,xval):xs) = do {(pag,info) <- readStateH
    				     ;let nombreCampo = getName True info
				     ;val <- (input (simpleHTMLElement "INPUT") nombreCampo noEntryInput (pType =: vRadio <+> pValue =: value (show xval) <+> if check then aChecked else none))
				     ;x		-- ejecutamos la mónada para que se escriba la descripción del botón
				     ;(pag',info') <- readStateH 		-- por si ha cambiado algo
				     ;let info'' = nextId True info'
				     ;if (isNothing val) then 	-- Hemos añadido el campo a la página, no hay valores aún
				        if (null xs) then	-- Ultimo elemento de radio, incrementamos contador y devolvemos Nothing
					  do {writeStateH (pag',info'')
					     ;return Nothing
					  }
					else
					  getRadioValue False xs
				      else	-- hemos encontrado algo, veamos qué es
				        if ((fromJust val) == noEntryInput) then -- Ningun radial ha sido pulsado (es el valor por defecto devuelto)
					  do {writeStateH (pag',info'')
					     ;return Nothing
					  }
					else	-- No es el valor por defecto, hemos encontrado el valor pulsado, lo devolvemos
					  do {writeStateH (pag',info'')
					     ;return (Just $ read $ fromJust val)
					  }
    				  }

-- | Esta función crea una etiqueta que nos permite el envío de archivos al servidor. Para ello es necesario que el método
-- utilizado para enviar el formulario sea 'Post' y el atributo 'pEncType' sea \"multipart\/form-data\". Actualmente, 
-- sin embargo, no es posible utilizar este método de envío, puesto que no está implementado en la biblioteca la 
-- decodificación de este formato en la recepción de los datos.
-- (si utilizamos el modo \"multipart\/form-data\" obtendremos un error en la ejecución de la función 'AP.Cgi.Lib.CgiBase.runCgi').
-- No obstante, si no especificamos el valor de 'pEncType' tal y como hemos citado antes, 
-- podremos obtener una cadena de caracteres que corresponde con el archivo que el cliente quería enviar al servidor. Esto 
-- es útil si pretendemos asociar algún programa que se encargue del envío del fichero a partir del nombre
-- del mismo (podemos incluir en el lado del cliente un gestor de envio de ficheros en /javascript/ por ejemplo).
hFileInput :: Html String
hFileInput = makeInput True "" (pType =: vFile)

{- | Creación de un campo oculto en el formulario. El valor devuelto es el valor proporcionado en su creación (por medio
  del atributo 'Attributes.pValue').
	
	Ejemplo:

	@
		do {valor \<- 'withAttr' ('pValue' =: v) $ hHiddenInput
		   ;...
		}
		  where
		    v :: 'Attributes.Value'
		    v = ...
	@
-}
hHiddenInput :: Html String	-- ^ Valor del campo oculto
hHiddenInput = makeInput True "" (pType =: vHidden)

simpleButton :: Attr -> Html ()
simpleButton attrs = do {(pag,info) <- readStateH
		        ;if (currentAsk info == -1) then	-- nos toca mostrar
		           withAttr attrs $ hInput
		         else
		           return ()
	             }

-- | Esta función permite añadir botones normales que no devuelven ningún resultado. (Son útiles si se combinan
-- con otros elementos, como por ejemplo /javascript/).
hPrimButtonInput :: Html ()
hPrimButtonInput = simpleButton (pType =: vButton)

-- | Incluye un botón de aceptación del formulario. Al pulsarse, los datos serán enviados al servidor.
hSubmitInput :: Html ()
hSubmitInput = simpleButton (pType =: vSubmit)

-- | Botón de /Reset/. Inicializa los campos, borrando el contenido que hubiéramos escrito en ellos.
hResetInput :: Html ()
hResetInput = simpleButton (pType =: vReset)

{- | Tiene el mismo objetivo que 'hButtonInput', funciona como si fuera un botón /submit/, pero en vez de presentarlo, 
 presenta una imagen en la que podemos \"pinchar\" con el ratón. 
 Dependiendo de la zona donde pinchemos (coordenadas x e y de la imagen) se llevará a cabo una acción u otra 
 (depende de la función que especifiquemos).

	Ejemplo:
 
 	@
		do {...
		   ;hImageInput \"imagen.jpg\" compruebaCoordenadas
		}
		  where
		    compruebaCoordenadas :: (Int,Int) -\> 'Cgi' ()
		    compruebaCoordenadas (x,y) =
		    	if (x \> unValorX) then
			  if (y \> unValorY) then
			    'tell' pagina1
			  else
			    'tell' pagina2
			else
			  if (y \> otroValorY) then
			    'tell' pagina3
			  else
			    'tell' pagina4
		    pagina1 :: 'Html' ()
		    pagina2 :: 'Html' ()
		         ...
	@
-}
hImageInput :: String 		    -- ^ Path de la imagen que queremos mostrar.
           -> ((Int,Int) -> Cgi ()) -- ^ Función que toma una tupla de dos enteros (las coordenadas) y devuelve una acción /Cgi ()/ (otra página).
	   -> Html ()
hImageInput pathImagen fcgi = 
                        do {(pag,info) <- readStateH
			   ;let nombreCampo = getName True info
			        nombreX = (getName True info) ++ ".x"
			        nombreY = (getName True info) ++ ".y"
			        info'   = nextId True info
			   ;writeStateH (pag,info')
			   ;if (currentAsk info' == -1) then	-- es la primera vez que ejecutamos, añadimos la imagen (no hay valores)
			      do {input (simpleHTMLElement "INPUT") nombreCampo noEntryInput (pType =: vImage <+> pSrc =: value pathImagen)
			         ;return undefined -- (-1,-1)  ¿qué es más correcto?
			      }
			    else	-- no es la primera vez, buscamos el ".x" y el ".y"
			      do {valorX <- input (simpleHTMLElement "INPUT") nombreX noEntryInput none
			         ;if (fromJust valorX == noEntryInput) then	-- No han pulsado la imagen
				     return undefined -- (-1,-1)
				  else	-- si han pulsado la imagen, ejecutamos la acción
			            do {valorY <- input (simpleHTMLElement "INPUT") nombreY noEntryInput none  -- si encontramos el ".x", tambien el ".y"
			               ; let pt = (read $ fromJust valorX, read $ fromJust valorY)
				       ; let Cgi_ c = fcgi pt
				       ; H c
				    }
			      }
			}

-- esta función recibe una cadena de caracteres, que será el valor asociado al botón (mensaje que saldrá en el botón)
-- El parámetro de tipo "Attr" es el/los atributo/s para el elemento <DOCTYPE> de la nueva página

{-| Esta función permite especificar una acción asociada al botón que vamos a crear. 
 De esta forma, si el botón se pulsa, se llevará acabo la acción especificada. Si disponemos
 varios botones en un mismo formulario, damos la oportunidad de presentar al cliente diferentes resultados. Esta acción
 normalmente estará relacionada con los datos recogidos en el formulario actual (en el que se incluye el botón).

 	Ejemplo:

	@
	 	do {nombre \<- 'hTextInput'
 		   ;pass   \<- 'hPasswordInput'
	 	   ;hButtonInput \"Entrar servidor 1\" (comprueba1 nombre pass)
 		   ;hButtonInput \"Entrar servidor 2\" (comprueba2 nombre pass)
	 	     ...
 		}
	 	  where
 		    comprueba1 :: String -\> String -\> 'Cgi' ()
	 	    comprueba1 n p = if (correcto1 n p) then
 				       'tell' paginaCorrectaServidor1
 				     else
 				       'ask' formularioEntradaOtraVez
 	 	    comprueba2 :: String -\> String -\> 'Cgi' ()
 		    comprueba2 n p = if (correcto2 n p) then
 				       'tell' paginaCorrectaServidor2
 				     else
	 			       'ask' formularioEntradaOtraVez
	@

	No debemos utilizar los valores recogidos en el mismo formulario en el que se presentan las etiquetas, 
	sino en el siguiente, tal y como se muestra en el ejemplo, ya que en la primera ejecución, sólo se crearán las 
	etiquetas (necesitamos ejecutarlo al menos una vez más para poder recoger los valores introducidos por el cliente).
	La segunda vez que se ejecute el programa, se recogerán todos los valores y se mostrará la página generada por 
	la función que hemos proporcionado a 'hButtonInput' sólo si ese botón ha sido pulsado.
-}
hButtonInput :: String 	-- ^ Mensaje que aparecerá en el botón.
             -> Cgi ()  -- ^ Acción a ejecutar si se pulsa el botón.
	     -> Html ()

hButtonInput s (Cgi_ b) = do {res <- (makeInput False noEntryInput (pType =: vSubmit <+> pValue =: value s))
		             ;(pag,info) <- readStateH
			     ;if (res == noEntryInput) then	-- no se ha pulsado
			        writeStateH (pag,info)	-- lo dejamos tal cual
			      else	-- hemos encontrado nuestro valor, generamos la nueva página
			        H b
		          }

-- | Con este tipo nos cercioramos de que los parámetros utilizados con la función 'hSelectInputExt'
-- son etiquetas /\<Option\>/ o bien, etiquetas /\<Optgroup\>/ (o una combinación de ellas lógicamente).
data SelectOption a = S (CGI_ ()) a

toSelectOption :: Html () -> a -> SelectOption a
toSelectOption (H cgi) x = S cgi x

fromSelectOption :: SelectOption a -> Html ()
fromSelectOption (S cgi _) = H cgi

----------
-- | Esta función crea una opción simple para ser utilizada en la función 'hSelectInputExt'.
hOption :: (CgiInput a) => a 	-- ^ Elemento para la opcion.
        -> SelectOption a
hOption x = let
	      option = hOption' $ hText (toString x)
	    in
	      toSelectOption option x

-- | Creación de un grupo de opciones relacionadas bajo un nombre de grupo (el navegador las presenta de forma diferente
-- al resto). La forma de hacerlo depende del navegador, pero usualmente el nombre de grupo viene representado en negrita y 
-- cursiva, y las opciones dentro del grupo un poco desplazadas a la derecha con respecto a las demás opciones no agrupadas.
hOptGroup :: (CgiInput a) => String 	-- ^ Nombre para el grupo (será la cabecera).
	  -> [a]   			-- ^ Lista de elementos para crear las opciones de dicho grupo.
	  -> SelectOption a
hOptGroup titulo lista@(x:xs) = 
		  let
		    listOpt  = map (fromSelectOption. hOption) lista
		  in
		    toSelectOption (withAttr (pLabel =: value titulo) $ hOptGroup' $ sequence_ listOpt) x

hOptGroup' :: Html a -> Html a
hOptGroup' = normalTag "OPTGROUP" True

hOption' :: Html a -> Html a
hOption' = normalTag "OPTION" True
----------

{- | Esta función crea una lista de selección no múltiple (sólo devuelve un dato) y sin opciones agrupadas. Es un poco
 menos potente que la función 'hSelectInputExt' (la versión extendida) pero más fácil de utilizar. Sólo es apropiada
 cuando queremos crear una lista de selección con opciones simples. Al igual que 'hRadioGroupInput', realiza conversión
 de tipos de forma automática (el valor devuelto no tiene por qué ser una cadena de caracteres).
	Ejemplo:

	@
		data Día = Lunes | Martes | Miercoles | Jueves | Viernes | Sábado | Domingo
		deriving (Show,Read,Enum)
\		
		instance CgiInput Día
\
		do { 'hText' \"Select con opciones simples\"
		   ; valor \<- hSelectInput [Lunes..Viernes]
		   ; 'hButtonInput' \"Ver valor\" (verValor valor)
			...
		}
		  where
		    verValor :: Día -\> 'Cgi' ()
		    verValor d = ...
	@	
-}
hSelectInput :: (CgiInput a) => [a] 	-- ^ Lista de valores (serán visualizados a partir de la implementación de 'toString' (clase 'CgiInput').
			     -> Html a  -- ^ Devuelve el valor seleccionado con el tipo apropiado.
hSelectInput xs = 
 do { ys <- hSelectInputExt (map hOption xs)
    ;return (head ys)	-- Siempre hay alguna opcion seleccionada (por defecto se selecciona la primera opcion)
    --; return (read' (head ys))	
    }

{- | /hSelectInputExt/ recibe una lista de opciones /select/ (que representan a las posibles elecciones) y devuelve una lista de valores,
 que son los valores seleccionados (puede ser un select multiple añadiendo el atributo 'Attributes.aMultiple').
 Al igual que 'hSelectInput', realiza la conversión de tipos de forma automática para devolver un dato del mismo tipo. 
 El único requisito es que todas las opciones se construyan con el mismo tipo instanciado, es decir, si queremos 
 añadir valores enteros, todas las opciones deben ser construidos a partir de valores enteros (no podemos añadir
 una opción de tipo entero y otra de tipo /Char/ por ejemplo).
	Ejemplo:

	@
		do { 'hText' \"Select con opciones múltiples\"
		   ; lista \<- 'withAttr' 'aMultiple' $ hSelectInputExt opciones
		   ; 'hButtonInput' \"Ver valores\" (verValores lista)
			...
		}
		  where
		    opciones =  ['hOption'   Lunes,
		    		 'hOption'   Martes,
				 'hOptGroup' \"Fin de Semana\" [Viernes,Sábado,Domingo],
				 'hOption'   Jueves]
	\
		    verValores :: [Día] -\> 'Cgi' ()
		    verValores lista = ...
	@

 Podemos observar que /verValores/ recibe una lista de /Día/, que es el tipo de los valores que habíamos utilizado
 para crear la lista de selección (con /hSelectInputExt/). Para ver la definición del tipo /Día/, acudir a la definición
 de la función 'hSelectInput'.
-}

-- La lista de acciones puede estar agrupada o no.. eso depende del usuario, nosotros solo ejecutamos el CGI_ ()
hSelectInputExt :: (CgiInput a) => [SelectOption a]	-- ^ Lista de opciones creadas con una combinación de 'hOption' y 'hOptGroup'
				-> Html [a]	  	-- ^ Lista de valores seleccionados.

hSelectInputExt lista = do {(pag,info) <- readStateH
		       ;let nombreCampo = getName True info
		            info' = nextId True info
		       ;writeStateH ((htmlElement "SELECT" True),info')  -- escribimos la etiqueta padre (select)
		       ;sequence_ lista'		-- añadimos cada una de las opciones posibles
		       ;(pag'',info'') <- readStateH	-- leemos la pagina resultante
		       ;writeStateH (pag,info'')	-- Volvemos a poner la pagina original (sin el select), pero info actualizada
		       ;compruebaValores pag'' nombreCampo none True
		    }
  where
    lista' = map fromSelectOption lista

-- El booleano especifica si se busca más de una vez la misma entrada (cuando esta a TRUE)
compruebaValores :: (CgiInput a) => HTML -> String -> Attr -> Bool -> Html [a]
compruebaValores p n a b = do {val <- input p n noEntryInput a -- buscamos datos de entrada de ejecuciones anteriores
			      ;if (isNothing val) then		-- si es la primera vez, añade la pagina con el select y las opciones
                                 return []  -- Hemos añadido el select, aun no tenemos datos de entrada
			       else		-- Hemos recibido algo
				 if (fromJust val == noEntryInput) then -- No hemos recibido ninguna información del select
				   return []	-- es el campo por defecto
				 else 	-- Hemos recibido algo, lo añadimos y seguimos buscando
				   do {valorTipado <- unsafeIO(readIO' $ fromJust val)
		     		      ;if (isNothing valorTipado) then   -- No ha sido posible la conversión de tipos, mostramos página de error.
				       	 do {H $ cgiToCGI_ $ tellExt transDocAttribute (errorPage_ "Página de error" $ msg (fromJust valorTipado))
					    ;return undefined
					 }
				       else  -- Devolvemos el valor obtenido
				         if b then
					   do {final <- compruebaValores p n a True
					      ;unsafeIO $ return ([fromJust valorTipado]++final)
					   }
					 else
					   unsafeIO (return $ [fromJust valorTipado])
				   }
			   }

{-| Creación de un área de texto en la que podemos escribir párrafos de larga longitud. Las dimensiones son
/a/ filas por /b/ columnas, donde /a/ y /b/ son, respectivamente, el primer y segundo argumento que recibe.
	
  Ejemplo:
  
  @
  		do {'hText' \"Introduce un texto\"
  		   ;texto \<- hTextArea 5 30
  		   ;'hButtonInput' \"Mándalo!!!\" (leerTexto texto)
  		}
  		  where 
  		    leerTexto :: String -\> 'Cgi' ()
  		    leerTexto s = ...
  @
-}
hTextAreaInput :: Value 	-- ^ Número de filas.
	       -> Value 	-- ^ Número de columnas.
	       -> Html String	-- ^ Devuelve el texto escrito por el usuario.
hTextAreaInput r c = do {(pag,info) <- readStateH
		       ;let nombreCampo = getName True info
		            info' = nextId True info
		       ;writeStateH (htmlElement "TEXTAREA" False, info')	-- Creamos la pagina con el textArea
		       ;(pag'',info'') <- readStateH				-- leemos pagina resultante
		       ;writeStateH (pag,info'')				-- La dejamos tal y como estaba
		       ;val <- (compruebaValores pag'' nombreCampo (pRows =: r <+> pCols =: (c+1)) False) 
		       ;if (null val) then
		          return ""		-- No ha encontrado ningún valor, devolvemos la cadena vacia
		        else
		          return (head val)		-- si ha encontrado algún valor, sólo habrá uno.
		    }

hTextAreaInputExt :: Value 	-- ^ Número de filas.
	       -> Value 	-- ^ Número de columnas.
               -> String        -- ^ Contenido inicial
	       -> Html String	-- ^ Devuelve el texto escrito por el usuario.
hTextAreaInputExt r c str
                  = do {(pag,info) <- readStateH
		       ;let nombreCampo = getName True info
		            info' = nextId True info
		       ;writeStateH (Htag "TEXTAREA" none none [Htext str] False, info')	-- Creamos la pagina con el textArea
		       ;(pag'',info'') <- readStateH				-- leemos pagina resultante
		       ;writeStateH (pag,info'')				-- La dejamos tal y como estaba
		       ;val <- (compruebaValores pag'' nombreCampo (pRows =: r <+> pCols =: (c+1)) False) 
		       ;if (null val) then
		          return ""		-- No ha encontrado ningún valor, devolvemos la cadena vacia
		        else
		          return (head val)		-- si ha encontrado algún valor, sólo habrá uno.
		    }


-----------------
hInput :: Html ()
hInput = simpleTag "INPUT"

hButton :: Html a -> Html a
hButton = normalTag "BUTTON" True

hFieldSet :: Html a -> Html a
hFieldSet = normalTag "FIELDSET" True

hLegend :: Html a -> Html a
hLegend = normalTag "LEGEND" True

hLabel :: Html a -> Html a
hLabel = normalTag "LABEL" True

{---------- Esto se ha mejorado
hSelect :: Html () -> Html ()
hSelect = normalTag "SELECT" True

hTextArea :: Html () -> Html ()
hTextArea = normalTag "TEXTAREA" True
-}

{-
Elementos "en linea" especiales

    * A - Anchor
    * APPLET - Java applet
    * BASEFONT - Cambio de fuente base
    * BDO - Sobreescribir BiDi (Bidirectional Algorithm)
    * BR - Salto de linea
    * FONT - Cambio de fuente
    * IFRAME - Frame "en linea"
    * IMG - Imagen "en linea"
    * MAP - Mapa de imagen
          o AREA - Region de imagen del mapa
    * OBJECT - Objeto (código, imágenes, videos...)
    * PARAM - Parámetro de entrada para el objeto
    * Q - Short quotation
		    * SCRIPT - Script del lado del cliente   -- Ya se ha implementado esta función (Elementos Head)
    * SPAN - Contenedor genérico "en linea"
    * SUB - Subíndice
    * SUP - Superíndice
-}
hA :: Html a -> Html a
hA = normalTag "A" True

hApplet :: Html a -> Html a
hApplet = normalTag "APPLET" True

hBaseFont :: Html ()
hBaseFont = simpleTag "BASEFONT"

hBdo :: Html a -> Html a
hBdo = normalTag "BDO" True

hBr :: Html ()
hBr = simpleTag "BR"

hFont :: Html a -> Html a
hFont = normalTag "FONT" False

hIFrame :: Html a -> Html a
hIFrame = normalTag "IFRAME" True

hImg :: Html ()
hImg = simpleTag "IMG"

hMap :: Html a -> Html a
hMap = normalTag "MAP" True

hArea :: Html ()
hArea = simpleTag "AREA"

hObject :: Html a -> Html a
hObject = normalTag "OBJECT" True

hParam :: Html ()
hParam = simpleTag "PARAM"

hQ :: Html a -> Html a
hQ = normalTag "Q" True

hSpan :: Html a -> Html a
hSpan = normalTag "SPAN" False

hSub :: Html a -> Html a
hSub = normalTag "SUB" True

hSup :: Html a -> Html a
hSup = normalTag "SUP" True

{-
Elementos de frase

    * ABBR - Abbreviation
    * ACRONYM - Acronym
    * CITE - Citation
    * CODE - Computer code
		    * DEL - Deleted text  -- ya ha sido definida en "elementos de bloque genericos"
    * DFN - Defined term
    * EM - Emphasis
		    * INS - Inserted text  -- ya ha sido definida en "elementos de bloque genericos"
    * KBD - Text to be input
    * SAMP - Sample output
    * STRONG - Strong emphasis
    * VAR - Variable
-}
hAbbr :: Html a -> Html a
hAbbr = normalTag "ABBR" True

hAcronym :: Html a -> Html a
hAcronym = normalTag "ACRONYM" True

hCite :: Html a -> Html a
hCite = normalTag "CITE" True

hCode :: Html a -> Html a
hCode = normalTag "CODE" True

hDfn :: Html a -> Html a
hDfn = normalTag "DFN" True

hEm :: Html a -> Html a
hEm = normalTag "EM" True

hKbd :: Html a -> Html a
hKbd = normalTag "KBD" True

hSamp :: Html a -> Html a
hSamp = normalTag "SAMP" True

hStrong :: Html a -> Html a
hStrong = normalTag "STRONG" True

hVar :: Html a -> Html a
hVar = normalTag "VAR" True

{-
Font Style Elements

    * B - Bold text
    * BIG - Large text
    * I - Italic text
    * S - Strike-through text
    * SMALL - Small text
    * STRIKE - Strike-through text
    * TT - Teletype text
    * U - Underlined text
-}
hB :: Html a -> Html a
hB = normalTag "B" True

hBig :: Html a -> Html a
hBig = normalTag "BIG" True

hI :: Html a -> Html a
hI = normalTag "I" True

hS :: Html a -> Html a
hS = normalTag "S" True

hSmall :: Html a -> Html a
hSmall = normalTag "SMALL" True

hStrike :: Html a -> Html a
hStrike = normalTag "STRIKE" True

hTt :: Html a -> Html a
hTt = normalTag "TT" True

hU :: Html a -> Html a
hU = normalTag "U" True

{-
Frames
	
    		* FRAMESET - Frameset  -- ya ha sido definida en "elementos mas externos"
          o FRAME - Frame
    * NOFRAMES - Contenido alternativo a los frames (cuando el navegador no soporta frames)
-}
hFrame :: Html ()
hFrame = simpleTag "FRAME"

hNoFrames :: Html a -> Html a
hNoFrames = normalTag "NOFRAMES" True
