{-# LANGUAGE ForeignFunctionInterface #-}
-- #prune,hide

{-|   (c) /Pepe Gallardo (pepeg\@lcc.uma.es), 2003/

     Primitivas en C.

-}

module AP.Cgi.Lib.Lock ( HANDLE
            , lockFile
            , unlockFile

        {-| Ejemplo: 
          
          @
                main = do { h \<- 'lockFile' \"file1\"
                          ; ...
                          ; operaciones sobre \"file1\"
                          ; ...
                          ; 'unlockFile' h
                          }
          @
        -}

        -- *** Operaciones bajo exclusin mutua
        -- |Las siguientes funciones utilizan automticamente las primitivas 'lockFile' y 'unlockFile' para 
        -- realizar las operaraciones sobre el fichero.

          -- **** Lectura de un fichero
            , exclusiveReadFile
          -- **** Escritura en un fichero (si no existe, se crea automticamente)
            , exclusiveWriteFile
          -- **** Aadir contenido al final de un fichero
            , exclusiveAppendFile

        {-| Ejemplo: Lectura, escritura o adicin bajo exclusin mutua.
          
          @
                main = do { cont \<- 'exclusiveReadFile' \"file1\"
                          ; 'exclusiveWriteFile'  \"file2\" cont
                          ; 'exclusiveAppendFile' \"file3\" cont
                          }
          @
        -}
            ) where

import Char
import Foreign.Marshal.Array
import Foreign.C.Types
import Foreign.Ptr ( Ptr 
                   , nullPtr
                   )
import Foreign.C.String ( CString
                        , withCString
                        )
-- | Es como un manejador de ficheros.
type HANDLE = Ptr () 

foreign import ccall unsafe "primwin32Aux.h _lockFile"    _lockFile     :: CString -> IO HANDLE
foreign import ccall unsafe "primwin32Aux.h _unlockFile"  _unlockFile   :: HANDLE -> IO Bool
foreign import ccall unsafe "primwin32Aux.h _getErrorMsg" _getLastError :: IO CString

-- He tenido que redefinir la funcin castCCharToChar en el mdulo CString
-- para que trabaje bien con caracteres acentuados
peekCString    :: CString -> IO String
peekCString cp  = do cs <- peekArray0 endChar cp; return (cCharsToChars cs)
 where
  endChar :: CChar
  endChar = 13

  cCharsToChars :: [CChar] -> [Char]
  cCharsToChars  = map castCCharToChar

  castCCharToChar :: CChar -> Char
  castCCharToChar ch 
   | ch < 0    = chr (256 - fromIntegral (abs ch))
   | otherwise = chr (fromIntegral ch)



getLastError :: IO String
getLastError = 
 do { cString <- _getLastError
    ; str <- peekCString cString
    ; return str
    }

-- | Recibe el nombre del archivo (con su path) y lo bloquea, devolviendo el manejador de archivo correspondiente.
lockFile :: FilePath -> IO HANDLE
lockFile path = 
 do { h <- withCString path _lockFile
    ; if h == nullPtr then do { str <- getLastError
                              ; ioError $ userError str
                              }
                      else return h
    }                  


-- | Recibe el manejador del archivo y lo libera.
unlockFile :: HANDLE -> IO ()
unlockFile h =
 do { b <- _unlockFile h
    ; if not b then do { str <- getLastError
                       ; ioError $ userError str
                       }
               else return ()
    }           
    

-- sleep suspende al proceso llamante durante el nmero de segundos que le especifiquemos en su argumento.
-- foreign import ccall unsafe "windows.h Sleep" sleep :: Int -> IO ()

doLocked :: FilePath -> IO a -> IO a
doLocked path io = 
 do { h <- lockFile path
    ; x <- io
    ; unlockFile h
    ; return x
    }
    

-- | Lectura del contenido de un fichero
exclusiveReadFile :: FilePath   -- ^ Nombre del fichero (path incluido)
                  -> IO String  -- ^ Devuelve el contenido completo
exclusiveReadFile path = doLocked path $ readFile path

-- | Escritura en un fichero
exclusiveWriteFile :: FilePath  -- ^ Nombre del fichero (path incluido)
                    -> String   -- ^ Datos a escribir en el fichero
                    -> IO ()    -- ^ No devuelve nada
exclusiveWriteFile path str = doLocked path $ writeFile path str

-- | Concatenacin (se aade al final del fichero)
exclusiveAppendFile :: FilePath -- ^ Nombre del fichero.
                      -> String -- ^ Datos a aadir en el fichero.
                      -> IO ()  -- ^ No devuelve nada
exclusiveAppendFile path str = doLocked path $ appendFile path str 

